##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# Uplink.pm --
#
#       A basic class that contains information representing a uplink.
#       Information can be loaded from a configuration tree.
#

package VMware::Net::Uplink;

use VMware::Log qw(:log);

use strict;


########################################################################
#
# Uplink::new --
#
#       Constructor for a VMware::Net::Uplink object.
#       An empty config tree is valid but a warning will be issued
#       if the pnic is not set before config data is retrieved.
#
# Results:
#       The new object, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;            # IN: Invoking class.
   my $cfgTree = (shift || {});  # IN: Tree structure of data from the
                                 #     config file.
   my $self = {
      cfgTree => $cfgTree,
   };

   bless $self => $class;
   return $self;
}


########################################################################
#
# Uplink::GetConfigData --
#
#       Get the uplink info in a structure suitable for the config file.
#
# Results:
#       The data structure.
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigData
{
   my $self = shift;  # IN: Invoking instance.

   if (not defined $self->{cfgTree}->{pnic}) {
      LogWarn("Config data retrieved on an incomplete Uplink (no pnic).");
   }
   return $self->{cfgTree};
}


#
# Trivial Accessor Functions
#

sub GetPnic
{
   my $self = shift;
   if (not defined $self->{cfgTree}->{pnic}) {
      LogWarn("Config data retrieved on an incomplete Uplink (no pnic).");
   }
   return $self->{cfgTree}->{pnic};
}

sub GetPnicNumber
{
   my $self = shift;

   my $pnic = $self->GetPnic();
   return undef if not defined $pnic;
   return ($pnic =~ /(\d+)$/)[0];
}


########################################################################
#
# Uplink::GetPnicDevice --
#
#       Find the device associated with the uplink pnic.
#
# Results:
#       Returns the Device object if one is found, or undef
#       otherwise.
#
# Side effects:
#       Queries the device manager for the pnic device.
#
########################################################################

sub GetPnicDevice
{
   my $self = shift;    # IN: Invoking instance.
   my $devmgr = shift;  # IN: The device manager

   my $pnic = $self->GetPnic();
   return undef if not defined $pnic;

   my $pciStr = $devmgr->FindPCIStringByVMKName($self->GetPnic());
   return $devmgr->GetDevice($pciStr);
}


########################################################################
#
# Uplink::SetPnic --
#
#       Set the physical nic for this uplink.  Do not allow
#       the physical nic to become undefined (callers should destroy
#       the entire Uplink object).
#
# Results:
#       None.
#
# Side effects:
#       Set the pnic.
#
########################################################################

sub SetPnic
{
   my $self = shift; # IN/OUT: Invoking instance.
   my $pnic = shift;  # IN: New pnic.

   if (not defined $pnic) {
      LogWarn("Cannot undefine an Uplink's pnic.");
      return;
   }

   $self->{cfgTree}->{pnic} = $pnic;  # IN: New pnic.
}


1;
