##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# VSwitch.pm --
#
#       A class that contains information representing a virtual switch.
#       Information can be loaded from a configuration tree.
#

package VMware::Net::VSwitch;

use VMware::Log qw(:log);
use VMware::Net::Beacon;
use VMware::Net::PortGroup;
use VMware::Net::Uplink;
use VMware::Net::TeamPolicy;

use strict;

use constant VSWITCH_NAME_BASE => "vSwitch";


########################################################################
#
# VSwitch::new --
#
#       Constructor for a VMware::Net::VSwitch object.
#
# Results:
#       The new object, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;            # IN: Invoking class.
   my $cfgTree = (shift || {});  # IN: Tree structure of data from the
                                 #     config file.

   my $self = {
      cfgTree => $cfgTree,
      beacon => undef,
      portgroups => [],
      uplinks => [],
      teamPolicy => undef,
   };
  
   if (defined $cfgTree->{beacon}) {
      my $beacon = new VMware::Net::Beacon($cfgTree->{beacon});
      if (defined $beacon) {
         $self->{beacon} = $beacon;
      } else {
         LogError("Could not build beacon object for vswitch");
         return undef;
      }
   }

   if (defined $cfgTree->{portgroup}->{child}) {
      foreach my $child (@{$cfgTree->{portgroup}->{child}}) {
         # Skip empty indicies, but maintain index order.
         unless (defined $child) {
            push @{$self->{portgroups}}, undef;
            next;
         }

         my $portgroup = new VMware::Net::PortGroup($child);
         if ($portgroup) {
            push @{$self->{portgroups}}, $portgroup;
         } else {
            LogError("Could not build portgroup object for vswitch");
            return undef;
         }
      }
   }

   if (defined $cfgTree->{uplinks}->{child}) {
      foreach my $child (@{$cfgTree->{uplinks}->{child}}) {
         # Skip empty indicies, but maintain index order.
         unless (defined $child) {
            push @{$self->{uplinks}}, undef;
            next;
         }

         my $uplink = new VMware::Net::Uplink($child);
         if ($uplink) {
            push @{$self->{uplinks}}, $uplink;
         } else {
            LogError("Could not build uplink object for vswitch");
            return undef;
         }
      }
   }

   if (defined $cfgTree->{teamPolicy}) {
      $self->{teamPolicy} = new VMware::Net::TeamPolicy($cfgTree->{teamPolicy});
   }

   bless $self => $class;
   return $self;
}


########################################################################
#
# VSwitch::GetConfigData --
#
#       Return this object's information as a data structure
#       suitable for setting in a vmacore-style config file.
#
# Results:
#       The data structure.
#
# Side effects:
#       Updates $self->{cfgTree} from other data members.
# 
########################################################################

sub GetConfigData
{
   my $self = shift; # IN: Invoking instance.

   #
   # Note, 'portgroup' in cfgTree, 'portgroups' in $self.
   #

   $self->{cfgTree}->{portgroup}->{child} =
      [ map { $_->GetConfigData(); } @{$self->{portgroups}} ];
   $self->{cfgTree}->{uplinks}->{child} =
      [ map { $_->GetConfigData(); } @{$self->{uplinks}} ];
   if (defined $self->{beacon}) {
      $self->{cfgTree}->{beacon} = $self->{beacon}->GetConfigData();
   }
   if (defined $self->{teamPolicy}) {
      $self->{cfgTree}->{teamPolicy} = $self->{teamPolicy}->GetConfigData();
   }

   return $self->{cfgTree};
}


#
# Trivial Accessor Functions
#

sub GetBeacon
{
   my $self = shift;
   return $self->{beacon};
}

sub GetTeamPolicy
{
   my $self = shift;
   return $self->{teamPolicy};
}

sub GetName
{
   my $self = shift;
   return $self->{cfgTree}->{name};
}

sub GetNumPorts
{
   my $self = shift;
   return $self->{cfgTree}->{numPorts};
}

sub GetNumber
{
   my $self = shift;
   return ($self->GetName() =~ /(\d+)$/)[0];
}


########################################################################
#
# GetPortGroups --
#
#       List the portgroups on the virtual switch.
#
# Results:
#       Returns a list of VMware::Net::PortGroup objects corresponding
#       to the virtual switch portgroups. 
#
# Side effects:
#       None.
#
########################################################################

sub GetPortGroups
{
   my $self = shift;

   # The portgroup list can have undef placeholders, filter them out.
   my @portgroups = grep { defined $_ } @{$self->{portgroups}};
   return @portgroups;
}


########################################################################
#
# GetUplinks --
#
#       List the uplinks on the virtual switch.
#
# Results:
#       Returns a list of VMware::Net::PortGroup objects corresponding
#       to the virtual switch uplinks. The list will be sorted in
#       increasing order of pnic number.
#
# Side effects:
#       None.
#
########################################################################

sub GetUplinks
{
   my $self = shift; # IN: Invoking instance.

   # The uplink list can have undef placeholders, filter them out.
   my @uplinks = grep { defined $_ } @{$self->{uplinks}};
   @uplinks = sort { $a->GetPnicNumber() <=> $b->GetPnicNumber() } @uplinks;
   return @uplinks;
}


########################################################################
#
# VSwitch::AddPortGroup --
#
#       Add a port group to this virtual switch.  The port group is
#       assumed to have sane data for adding to this group (i.e.
#       not have a duplicate name or number, etc.).  This method
#       does not check.
#
# Results:
#       None.
#
# Side effects:
#       Adds a port group.
#
########################################################################

sub AddPortGroup
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $pg = shift;    # IN: The VMware::Net::PortGroup to add.

   push(@{$self->{portgroups}}, $pg);
}


########################################################################
#
# VSwitch::AddUplink --
#
#       Add an Uplink object to this VSwitch.
#
# Results:
#       None.
#
# Side effects:
#       Adds an uplink.
#
########################################################################

sub AddUplink
{
   my $self = shift; # IN/OUT: Invoking instance.
   my $uplink = shift; # IN: VMware::Net::Uplink instance.
   my $first = shift;  # IN: Make this the first (index 0) uplink.
                       #     This can be important for failover order.

   if ($first) {
      unshift(@{$self->{uplinks}}, $uplink);
   } else {
      push(@{$self->{uplinks}}, $uplink);
   }
}


########################################################################
#
# VSwitch::SetBeacon --
#
#      Set the Beacon object for this VSwitch.  Replaces the current
#      Beacon if any was configured.
#
# Results:
#      None.
#
# Side effects:
#      Sets Beacon.
#
########################################################################

sub SetBeacon
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $beacon = shift;  # IN: New Beacon.

   $self->{beacon} = $beacon;
}


########################################################################
#
# VSwitch::SetTeamPolicy --
#
#       Sets the overall TeamPolicy for the VSwitch.  This TeamPolicy
#       can be overridden on a per-portgroup level by setting the
#       TeamPolicy on a PortGroup object.  This method replaces all
#       current TeamPolicy settings at the VSwitch level, but does
#       not touch existing per-portgroup settings.
#
# Results:
#       None.
#
# Side effects:
#       Sets TeamPolicy.
#
########################################################################

sub SetTeamPolicy
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $policy = shift;  # IN: New team policy.

   $self->{teamPolicy} = $policy;
}


########################################################################
#
# VSwitch::SetName --
#
#       Sets the name of this VSwitch.
#
# Results:
#       None.
#
# Side effects:
#       Sets the name.
#
########################################################################

sub SetName
{
   my $self = shift;  # IN/OUT: Invoking instance.
   $self->{cfgTree}->{name} = shift;  # IN: The new name.
}


1;
