##########################################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# Device.pm --
#
#       A basic class that contains information representing a PCI device.
#       Information includes bus #, device/slot #, function #, name, irq,
#       owner.
#

package VMware::PCI::Device;

use VMware::Log qw(:log);
use VMware::DataStruct;

use VMware::PCI::PCIString qw(:all);
use VMware::PCI::DeviceMap;
use VMware::PCI::DeviceMapEntry;

use strict;

use constant COS => "serviceConsole";
use constant VMKERNEL => "vmkernel";

use constant EQUIV_NIC => 'nic';
use constant EQUIV_FC => 'fc';
use constant EQUIV_SCSI => 'scsi';
use constant EQUIV_IDE => 'ide';

use constant MASK_BASE_CLASS => 0x00ff0000; # High 8 bits.
use constant MASK_CLASS => 0x00ffff00;      # High 16 bits.


#
# Device classes and subclasses as defined in linux/pci.h
#

use constant PCI_CLASS_NOT_DEFINED =>             0x000000;
use constant PCI_CLASS_NOT_DEFINED_VGA =>         0x000100;

use constant PCI_BASE_CLASS_STORAGE =>            0x010000;
use constant PCI_CLASS_STORAGE_SCSI =>            0x010000;
use constant PCI_CLASS_STORAGE_IDE =>             0x010100;
use constant PCI_CLASS_STORAGE_FLOPPY =>          0x010200;
use constant PCI_CLASS_STORAGE_IPI =>             0x010300;
use constant PCI_CLASS_STORAGE_RAID =>            0x010400;
use constant PCI_CLASS_STORAGE_OTHER =>           0x018000;

use constant PCI_BASE_CLASS_NETWORK =>            0x020000;
use constant PCI_CLASS_NETWORK_ETHERNET =>        0x020000;
use constant PCI_CLASS_NETWORK_TOKEN_RING =>      0x020100;
use constant PCI_CLASS_NETWORK_FDDI =>            0x020200;
use constant PCI_CLASS_NETWORK_ATM =>             0x020300;
use constant PCI_CLASS_NETWORK_OTHER =>           0x028000;

use constant PCI_BASE_CLASS_DISPLAY =>            0x030000;
use constant PCI_CLASS_DISPLAY_VGA =>             0x030000;
use constant PCI_CLASS_DISPLAY_XGA =>             0x030100;
use constant PCI_CLASS_DISPLAY_OTHER =>           0x038000;

use constant PCI_BASE_CLASS_MULTIMEDIA =>         0x040000;
use constant PCI_CLASS_MULTIMEDIA_VIDEO =>        0x040000;
use constant PCI_CLASS_MULTIMEDIA_AUDIO =>        0x040100;
use constant PCI_CLASS_MULTIMEDIA_OTHER =>        0x048000;

use constant PCI_BASE_CLASS_MEMORY =>             0x050000;
use constant PCI_CLASS_MEMORY_RAM =>              0x050000;
use constant PCI_CLASS_MEMORY_FLASH =>            0x050100;
use constant PCI_CLASS_MEMORY_OTHER =>            0x058000;

use constant PCI_BASE_CLASS_BRIDGE =>             0x060000;
use constant PCI_CLASS_BRIDGE_HOST =>             0x060000;
use constant PCI_CLASS_BRIDGE_ISA =>              0x060100;
use constant PCI_CLASS_BRIDGE_EISA =>             0x060200;
use constant PCI_CLASS_BRIDGE_MC =>               0x060300;
use constant PCI_CLASS_BRIDGE_PCI =>              0x060400;
use constant PCI_CLASS_BRIDGE_PCMCIA =>           0x060500;
use constant PCI_CLASS_BRIDGE_NUBUS =>            0x060600;
use constant PCI_CLASS_BRIDGE_CARDBUS =>          0x060700;
use constant PCI_CLASS_BRIDGE_OTHER =>            0x068000;

use constant PCI_BASE_CLASS_COMMUNICATION =>      0x070000;
use constant PCI_CLASS_COMMUNICATION_SERIAL =>    0x070000;
use constant PCI_CLASS_COMMUNICATION_PARALLEL =>  0x070100;
use constant PCI_CLASS_COMMUNICATION_OTHER =>     0x078000;

use constant PCI_BASE_CLASS_SYSTEM =>             0x080000;
use constant PCI_CLASS_SYSTEM_PIC =>              0x080000;
use constant PCI_CLASS_SYSTEM_DMA =>              0x080100;
use constant PCI_CLASS_SYSTEM_TIMER =>            0x080200;
use constant PCI_CLASS_SYSTEM_RTC =>              0x080300;
use constant PCI_CLASS_SYSTEM_OTHER =>            0x088000;

use constant PCI_BASE_CLASS_INPUT =>              0x090000;
use constant PCI_CLASS_INPUT_KEYBOARD =>          0x090000;
use constant PCI_CLASS_INPUT_PEN =>               0x090100;
use constant PCI_CLASS_INPUT_MOUSE =>             0x090200;
use constant PCI_CLASS_INPUT_OTHER =>             0x098000;

use constant PCI_BASE_CLASS_DOCKING =>            0x0a0000;
use constant PCI_CLASS_DOCKING_GENERIC =>         0x0a0000;
use constant PCI_CLASS_DOCKING_OTHER =>           0x0a0100;

use constant PCI_BASE_CLASS_PROCESSOR =>          0x0b0000;
use constant PCI_CLASS_PROCESSOR_386 =>           0x0b0000;
use constant PCI_CLASS_PROCESSOR_486 =>           0x0b0100;
use constant PCI_CLASS_PROCESSOR_PENTIUM =>       0x0b0200;
use constant PCI_CLASS_PROCESSOR_ALPHA =>         0x0b1000;
use constant PCI_CLASS_PROCESSOR_POWERPC =>       0x0b2000;
use constant PCI_CLASS_PROCESSOR_CO =>            0x0b4000;

use constant PCI_BASE_CLASS_SERIAL =>             0x0c0000;
use constant PCI_CLASS_SERIAL_FIREWIRE =>         0x0c0000;
use constant PCI_CLASS_SERIAL_ACCESS =>           0x0c0100;
use constant PCI_CLASS_SERIAL_SSA =>              0x0c0200;
use constant PCI_CLASS_SERIAL_USB =>              0x0c0300;
use constant PCI_CLASS_SERIAL_FIBER =>            0x0c0400;
use constant PCI_CLASS_SERIAL_SMBUS =>            0x0c0500;

use constant PCI_BASE_CLASS_INTELLIGENT =>        0x0e0000;
use constant PCI_CLASS_INTELLIGENT_I2O  =>        0x0e0000;

use constant PCI_CLASS_HOT_SWAP_CONTROLLER =>     0xff0000;

use constant PCI_CLASS_OTHERS =>                  0xff0000;

#
# 'use constant' is a bit misleading here and for the other hash
# ref constants, as it is only the hash ref and not the contents that
# are constant.  But it is visually consistant with scalar constants
# and the best Perl can do.
#

use constant CLASS_NAME => {
   PCI_CLASS_NOT_DEFINED() => 'Undefined',
   PCI_CLASS_NOT_DEFINED_VGA() => 'VGA controller',

   PCI_BASE_CLASS_STORAGE() => 'Storage controller',
   PCI_CLASS_STORAGE_SCSI() => 'SCSI storage controller',
   PCI_CLASS_STORAGE_IDE() => 'IDE storage controller',
   PCI_CLASS_STORAGE_FLOPPY() => 'Floppy storage controller',
   PCI_CLASS_STORAGE_IPI() => 'IPI storage controller',
   PCI_CLASS_STORAGE_RAID() => 'RAID storage controller',
   PCI_CLASS_STORAGE_OTHER() => 'Other storage controller',

   PCI_BASE_CLASS_NETWORK() => 'Network controller',
   PCI_CLASS_NETWORK_ETHERNET() => 'Ethernet controller',
   PCI_CLASS_NETWORK_TOKEN_RING() => 'Token Ring controller',
   PCI_CLASS_NETWORK_FDDI() => 'FDDI controller',
   PCI_CLASS_NETWORK_ATM() => 'ATM controller',
   PCI_CLASS_NETWORK_OTHER() => 'Other network controller',

   PCI_BASE_CLASS_DISPLAY() => 'Display controller',
   PCI_CLASS_DISPLAY_VGA() => 'VGA controller',
   PCI_CLASS_DISPLAY_XGA() => 'XGA controller',
   PCI_CLASS_DISPLAY_OTHER() => 'Other display controller',

   PCI_BASE_CLASS_MULTIMEDIA() => 'Multimedia controller',
   PCI_CLASS_MULTIMEDIA_VIDEO() => 'Video controller',
   PCI_CLASS_MULTIMEDIA_AUDIO() => 'Audio controller',
   PCI_CLASS_MULTIMEDIA_OTHER() => 'Other multimedia controller',

   PCI_BASE_CLASS_MEMORY() => 'Memory',
   PCI_CLASS_MEMORY_RAM() => 'RAM memory',
   PCI_CLASS_MEMORY_FLASH() => 'Flash memory',
   PCI_CLASS_MEMORY_OTHER() => 'Other memory',

   PCI_BASE_CLASS_BRIDGE() => 'Bridge',
   PCI_CLASS_BRIDGE_HOST() => 'Host Bridge',
   PCI_CLASS_BRIDGE_ISA() => 'ISA Bridge',
   PCI_CLASS_BRIDGE_EISA() => 'EISA Bridge',
   PCI_CLASS_BRIDGE_MC() => 'MC Bridge',
   PCI_CLASS_BRIDGE_PCI() => 'PCI Bridge',
   PCI_CLASS_BRIDGE_PCMCIA() => 'PCMCIA Bridge',
   PCI_CLASS_BRIDGE_NUBUS() => 'NUBus Bridge',
   PCI_CLASS_BRIDGE_CARDBUS() => 'CardBus Bridge',
   PCI_CLASS_BRIDGE_OTHER() => 'Other bridge',

   PCI_BASE_CLASS_COMMUNICATION() => 'Commomuncation controller',
   PCI_CLASS_COMMUNICATION_SERIAL() => 'Serial port controller',
   PCI_CLASS_COMMUNICATION_PARALLEL() => 'Parallel port controller',
   PCI_CLASS_COMMUNICATION_OTHER() => 'Other communication controller',

   PCI_BASE_CLASS_SYSTEM() => 'System controller',
   PCI_CLASS_SYSTEM_PIC() => 'PIC controller',
   PCI_CLASS_SYSTEM_DMA() => 'DMA controller',
   PCI_CLASS_SYSTEM_TIMER() => 'Timer controller',
   PCI_CLASS_SYSTEM_RTC() => 'RTC controller',
   PCI_CLASS_SYSTEM_OTHER() => 'Other system controller',

   PCI_BASE_CLASS_INPUT() => 'Input controller',
   PCI_CLASS_INPUT_KEYBOARD() => 'Keyboard controller',
   PCI_CLASS_INPUT_PEN() => 'Pen input controller',
   PCI_CLASS_INPUT_MOUSE() => 'Mouse controller',
   PCI_CLASS_INPUT_OTHER() => 'Other input controller',

   PCI_BASE_CLASS_DOCKING() => 'Docking controller',
   PCI_CLASS_DOCKING_GENERIC() => 'Generic docking controller',
   PCI_CLASS_DOCKING_OTHER() => 'Docking controller',

   PCI_BASE_CLASS_PROCESSOR() => 'Processor',
   PCI_CLASS_PROCESSOR_386() => '386 processor',
   PCI_CLASS_PROCESSOR_486() => '486 processor',
   PCI_CLASS_PROCESSOR_PENTIUM() => 'Pentium processor',
   PCI_CLASS_PROCESSOR_ALPHA() => 'Alpha processor',
   PCI_CLASS_PROCESSOR_POWERPC() => 'PowerPC processor',
   PCI_CLASS_PROCESSOR_CO() => 'Processor controller',

   PCI_BASE_CLASS_SERIAL() => 'Serial controller',
   PCI_CLASS_SERIAL_FIREWIRE() => 'Firewire controller',
   PCI_CLASS_SERIAL_ACCESS() => 'Access controller',
   PCI_CLASS_SERIAL_SSA() => 'SSA controller',
   PCI_CLASS_SERIAL_USB() => 'USB controller',
   PCI_CLASS_SERIAL_FIBER() => 'Fiber storage controller',
   PCI_CLASS_SERIAL_SMBUS() => 'SMBus controller',

   PCI_BASE_CLASS_INTELLIGENT() => 'Intelligent controller',
   PCI_CLASS_INTELLIGENT_I2O () => 'Intelligent Input/Output',

   PCI_CLASS_HOT_SWAP_CONTROLLER() => 'Hot swap controller',

   PCI_CLASS_OTHERS() => 'Other PCI controller'
};

use constant SERIAL_FIBER_DRIVERS => {
   "lpfcdd_2xx.o"  => 1,
   "lpfcdd_7xx.o"  => 1,
   "lpfcdd_732.o"  => 1,
   "lpfc_740.o"  => 1,
   "qla2200_7xx.o" => 1,
   "qla2200_607.o" => 1,
   "qla2200_707.o" => 1,
   "qla2300_7xx.o" => 1,
   "qla2300_607.o" => 1,
   "qla2300_707.o" => 1,
   "qla2300_707_vmw.o" => 1,
};


########################################################################
#
# Device::new --
#
#       Constructor for a PCI::Device object.
#
# Results:
#       The new object, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;           # IN: Invoking class.
   my $pciStr = shift;          # IN: PCIString for this class.
   my $cfgTree = (shift || {}); # IN: Tree structure of data from the
                                #     config file.

   my ($bus,
       $slot,
       $func) = SplitPCIString($pciStr);

   if (!defined($bus) || !defined($slot) || !defined($func)) {
      $pciStr = '' if not defined $pciStr;
      LogError("Malformed PCI bus:slot.function string '$pciStr', cannot " .
               "construct PCI Device object.");
      return undef;
   }

   if (defined($cfgTree->{owner}) &&
       $cfgTree->{owner} ne COS &&
       $cfgTree->{owner} ne VMKERNEL) {
      LogError("Malformed owner string '$cfgTree->{owner}' for device " .
               "'$pciStr'.  Cannot construct PCI Device object.");
      return undef;
   }

   my $self = {
      bus => $bus,
      slot => $slot,
      func => $func,
      cfgTree => $cfgTree,
      drivers => [],       # Array of drivers.
      equiv => undef,      # Equivalence class for all drivers.
   };

   $cfgTree->{name} = "Unknown Device" if not defined $cfgTree->{name};
   $cfgTree->{owner} = COS if not defined $cfgTree->{owner};
   $cfgTree->{options} = "" if not defined $cfgTree->{options};
   $cfgTree->{vendor} = 0 if not defined $cfgTree->{vendor};
   $cfgTree->{devID} = 0 if not defined $cfgTree->{devID};
   $cfgTree->{class} = 0x000000 if not defined $cfgTree->{class};
   $cfgTree->{subsysVendor} = 0 if not defined $cfgTree->{subsysVendor};
   $cfgTree->{subsysDevID} = 0 if not defined $cfgTree->{subsysDevID};

   #
   # vmkname defaults to undef on purpose.  And things are
   # happier if the key always exists even if it is not defined.
   #

   $cfgTree->{vmkname} = undef if not defined $cfgTree->{vmkname};

   bless $self => $class;
   return $self;
}


########################################################################
#
# Device::Copy --
#
#       Constructs a new Device instance based on this instance.
#
# Results:
#       A new Device identical to the original.
#
# Side effects:
#       None.
#
########################################################################

sub Copy
{
   my $self = shift;  # IN: Invoking instance (base for the copy).
   my $class = ref($self);

   my $clonedTree = VMware::DataStruct::Clone($self->{cfgTree});

   my $copy = $class->new($self->GetPCIString(), $clonedTree);

   #
   # These two members are never set up during construction.
   #

   @{$copy->{drivers}} = @{$self->{drivers}};
   $copy->{equiv} = $self->{equiv};

   return $copy;
}


#
# Trivial Accessor Functions
#

sub GetBus
{
   my $self = shift;
   return $self->{bus};
}

sub GetSlot
{
   my $self = shift;
   return $self->{slot};
}

sub GetFunction
{
  my $self = shift;
  return $self->{func};
}

sub GetName
{
  my $self = shift;
  return $self->{cfgTree}->{name};
}

sub GetClass
{
   my $self = shift;
   return hex($self->{cfgTree}->{class});
}

sub GetClassName
{
   my $self = shift;
   my $className = CLASS_NAME->{$self->GetClass()};
   if (defined($className)) {
      return $className;
   } else {
      return "Unknown class";
   }
}

sub GetDrivers
{
   my $self = shift;
   return $self->{drivers};
}

sub GetIrq
{
   my $self = shift;
   return $self->{cfgTree}->{irq};
}

sub GetOwner
{
   my $self = shift;
   return $self->{cfgTree}->{owner};
}

sub GetOptions
{
   my $self = shift;
   return $self->{cfgTree}->{options};
}

sub GetPCIString
{
   my $self = shift;
   return MakePCIString($self->GetBus(),
                        $self->GetSlot(),
                        $self->GetFunction());
}

sub GetVendorID
{
   my $self = shift;
   return hex($self->{cfgTree}->{vendor});
}

sub GetDeviceID
{
   my $self = shift;
   return hex($self->{cfgTree}->{devID});
}

sub GetSubsysVendorID
{
   my $self = shift;
   return hex($self->{cfgTree}->{subsysVendor});
}

sub GetSubsysDeviceID
{
   my $self = shift;
   return hex($self->{cfgTree}->{subsysDevID});
}


#
# Non-trivial Accessor Functions
#


########################################################################
#
# Device::GetEquiv --
#
#       This returns the name of the equivalence class for this device.
#       The equivalence class is associated with the drivers for the
#       device.  It is used when generating VMkernel names for
#       the device, and in the information passed to vmkload_mod.
#
#       Note that there is no SetEquiv().  See SetDriverInfo().
#
#       Note also that this method is useless (returns undef) if
#       driver information has not been set.
#
# Results:
#       The equivalence class of the device currently at this location.
#
# Side effects:
#       None.
#
########################################################################

sub GetEquiv
{
   my $self = shift;  # IN: Invoking instance.
   return $self->{equiv};
}


########################################################################
#
# Device::GetVMKDeviceName --
#
#       Get the VMkernel's name for the device, if any.
#       This may return a name even if the device is currently not
#       owned by the VMkernel (although as of ESX Dali, that is an
#       unusual circumstance to begin with for devices that can
#       be owned by the VMkernel at all).
#
# Results:
#       The name or undef if this device is not named.
#
# Side effects:
#       None.
#
########################################################################

sub GetVMKDeviceName
{
   my $self = shift;   # IN: Invoking instance.

   return $self->{cfgTree}->{vmkname};
}


########################################################################
#
# Device::GetConfigData --
#
#       Retrieve a data structure of the same form as that taken
#       by the constructor.  This makes it easy to set any changed
#       information from this object back into the config file.
#
# Results:
#       Configuration data structure (reference, not deep copy).
#
# Side effects:
#       None.
#       
########################################################################

sub GetConfigData
{
   my $self = shift;  # IN: Invoking instance.

   return $self->{cfgTree};
}


#
# Trivial Query Functions
#

sub IsNetClassDevice
{
   my $self = shift;  # IN: Invoking instance.
   my $devClass = ($self->GetClass() & MASK_BASE_CLASS);

   if ($devClass == PCI_BASE_CLASS_NETWORK) {
      return 1;
   } else {
      return 0;
   }
}

sub IsNetEthernetDevice
{
   my $self = shift;  # IN: Invoking instance.
   my $devSubclass = ($self->GetClass() & MASK_CLASS);

   if ($devSubclass == PCI_CLASS_NETWORK_ETHERNET) {
      return 1;
   } else {
      return 0;
   }
}

sub IsStorageClassDevice
{
   my $self = shift;
   my $devClass = ($self->GetClass() & MASK_BASE_CLASS);

   if ($devClass == PCI_BASE_CLASS_STORAGE) {
      return 1;
   } else {
      return 0;
   }
}

sub IsStorageSCSIDevice
{
   my $self = shift;
   my $devSubclass = ($self->GetClass() & MASK_CLASS);

   if ($devSubclass == PCI_CLASS_STORAGE_SCSI) {
      return 1;
   } else {
      return 0;
   }
}

sub IsStorageRAIDDevice
{
   my $self = shift;
   my $devSubclass = ($self->GetClass() & MASK_CLASS);

   if ($devSubclass == PCI_CLASS_STORAGE_RAID) {
      return 1;
   } else {
      return 0;
   }
}

sub IsSerialFiberDevice
{
   my $self = shift;
   my $devSubclass = ($self->GetClass() & MASK_CLASS);

   if ($devSubclass == PCI_CLASS_SERIAL_FIBER) {
      return 1;
   } else {
      foreach my $driver (@{$self->GetDrivers()}) {
         if (SERIAL_FIBER_DRIVERS->{$driver}) {
            return 1;
         }
      }
   }
   return 0;
}


sub IsNicEquiv
{
   my $self = shift;
   return ($self->GetEquiv() eq EQUIV_NIC);
}
    

sub IsScsiEquiv
{
   my $self = shift;
   return ($self->GetEquiv() eq EQUIV_SCSI);
}
    
sub IsIdeEquiv
{
   my $self = shift;
   return ($self->GetEquiv() eq EQUIV_IDE);
}

sub IsFcEquiv
{
   my $self = shift;
   return ($self->GetEquiv() eq EQUIV_FC);
}


########################################################################
#
# Device::Equals --
#
#       Compares this Device to another by member-wise comparison
#       and the hash function.
#
#       Note that name, irq, ownership, driver options and vmkname do
#       not factor into the equality relationship.
#
# Results:
#       True if these Devices are equal, false otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub Equals
{
   my $self = shift;  # IN: Invoking instance.
   my $dev = shift;   # IN: Other Device instance.

   return $self->GetPCIString() eq $dev->GetPCIString() &&
          $self->GetDeviceID() == $dev->GetDeviceID() &&
          $self->GetVendorID() == $dev->GetVendorID() &&
          $self->GetSubsysVendorID == $dev->GetSubsysVendorID() &&
          $self->GetSubsysDeviceID == $dev->GetSubsysDeviceID() &&
          $self->GetClass() == $dev->GetClass();
}


#
# Non-trivial Query Functions.
#


########################################################################
#
# Device::IsVMwareManageable --
#
#       Determine if we can manage this device by checking to see that
#       we have at least one driver for it.
#
#       Note that this function is meaningless if we have not yet
#       set driver information for this device.  See SetDriverInfo().
#
# Results:
#       True if we can manage the device, false otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub IsVMwareManageable
{
   my $self = shift;

   my $drivers = $self->GetDrivers();
   return $#$drivers > -1;
}


#
# Trivial Mutator Functions
#

sub SetName
{
   my $self = shift;
   $self->{cfgTree}->{name} = shift;
}

sub SetIrq
{
   my $self = shift;
   #
   # If unsetting the IRQ, should pass in the undef value
   #

   $self->{cfgTree}->{irq} = shift;
}

sub SetOptions
{
   my $self = shift;
   $self->{cfgTree}->{options} = shift;
}

sub SetVendorID
{
   my $self = shift;
   my $vendorNum = shift;
   $self->{cfgTree}->{vendor} = sprintf("%04x", $vendorNum);
}

sub SetDeviceID
{
   my $self = shift;
   my $deviceIdNum = shift;
   $self->{cfgTree}->{devID} = sprintf("%04x", $deviceIdNum);
}

sub SetClass
{
   my $self = shift;
   my $classNum = shift;
   $self->{cfgTree}->{class} = sprintf("%06x", $classNum);
}

sub SetSubsysVendorID
{
   my $self = shift;
   my $subsysVendorNum = shift;
   $self->{cfgTree}->{subsysVendor} = sprintf("%04x", $subsysVendorNum);
}

sub SetSubsysDeviceID
{
   my $self = shift;
   my $subsysDevIDNum = shift;
   $self->{cfgTree}->{subsysDevID} = sprintf("%04x", $subsysDevIDNum);
}


#
# Non-trival Mutator Functions.
#


########################################################################
#
# Device::SetDriverInfo --
#
#       Set driver-related information.  This includes both the
#       list of drivers and the equivalence class that they share.
#       This information is retrieved from a PCI::DeviceMap object,
#       which loads the map from a generated file.
#
#       Note that there is no GetDriverInfo() function- use
#       GetDrivers() and GetEquiv() instead.
#
# Results:
#       None.
#
# Side effects:
#       Sets driver and equivalence class information for the Device.
#
########################################################################

sub SetDriverInfo
{
   my $self = shift;
   my $deviceMap = shift;  # IN: PCI::DeviceMap object with driver info.

   LogDebug("Looking for drivers for device " . $self->GetPCIString(),
            2, "pci");
   my $mapEntries = $deviceMap->LookupDevice($self->GetVendorID(),
                                             $self->GetDeviceID());

   LogDebug("Matched " . (defined($mapEntries) ? scalar(@$mapEntries) : 0) .
            " entries.", 2, "pci");
   return unless $mapEntries;

   # Sort first by constraints matched, then by file order.
   # We do this by using the number of constraints as hash keys.
   # These keys then map to lists of entries with that number of
   # constraints. To extract a sorted list of entries, we first
   # sort by the keys, then concatenate the lists contained within
   # the hash.
   my %savedMapEntries = ();
   foreach my $entry (@$mapEntries) {
      my $pass = 1;
      my $constraints = $entry->GetConstraints();
      foreach my $key (keys(%$constraints)) {
         my $value = $constraints->{$key};
         LogDebug("Checking constraint $key=$value", 2, "pci");

         if ($key eq "subsys_vendorid") {
            unless ($self->GetSubsysVendorID() == $value) {
               $pass = 0;
               last;
            }
         } elsif ($key eq "subsys_deviceid") {
            unless ($self->GetSubsysDeviceID() == $value) {
               $pass = 0;
               last;
            }
         }
      }
      if ($pass) {
         my $count = scalar(keys(%$constraints));
         push(@{$savedMapEntries{$count}}, $entry);
      }
   }

   # Place the entries which matched the most constraints
   # at the first of the list.
   my @sortedMapEntries = ();
   foreach my $count (reverse(sort(keys(%savedMapEntries)))) {
      push @sortedMapEntries, @{$savedMapEntries{$count}};
   }
   
   LogDebug("Matched " . scalar(@sortedMapEntries) .
            " entries with constraints", 2, "pci");
   return unless @sortedMapEntries;

   $self->{drivers} = [ (map { $_->GetDriver(); } @sortedMapEntries) ];

   my @equivs = map { $_->GetEquiv(); } @sortedMapEntries;
   $self->{equiv} = $equivs[0];
   shift(@equivs);
   if (grep { $_ ne $equivs[0]; } @equivs) {
      #
      # The code that generates the map file *should* ensure that
      # this never happens.
      #

      Panic("Found drivers with multiple different equivalences " .
            "for device " . $self->GetPCIString());
   }
}


########################################################################
#
# Device::SetOwner --
#
#       Set the owner if a valid owner is supplied.
#
# Results:
#       None.
#
# Side effects:
#       The owner is set if the requested owner is valid.
#       An error is logged otherwise.
#
########################################################################

sub SetOwner
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $owner = shift; # IN: Owner to set.

   if (not defined $owner or
       $owner ne COS and $owner ne VMKERNEL) {
      $owner = '<undef>' if not defined $owner;
      LogError("Attempted to set invalid owner $owner for a PCI device.");
      return;
   }

   if ($owner eq COS) {
      LogDebug("Setting owner of device " . $self->GetPCIString() . " to " .
               COS() . ".", 2, "pci");
   } elsif ($owner eq VMKERNEL) {
      LogDebug("Setting owner of device " . $self->GetPCIString() . " to " .
               VMKERNEL() . ".", 2, "pci");
   }
   $self->{cfgTree}->{owner} = $owner;
}


########################################################################
#
# Device::SetVMKDeviceName --
#
#       Set the VMkernel's name for the device.
#
#       WARNING: This method should never be called by anyone other
#                than the DeviceManager if the name is in the form
#                of an auto-generated name (i.e. vmhba0, vmnic1).
#                The DeviceManager must track such names.  In general,
#                DeviceNames should be set through the DeviceManager.
#
# Results:
#       None.
#
# Side effects:
#       Sets the name associated with the given equivalence.
#
########################################################################

sub SetVMKDeviceName
{
   my $self = shift;   # IN/OUT: Invoking instance.
   my $name = shift;   # IN: The name to give it.

   $self->{cfgTree}->{vmkname} = $name;
}


########################################################################
#
# Device::ToString --
#
#       Returns device information in a string form.
#
# Results:
#       A human-readable string.
#
# Side effects:
#       None.
#
########################################################################

sub ToString
{
   my $self = shift;  # IN: Invoking instance.

   my $s =
      "Bus " . $self->{bus} . "," .
      " slot " . $self->{slot} . "," .
      " function " . $self->{func} .
      " class " . $self->{cfgTree}->{class} .
      "\n" .
      "Name: " . $self->{cfgTree}->{name} . "\n" .
      "Owner: " . $self->GetOwnerString() .
      (defined($self->{irq}) ? (", IRQ " . $self->{irq}) : "") . "\n" .
      "Vendor: " . $self->{cfgTree}->{vendor} .
      " Device ID: " . $self->{cfgTree}->{devID} . "\n" .
      ($self->GetSubsysVendor() != 0 ?
         ("Subsystem: " . $self->{cfgTree}->{subsysVendor} . ':' .
          $self->{cfgTree}->{subsysDevID} . "\n") :
         "");

   my @drivers = @{$self->GetDrivers()};
   my $equiv = $self->GetEquiv();
   if ($#drivers > -1) {
      $s .= "Drivers:";

      for(my $i = 0; $i <= $#drivers; $i++ ) {
         #
         # There used to be a different name for each driver,
         # but now there is one equivalence for all of them.
         #
         $s .= " " . $drivers[$i] . " (" . $equiv . ") ";
      }
      $s .= "\n";
   }
   return($s);
}


1;
