##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# DeviceManager.pm --
#
#       This class holds and manages a set of VMware::PCI::Device objects.
#       It allows for the construction of such a set from various sources,
#       and supports a number of other operations such as generating
#       and suggesting device names.
#

package VMware::PCI::DeviceManager;

use VMware::Log qw(:log);
use VMware::FileSys::StandardLocator qw(StdFile);
use VMware::PCI::Device;
use VMware::PCI::PCIString qw(:all);

use strict;

use constant VMK_DEVICE_TYPE_NIC => "vmnic";
use constant VMK_DEVICE_TYPE_HBA => "vmhba";


########################################################################
#
# DeviceManager::new --
#
#       Constructor.  Optionally takes the configuration tree
#       representing all devices from the main config file and builds
#       its internal set of devices from the tree.
#
# Results:
#       The new instance.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;              # IN: Invoking class name.
   my $deviceCfg = (shift || {});  # IN: Config tree, defaults to empty.

   my $self = {
      devices => {},
      devNameNumbers => {},
   };

   foreach my $str (keys %$deviceCfg) {
      if (defined($self->{devices}->{$str})) {
         LogError("Attempt to manage two devices at the same location '$str'!");
         return undef;
      }

      my $dev = new VMware::PCI::Device::($str, $deviceCfg->{$str});
      Panic("Could not construct device at '$str'!") unless ($dev);
      $self->{devices}->{$str} = $dev;

      unless (defined($self->{devices}->{$str})) {
         LogError("Could not build device object for device '$str'!");
         return undef;
      }
   }

   bless $self => $class;
   return $self;
}


########################################################################
#
# DeviceManager::Copy --
#
#       Constructs a new device manager instance based on this instance.
#
# Results:
#       A new DeviceManager identical to the original.
#
# Side effects:
#       None.
#
########################################################################

sub Copy
{
   my $self = shift;  # IN: Invoking instance (base for the copy).
   my $class = ref($self);

   my $copy = $class->new();
   $copy->{devNameNumbers} =
      VMware::DataStruct::Clone($self->{devNameNumbers});

   foreach my $str (keys %{$self->{devices}}) {
      $copy->{devices}->{$str} = $self->{devices}->{$str}->Copy();
   }
   return $copy;
}


########################################################################
#
# DeviceManager::AddDevice --
#
#       Add a VMware::PCI::Device to the manager.
#
#       We should not need to worry about device names and our
#       internal name generator because this device cannot contribute
#       a name until LoadDriverInfo() is called.  And that method
#       will handle the necessary updates.
#
# Results:
#       None.
#
# Side effects:
#       The device is added.
#
########################################################################

sub AddDevice
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $dev = shift;   # IN: The Device to add.

   $self->{devices}->{$dev->GetPCIString()} = $dev;
}


########################################################################
#
# DeviceManager::RemoveDevice --
#
#       Remove a device from the device manager.
#
# Results:
#       The device that was removed.
#
# Side effects:
#       The device is removed and its key in the hashtable deleted.
#
########################################################################

sub RemoveDevice
{
   my $self = shift;   # IN/OUT: Invoking instance.
   my $pciStr = shift; # IN: PCIString of the device to remove.

   my $dev = $self->{devices}->{$pciStr};
   delete $self->{devices}->{$pciStr};
   return $dev;
}


########################################################################
#
# DeviceManager::GetDevices --
#
#       This function returns a sorted list of references to PCI::Device
#       objects.
#
#       NOTE: This list consists of references to the PCI::Device objects
#       still held by the DeviceManager, so changes to the Devices are
#       automatically reflected.
#
# Results:
#       An array ref of references to the Device objects in the manager.
#
# Side effects:
#       None.
#
########################################################################

sub GetDevices
{
   my $self = shift;    # IN: Invoking instance.

   #
   # sort requires the use of package globals $a and $b in the functions.
   #

   local $a;
   local $b;
   my @devices = sort {
      my $rc;
      if ($rc = ($a->GetBus() <=> $b->GetBus())) {
         return $rc;
      } elsif ($rc = ($a->GetSlot() <=> $b->GetSlot())) {
         return $rc;
      } else {
         return ($a->GetFunction() <=> $b->GetFunction());
      }
   } (values(%{$self->{devices}}));
   return \@devices;
}


########################################################################
#
# DeviceManager::GetDevice --
#
#       Retrieve a reference to one PCIDevice object in the manager.
#       This takes a PCIString because that's how the devices are
#       indexed.  Call MakePCIString to build one from bus/slot/func.
#
# Results:
#       The requested Device, or undef if it is not present.
#
# Side effects:
#       None.
#
########################################################################

sub GetDevice
{
   my $self = shift;       # IN: Invoking instance.
   my $pciString = shift;  # IN: PCIString identifying the device.

   return $self->{devices}->{$pciString};
}


########################################################################
#
# DeviceManager::GetConfigData --
#
#       Return a configuration data structure of the same form used
#       in the constructor.  This makes it easy to update the config
#       file with any changes made to this object.
#
# Results:
#       The config data structure.  Contains refs to data within
#       the managed Device objects (not a deep copy).
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigData
{
   my $self = shift;  # IN: Invoking instance.

   my $data = {};
   foreach my $str (keys %{$self->{devices}}) {
      $data->{$str} = $self->{devices}->{$str}->GetConfigData();
   }
   return $data;
}

########################################################################
#
# DeviceManager::GenerateVMKDeviceNames --
#
#       Go through the managed devices and generate names for
#       all devices owned by the VMkernel that do not have names already.
#
#       The generated name will be based on the string ID of the
#       device type and a running number.
#
#       Note that the Devices must have driver information loaded
#       in order for this to be a usable method.  See the
#       LoadDriverInfo() method.
#
#       Note also that existing names are checked and
#       UpdateDevNameNumbers() is called at that time so we are
#       not in danger of stomping on existing names.
#
# Results:
#       None.
#
# Side effects:
#       VMK names are set, counters for constructing names are changed.
#
########################################################################

sub GenerateVMKDeviceNames
{
   my $self = shift;     # IN/OUT: Invoking instance.

   my @vmkDevs = grep {
      my $o = $_->GetOwner();
      $o eq VMware::PCI::Device::VMKERNEL();
   } @{$self->GetDevices()};

   foreach my $dev (@vmkDevs) {
      next if defined $dev->GetVMKDeviceName();

      my $equiv = $dev->GetEquiv();
      my $pciStr = $dev->GetPCIString();

      unless ($equiv) {
         if ($dev->IsVMwareManageable()) {
            LogError("No device equivalence exists for device '$pciStr'.");
         } else {
            LogError("Device at '$pciStr' is not manageable by the VMkernel " .
                     "but has been assigned to it.");
         }
         next;
      }

      my $stem;
      if ($dev->IsNicEquiv()) {
         $stem = VMK_DEVICE_TYPE_NIC;
      } elsif ($dev->IsFcEquiv() || $dev->IsScsiEquiv() || $dev->IsIdeEquiv()) {
         $stem = VMK_DEVICE_TYPE_HBA;
      } else {
         $stem = $equiv;
      }
      unless (defined $self->{devNameNumbers}->{$stem}) {
         $self->{devNameNumbers}->{$stem} = 0;
      }

      #
      # Append the current number to the prefix/stem to
      # construct the new name.
      #

      my $name = $stem . $self->{devNameNumbers}->{$stem};
      $self->SetVMKDeviceName($pciStr, $name);
   }
}


########################################################################
#
# DeviceManager::SetVMKDeviceName --
#
#       Set the VMkernel name for a given device.  This is the only
#       "public" interface to set the name on an individual device-
#       the method of the same name of the actual Device object should
#       not be called by anyone but this method.  This is to keep
#       track of the generated names and other names that conform
#       to their syntax.
#
#       This method assumes the caller knows what it is doing and will
#       not set a name that's already taken.  End-user interactive
#       tools should never allow this to be called directly with
#       user input.  It is for GenerateVMKDeviceNames and internal
#       debugging tools.
#
# Results:
#       None.
#
# Side effects:
#       Sets the VMkernel device name on the given device.
#       See also side effects for UpdateDevNameNumbers().
#
########################################################################

sub SetVMKDeviceName {
   my $self = shift;   # IN/OUT: Invoking instance.
   my $pciStr = shift; # IN: PCIString for the device to set.
   my $name = shift;   # IN: The name to set.

   $self->UpdateDevNameNumbers($name);
   $self->{devices}->{$pciStr}->SetVMKDeviceName($name);
}


########################################################################
#
# DeviceManager::UpdateDevNameNumbers --
#
#       Make certain that the numbers we use for generating
#       vmkernel device names are set correctly.  This means that
#       the number stored must be greater than all numbers currently
#       known to be in use for the base name.
#
# Results:
#       None.
#
# Side effects:
#       If the name matches the auto-generated format, ensure that
#       the number we store to use in the next auto-generated name
#       is greater than the number we're setting here.
#       
########################################################################

sub UpdateDevNameNumbers
{
   my $self = shift; # IN/OUT: Invoking instance.
   my $name = shift; # IN: The name to examine.

   if ($name =~ /^(\D+)(\d+)$/) {
      my $type = $1;
      my $num = $2;
      
      if (!exists($self->{devNameNumbers}->{$type}) ||
          $num >= $self->{devNameNumbers}->{$type}) {
         $name .= $self->{devNameNumbers}->{$type} = ++$num;
      }
   }
}


########################################################################
#
# DeviceManager::FindPCIStringByVMKName --
#
#       Looks up the PCIString (Bus:Slot.Function location) for the
#       given VMkernel device name.  There should be only one match.
#
# Results:
#       The PCIString, or undef if no matches.
#
# Side effects:
#       None.
#
########################################################################

sub FindPCIStringByVMKName
{
   my $self = shift; # IN: Invoking instance.
   my $name = shift; # IN: The name we're looking for.

   foreach my $str (keys(%{$self->{devices}})) {
      my $vmkdevname = $self->{devices}->{$str}->GetVMKDeviceName();
      if (defined($vmkdevname) && ($name eq $vmkdevname)) {
         return $str;
      }
   }
   return undef;
}


########################################################################
#
# DeviceManager::LoadDriverInfo --
#
#       Use the device driver map to find and set driver information
#       for the Devices in this DeviceManager.
#
# Results:
#       True on success, false if the DeviceMap could not be created.
#
# Side effects:
#       Sets driver information on all devices being managed.
#
########################################################################

sub LoadDriverInfo
{
   my $self = shift;  # IN/OUT: Invoking instance.

   my $mapFile = StdFile("pcidev_map", "esx");
   my $localMapFile = StdFile("pcidev_map_local", "esx");

   my @files = ($mapFile);
   push(@files, $localMapFile) if -r $localMapFile;

   my $deviceMap = new VMware::PCI::DeviceMap::(@files);
   return undef unless defined($deviceMap);

   foreach my $dev (@{$self->GetDevices()}) {
      $dev->SetDriverInfo($deviceMap);

      #
      # Check names for all devices, not just VMKERNEL owned ones.
      # A name should not get squashed just because the device
      # is temporarily assigned to the COS.
      #

      my $currentName = $dev->GetVMKDeviceName();
      if (defined($currentName)) {
         $self->UpdateDevNameNumbers($currentName);
      }
   }
   return 1;
}


1;
