##########################################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# DeviceMap.pm --
#
#       This class represents one or more PCI device map files.
#       These files are automatically generated from pci.xml.
#       They contain global configuration information, vendor
#       information, and device information about vmware-supported
#       PCI devices.
#

package VMware::PCI::DeviceMap;

use VMware::Log qw(:log);
use VMware::PCI::DeviceMapEntry;

use strict;

#
# The field definitions for the 2 fixed tables in the map file.
#

use constant VENDOR_FIELDS => {
   "type" =>       0,
   "vendor" =>     1,
   "short_name" => 2,
   "long_name" =>  3,
};

use constant DEVICE_FIELDS => {
   "type" =>   0,
   "vendor" => 1,
   "device" => 2,
   "equiv" =>  3,       # Equivalence class of the device.
   "desc" =>   4,
   "driver" => 5,
   "constraints" => 6,  # The only optional field
};


########################################################################
#
# DeviceMap::new --
#
#       Constructor.  If given one or more filenames for valid PCI
#       device map files, load their contents into the new instance.
#
#       Note that the contents of later files in the list will overwrite
#       matching contents from earlier files.
#
# Results:
#       The new instance, or undef if the files could not be loaded.
#
# Side effects:
#       File reading.
#
########################################################################

sub new
{
   my $class = shift;        # IN: Invoking class.
   my @deviceMapFiles = @_;  # IN: List of PCI device map files to
                             #     use in populating the new instance.

   my $self = {
      config => {},        # Table of configuration variables.
      vendors => [],       # Table of vendors.
      devices => [],       # Table of devices.
   };

   bless $self => $class;

   foreach my $file (@deviceMapFiles) {
      unless ($self->LoadDeviceMap($file)) {
         return undef;
      }
   }
   return $self;
}


########################################################################
#
# DeviceMap::GetVendorField --
#
#       Gets the index number of a named field in the vendor table.
#       This can be used in an array built from a vendor line.
#
# Results:
#       The integer index (possibly 0) or -1 if the field name
#       is invalid.
#
# Side effects:
#       None.
#
########################################################################

sub GetVendorField
{
   my $self = shift;      # IN: Invoking instance.
   my $fieldName = shift; # IN: Field to look up.

   if (exists(VENDOR_FIELDS->{$fieldName})) {
      return VENDOR_FIELDS->{$fieldName};
   } else {
      return -1;
   }
}


########################################################################
#
# DeviceMap::GetDeviceField --
#
#       Gets the index number of a named field in the device table.
#       This can be used in an array built from a device line.
#
# Results:
#       The integer index (possibly 0) or -1 if the field name
#       is invalid.
#
# Side effects:
#       None.
#
########################################################################

sub GetDeviceField
{
   my $self = shift;      # IN: Invoking instance.
   my $fieldName = shift; # IN: Field to look up.

   if (exists(DEVICE_FIELDS->{$fieldName})) {
      return DEVICE_FIELDS->{$fieldName};
   } else {
      return -1;
   }
}


########################################################################
#
# DeviceMap::GetConfigKeys --
#
#       Gets the keys (in an arbitrary order) from the config line.
#
# Results:
#       A list of keys (strings).
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigKeys
{
   my $self = shift; # IN: Invoking instance.
   return keys(%{$self->{config}});
}


########################################################################
#
# DeviceMap::GetConfigValue --
#
#       Gets the value for a given config key.
#
# Results:
#       The value (string).  Undef if the key does not exist.
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigValue
{
   my $self = shift;  # IN: Invoking instance.
   my $key = shift;   # IN: Config key to look up.

   return $self->{config}->{$key};
}


########################################################################
#
# DeviceMap::LookupDevice --
#
#       Returns all the DeviceMapEntry instances for which the
#       vendor and device ids match.
#
# Results:
#       An array of matching DeviceMapEntry objects.
#
#       Note:  This array contains refs to the data in this object,
#              So don't write to it.
#
# Side effects:
#       None.
#
########################################################################

sub LookupDevice
{
   my $self = shift;  # IN: Invoking instance.
   my $vend = shift;  # IN: The vendor to look up by.
   my $dev = shift;   # IN: The device ID to look up by.

   my @res;
   if ($vend && $dev) {
      @res = grep {
         $_->GetVendorID() == $vend && $_->GetDeviceID() == $dev;
      } @{$self->{deviceEntries}};
   }

   if (@res) {
      return \@res;
   } else {
      return undef;
   }
}


########################################################################
#
# DeviceMap::LookupVendor --
#
#       Finds the first vendor matching the ID passed in and returns
#       a ref to the array of strings representing its fields.
#
# Results:
#       The vendor string matching the given ID, or undef if not match.
#       vendor.
#
# Side effects:
#       None.
#
########################################################################

sub LookupVendor
{
   my $self = shift;  # IN: Invoking instance.
   my $vend = shift;  # IN: The vendor to look up.

   my @res;
   if ($vend) {
      @res = grep {
         $_->[VENDOR_FIELDS->{"vendor"}] == $vend;
      } @{$self->{vendors}};
   }

   if (@res) {
      return $res[0];
   } else {
      return undef;
   }
}


########################################################################
#
# DeviceMap::GetMapEntries --
#
#       Return the entire list of DeviceMapEntry objects from the map.
#
# Results:
#       A reference to the list of DeviceMapEntry objects.
#
#       Note:  This is a ref to the data in this object, *NOT* a
#              deep copy.  So don't write to it.
#
# Side effects:
#       None.
#
########################################################################

sub GetMapEntries
{
   my $self = shift;  # IN: Invoking instance.
   return $self->{deviceEntries};
}


########################################################################
#
# DeviceMap::ToString --
#
#       Produces a string form of the device map.  Note that this
#       string is *not* in quite the same format as the device map file.
#
# Results:
#       The string.
#
# Side effects:
#       None.
#
########################################################################

sub ToString
{
   my $self = shift; # IN: Invoking instance.
   my $s = "";

   my $rec;

   $s .= "Config Map\n";
   foreach $rec ($self->GetConfigKeys() ) {
      $s .= "$rec='" . $self->GetConfigValue($rec) . "'\n";
   }

   $s .= "Vendor Map\n";
   foreach $rec (@{$self->{vendors}}) {
      my $r;
      foreach $r (@$rec) {
         $s .= "'$r' ";
      }
      $s .= "\n";
   }

   $s .= "Device Map\n";
   foreach $rec (@{$self->{deviceEntries}}) {
      $s .= $rec->toString() . "\n";
   }

   return $s;
}


########################################################################
#
# DeviceMap::LoadDeviceMap --
#
#       Parse the given device map file and store its information
#       within this instance.  This adds to any existing information.
#
# Results:
#       True on success.  False if there was a problem opening,
#       reading or closing the file.
#
# Side effects:
#       The contents of the file are added to whatever was already
#       present in the invoking instance.
#
########################################################################

sub LoadDeviceMap
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $map = shift;   # IN: Device map file from which to read.

   my $maxvend = scalar(keys(%{VENDOR_FIELDS()})) - 1;
   my $maxdev = scalar(keys(%{DEVICE_FIELDS()})) - 1;

   if (! -r $map) {
      LogError("No read access to file $map: $!");
      return undef;
   }

   local *FD;
   unless(open(FD, "< $map")) {
      LogError("Could not open file $map: $!");
      return undef;
   }

   my $lineno = 0;         # Number of lines 
   while (my $line = <FD>) {
      chomp($line);
      $lineno++;

      #
      # Strip of comments (# and everything after) and problematic
      # trailing whitespace characters.
      #

      $line =~ s/\#.*//;
      $line =~ s/[\s\r]*$//;

      my @fields = split(/,/, $line);
      if ($#fields == -1) {
         next;
      }

      if ($fields[0] eq "config") {
         if ($#fields == 2) {
            my $key = $fields[1];
            my $value = $fields[2];
      
            # Store the configuration variable into the mapping.
            $self->{config}->{$key} = $value;
      
         } else {
            LogWarn("Wrong number of fields for config table " .
                    "on line $lineno.  Skipping.");
         }

      } elsif ($fields[0] eq "vendor") {
         if ($#fields == $maxvend) {
            if ($fields[VENDOR_FIELDS->{"vendor"}] =~ /0x([0-9a-f]+)/i) {
               $fields[VENDOR_FIELDS->{"vendor"}] = hex($1);
            }
            push(@{$self->{vendors}}, \@fields);

         } else {
            LogWarn("Wrong number of fields for vendor table " .
                    "on line $lineno.  Skipping.");
         }

      } elsif ($fields[0] eq "device") {
         if ($#fields >= $maxdev - 1) {

	   #
	   # IDE is for vmvisor only
	   #
	    if ($fields[DEVICE_FIELDS->{"driver"}] eq "ide.o") {
	       next;
	    }

            if ($fields[DEVICE_FIELDS->{"vendor"}] =~ /0x([0-9a-f]+)/i) {
               $fields[DEVICE_FIELDS->{"vendor"}] = hex($1);
            }
                
            if ($fields[DEVICE_FIELDS->{"device"}] =~ /0x([0-9a-f]+)/i) {
               $fields[DEVICE_FIELDS->{"device"}] = hex($1);
            }

            my $entry = VMware::PCI::DeviceMapEntry->new(@fields);
            push(@{$self->{deviceEntries}}, $entry);

         } else {
            LogWarn("Wrong number of fields for device table " .
                    "on line $lineno.  Skipping.");
         }
      }
      else {
         LogWarn("Unknown database type encountered " .
                 "on line $lineno.  Skipping.");
      }
   }

   unless (close(FD)) {
      LogError("Error closing file $map: $!");
      return undef;
   }
   return 1;
}


1;
