##########################################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# DeviceMapEntry.pm --
#

package VMware::PCI::DeviceMapEntry;

use strict;

use VMware::Log qw(:log);


########################################################################
#
# DeviceMapEntry::new --
#
#       Constructor that allows for initialization of all members.
#       Note:  Even if called on an existing instance, this does
#              not act a copy constructor.  Instead it just uses
#              the instance's type and procedes normally.
#
# Results:
#       A new instance initialized from the arguments.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;  # IN: Invoking class.

   my $self = {};
   $self->{type} = shift;         # IN: The type of entry (e.g. "device").
   $self->{vendor} = shift;       # IN: The vendor of this device.
   $self->{device} = shift;       # IN: The device ID.
   $self->{equiv} = shift;        # IN: The device equivalence class.
   $self->{desc} = shift;         # IN: The device description.
   $self->{driver} = shift;       # IN: The driver module.
   $self->{constraints} = shift;  # IN: A constraint string in the format
                                  #     key=value[;key=value]...
   bless $self => $class;
   return $self;
}


#
# Trivial Accessor Functions.
#

sub GetType
{
   my $self = shift;
   return $self->{type};
}

sub GetVendorID
{
   my $self = shift;
   return $self->{vendor};
}

sub GetDeviceID
{
   my $self = shift;
   return $self->{device};
}

sub GetEquiv
{
   my $self = shift;
   return $self->{equiv};
}

sub GetDesc
{
   my $self = shift;
   return $self->{desc};
}

sub GetDriver
{
   my $self = shift;
   return $self->{driver};
}


########################################################################
#
# DeviceMapEntry::GetConstraints --
#
#       Parses the constraints string and returns the information
#       in a hash format.
#
# Results:
#       A hash ref holding the parsed constraints.
#
# Side effects:
#       None.
#
########################################################################

sub GetConstraints
{
   my $self = shift;

   my $constraints = {};
   if (!$self->{constraints}) {
      return $constraints;
   }

   my @constraintStrings = split(/;/, $self->{constraints});
   my $str;
   foreach $str (@constraintStrings) {
      if ($str =~ /(\S+)=(\S+)/) {
         my $param = $1;
         my $value = $2;
         if ($value =~ /^0x([0-9a-f]+)$/i) {
            $value = hex($1);
         }
         $constraints->{$param} = $value;
      } else {
         Warning('Malformed constraint ' . $str);
      }
   }

   return $constraints;
}


########################################################################
#
# DeviceMapEntry::ToString --
#
#       Produce a string form of the map entry.
#
# Results:
#       The string.
#
# Side effects:
#       None.
#
########################################################################

sub ToString
{
   my $self = shift;  # IN: Invoking instance.
   my $s = "";

   $s .= "'" . $self->{type} . "',";
   $s .= "'" . sprintf("0x%04x", $self->{vendor}) . "',";
   $s .= "'" . sprintf("0x%04x", $self->{device}) . "',";
   $s .= "'" . $self->{equiv} . "',";
   $s .= "'" . $self->{desc} . "',";
   $s .= "'" . $self->{driver} . "',";

   my $c;
   my $constraints = $self->getConstraints();
   foreach $c (keys(%$constraints)) {
      my $v = $constraints->{$c};
      $s .= "'$c => " . sprintf("0x%04x", $v) . "' ";
   }

   return $s;
}


1;
