########################################################################## 
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
########################################################################## 

#
# DriverManager.pm --
#
#       A class that stores and managers driver information.
#

package VMware::PCI::DriverManager;

use VMware::Log qw(:log);
use VMware::PCI::Device;
use VMware::FileSys::StandardLocator qw(StdDir);

use strict;

########################################################################
#
# DriverManager::new --
#
#       Constructs a new DriverManager object.  A DriverManager object
#       is an array of hashes.  Each hash contains information about a
#       modules to be loaded.  The keys to each hash are
#
#       module: the name of the module.  ie. aic7xxx.o
#       options: driver-specific loading options.
#       enabled: whether the driver should load on boot
#
#       Formerly, there was an array of additonal parameters used
#       for shared devices, but we no longer have shared devices.
#
# Results:
#       A new PCIDriverManager instance consisting of drivers used by
#       the PCIDevices in this PCIDeviceList.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;    # IN: The class name.
   my $devMgr = shift;   # IN: A DeviceManager containing the devices
                         #     for which the driver list should be built.
   my $modTree = shift;  # IN: The existing module tree

   my $self = [];
   bless $self => $class;
   
   #  
   # Second load all those drivers from the config file but disable them to
   # ensure that removed devices are set to not load.  However build a temporary
   # database of driver state for later use.
   #
   my %driverState = ();
   foreach my $mod (keys %$modTree) {

      $driverState{$mod} = $modTree->{$mod}->{enabled};

      $self->AddDriver($mod, "false");
      
      my $modOptions = "";
      if (defined $modTree->{$mod}->{options}) {
         $modOptions = $modTree->{$mod}->{options};
      }
      
      $self->SetDriverOptions($mod, $modOptions);
   }

   my $devs = $devMgr->GetDevices();

   #
   # Get only devices owned by the VMkernel and have a driver an
   # equivalence (formerly driver name) to load.
   #

   my @vmkDevs = grep {
      $_->GetOwner() eq VMware::PCI::Device::VMKERNEL &&
         $_->IsVMwareManageable();
   } @$devs;

   #
   # Go through the vmk drivers
   #

   #
   # First we loop through once to find options that are device specific, which
   # we should clear from the options hash
   #
   foreach my $dev (@vmkDevs) {
      my $module = $dev->GetDrivers()->[0];
      my $options = $dev->GetOptions();
      $self->ClearDeviceOptions($module, $options);
   }


   #
   # Loop again to add the device speicific options in.
   # 
   foreach my $dev (@vmkDevs) {
      #
      # According to evan, choosing driver 0 is safe because there are
      # only multiple drivers if all of them will work. --handrews
      #

      my $module = $dev->GetDrivers()->[0];
      
      #new modules are always loaded 
      $self->AddDriver($module, "true");

      #We have disabled the drivers already in our map, so we are going to force
      #re-enable those that are not enabled.  This has the poor effect that when
      #someone disables a device driver explicitly, then scans we will re-enable
      #it, but this should be expected behaviour.
      if ($self->GetDriverEnabled($module) ne "true") {
         $self->SetDriverEnabled($module, "true");
      }

      my $options = $dev->GetOptions();
      $self->AddDriverOptions($module, $options);
   }

   return $self;
}


########################################################################
#
# DriverManager::GetDriverModules --
#
#       Gets all the driver modules that are in the list.
#
# Results:
#       An array ref consisting of the modules from each driver
#       in this list.
#
# Side effects:
#       None.
#
########################################################################

sub GetDriverModules
{
   my $self = shift;  # IN: Invoking instance.
   my @modules = map { $_->{module}; } @$self;
   return \@modules;
}


########################################################################
#
# DriverManager::GetDriverInternal --
#
#       Given a module, find the unique driver in this list that
#       is associated with that module.  Return the first one if
#       there are more than one as any of them should work.
#
#       This function should not be called externally
#
# Results:
#       A hash ref representing a driver, or undef on failure.
#
# Side effects:
#       None.
#
########################################################################

sub GetDriverInternal
{
   my $self = shift;   # IN: Invoking instance.
   my $module = shift; # IN: Module to match.

   my @driver = grep { $_->{module} eq $module; } @$self;

   if ($#driver == 0) {
      return($driver[0]);

   } elsif ($#driver > 0) {
      #
      # This should be enforced elsewhere, so just warn in case
      # something goes awry.
      #

      LogWarn("Duplicate driver found in the driver list.  " .
                 "Rep invariant violated.");
      return $driver[0];
   }
   return undef;
}


########################################################################
#
# DriverManager::GetDriverEnabled --
#
#       Gets whether a driver should be enabled given the module name.
#
# Results:
#       "true" if the driver should be enabled, false otherwise
#
# Side effects:
#       None.
#
########################################################################

sub GetDriverEnabled
{
   my $self = shift;   # IN: Invoking instance.
   my $module = shift; # IN: The module to match.

   my $driver = $self->GetDriverInternal($module);
   if (defined($driver)) {
      return $driver->{enabled};
   }
   return "true";
}

########################################################################
#
# DriverManager::SetDriverEnabled --
#
#       Sets whether a driver should be enabled given the module 
#       name.
#
# Results:
#       set the enabled flag to the value of the option, should be "true" 
#       or "false"
#
# Side effects:
#       None.
#
########################################################################

sub SetDriverEnabled
{
   my $self = shift;    # IN: Invoking instance.
   my $module = shift;  # IN: The module to match.
   my $enabled = shift; # IN: "true" or "false"
   
   if ($enabled ne "true" and $enabled ne "false") {
      #invalid, warn and return silently
      LogWarn("Invalid state given for module $module : $enabled");
      return;
   }

   my $driver = $self->GetDriverInternal($module);
   if (defined($driver)) {
      $driver->{enabled} = $enabled;
   }
}

########################################################################
#
# DriverManager::GetDriverOptions --
#
#       Gets an option string for configuring the devices
#       using this driver, suitable for appending to the
#       vmkload_mod command.  This string includes empty option
#       values to ensure that the right options are passed to
#       the right device in order.  In other words, if the 1st
#       device in bus:slot.function order sets option foo to 1, and
#       the second sets option bar to 2, while the third sets
#       options foo and zippy all to three, then the string is:
#
#       foo=1,,3 bar=,2, zippy=,,3
#
#       The settings are comma-separated, and an empty string between
#       commas (or before the first or after the last) denotes an
#       unset option.  Trailing commas are optional, but were easier
#       to implement left on.
#
# Results:
#       A string as described above, or an empty string if no drivers
#       match or no option names are present.
#
# Side effects:
#       None.
#
########################################################################

sub GetDriverOptions
{
   my $self = shift;   # IN: Invoking instance.
   my $module = shift; # IN: Module to match.

   my $driver = $self->GetDriverInternal($module);
   my @optionsList = ();
   if (defined($driver)) {
      foreach my $name (keys %{$driver->{options}}) {
         my $options = join(',', @{$driver->{options}->{$name}});
         push @optionsList, "$name=$options";
      }
   }
   return join(' ', @optionsList);
}

########################################################################
#
# DriverManager::SetDriverOptions --
#
#       Sets the options for this driver based on a string and put it 
#       the array format used by the internal objects of this class.
#
# Results:
#       Set the {options} portion of the driver to the correct values,
#       which should be a hash of arrays.  The has containing the key
#       and the array containing one or more comma sep values for the
#       key.
#
# Side effects:
#       None.
#
########################################################################

sub SetDriverOptions
{
   my $self = shift;    # IN: Invoking instance.
   my $module = shift;  # IN: Module to match.
   my $options = shift; # IN: Options to set.

   my $driver = $self->GetDriverInternal($module);
   my @optionsList = ();
   if (defined($driver)) {
      foreach my $option (split(' ', $options)) {
         if ($option =~ /^(\S+)=(\S+)$/) {
            my ($key, $value) = ($1, $2);
            
            $driver->{options}->{$key} = [];
            foreach my $subValue (split(',', $value)) {
               push(@{$driver->{options}->{$key}}, $subValue);
            }
         }
      }
   }
}

########################################################################
#
# DriverManager::AddDriver --
#
#       Adds a driver and its name to the list.  Only adds
#       a driver if it is unique.  Note that options and shared info
#       must be added separately.
#
# Results:
#       No return value.  Builds a new driver hash and adds it to
#       the internal list.
#
# Side effects:
#       Constructs new driver hash and adds it.
#
########################################################################

sub AddDriver
{
   my $self = shift;    # IN/OUT: Invoking instance.
   my $module = shift;  # IN: The module for the new driver.
   my $enabled = shift; # IN: Whether the driver is enabled or not.

   my $driver = $self->GetDriverInternal($module);
   if (!defined($driver)) {
      my $driver = {
         module => $module,
         enabled => $enabled,
         options => {},
         numDevices => 0,      # Number of devices using this driver.
      };
	   push(@$self, $driver);
   }
}

########################################################################
#
# DriverManager::ClearDeviceOptions --
#
#       Given a driver name clear the options on the driver option line
#       that will be regenerated from device specific settings.  For 
#       example if each device sets a Foo=X but the option Bar=2 is 
#       also set we want the resulting line to look like (for 3 devices)
#       Bar=2 Foo=X,X,X
#
#       Note that if a matching driver does not exist this method
#       silently returns.
#
# Results:
#       No return value.  Options passed in will be removed from the 
#       driver option hash
#
# Side effects:
#       Options passed in will be cleared.
#
########################################################################

sub ClearDeviceOptions
{
   my $self = shift;    # IN/OUT: Invoking instance.
   my $module = shift;  # IN: The module to which to add the options.
   my $options = shift; # IN: The options to add.

   my $driver = $self->GetDriverInternal($module);
   if (defined($driver)) {
      #
      # Device options are specified as "key1=value1 key2=value2..."
      # Convert into option list for all devices using this driver
      # Store as a hash from option name to list of values
      #

      my $used = {};
      foreach my $option (split(' ', $options)) {
         if ($option =~ /^(\S+)=(\S+)$/) {
            my ($key, $value) = ($1, $2);

            if (defined($driver->{options}->{$key})) {
               $driver->{options}->{$key} = [];
            }
         }
      }
   }
}


########################################################################
#
# DriverManager::AddDriverOptions --
#
#       Adds driver configuration options for a particular device.
#       For each option name, store a list of settings for the devices.
#       Pads the list appropriately with empty entries when a given
#       device does not use a particular driver option.  Pre-pad the
#       list when a new option is discovered on the 2nd device set
#       or later.
#
#       Note that if a matching driver does not exist this method
#       silently returns.
#
# Results:
#       No return value.  Options are added to the driver in the list
#       corresponding to the passed-in module.
#
# Side effects:
#       Options added as described above.
#
########################################################################

sub AddDriverOptions
{
   my $self = shift;    # IN/OUT: Invoking instance.
   my $module = shift;  # IN: The module to which to add the options.
   my $options = shift; # IN: The options to add.
   
   my $driver = $self->GetDriverInternal($module);
   if (defined($driver)) {
      #
      # Device options are specified as "key1=value1 key2=value2..."
      # Convert into option list for all devices using this driver
      # Store as a hash from option name to list of values
      #

      my $used = {};
      foreach my $option (split(' ', $options)) {
         if ($option =~ /^(\S+)=(\S+)$/) {
            my ($key, $value) = ($1, $2);
            $used->{$key} = 1;

            if (!defined($driver->{options}->{$key})) {
               #
               # First time we're seeing this option with this driver;
               # prepend blank list entries for preceding devices
               # This works even if $count is 0, as 1 .. 0 yields
               # an empty list.  
               #
               my $count = defined($driver->{numDevices}) ?
                  $driver->{numDevices} : 0;

               for (1 .. $count) {
                  push(@{$driver->{options}->{$key}}, '');
               }
            }
            push(@{$driver->{options}->{$key}}, $value);
            $driver->{devOptions}->{$key} = 1;
         }
      }
      ++$driver->{numDevices};

      #
      # Now insert blanks for options not used for this device
      # to ensure future options line up correctly.
      #

      foreach my $key (keys %{$driver->{options}}) {
         unless ($used->{$key} or not defined $driver->{devOptions}->{$key}) {
            push(@{$driver->{options}->{$key}}, '');
         }
      }
   }
}


########################################################################
#
# DriverManager::GetConfigData
#
#       Generates a config tree that can be set in a VmacoreConfigObj
#       that covers the information held in this class needed to
#       load the driver modules.
#
# Results:
#       The has ref for the root of the module tree.
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigData
{
   my $self = shift;  # IN: Invoking instance.
   
   my $data = {};
   
   foreach my $mod (@{$self->GetDriverModules()}) {
      my $options = $self->GetDriverOptions($mod);

      $data->{$mod}->{enabled} = $self->GetDriverEnabled($mod);
      $data->{$mod}->{options} = $options if $options ne '';
   }

   return $data;
}

1;
