##########################################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# HostInterface --
#
#       This class provides functions used to query PCI information
#       through host-supplied interfaces, notably lspci and /proc.
#

package VMware::PCI::HostInterface;

require Exporter;

use VMware::Log qw(:log);
use VMware::Panic qw(Panic);
use VMware::FileManip qw(FileSlurpArray);
use VMware::System qw(LogCommand);
use VMware::FileSys::StandardLocator qw(StdFile StdCommand);
use VMware::PCI::PCIString qw(:all);
use VMware::PCI::Device;
use VMware::PCI::DeviceManager;

use strict;

@VMware::PCI::HostInterface::ISA = qw(Exporter);
@VMware::PCI::HostInterface::EXPORT_OK =
   qw(GetKnownHostDevices GetDevicesFromHostProbe MakeCPCILine FromCPCILine);
%VMware::PCI::HostInterface::EXPORT_TAGS =
   (host => [qw(GetKnownHostDevices GetDevicesFromHostProbe)],
    cpci => [qw(MakeCPCILine FromCPCILine)],
    all => [@VMware::PCI::HostInterface::EXPORT_OK]);


########################################################################
#
# GetKnownHostDevices --
#
#       Return a DeviceManger with all devices known to the host
#       without probing.
#
# Results:
#       A DeviceManager containing the known devices.
#
# Side effects:
#       Panics if no devices are found.
#
########################################################################

sub GetKnownHostDevices
{
   my $pciMgr = ParseLSPCI(0);
   return $pciMgr if $pciMgr;

   #
   # Ok, we're desperate.  None of the current methods at finding PCI
   # devices work.  We're going to check a deprecated interface that doesn't
   # always give us all the info we need.
   #

   $pciMgr = ParseProcFile();
   return $pciMgr if $pciMgr;

   Panic("Could not find host PCI devices.  Cannot run lspci or " .
         "read the file " . StdFile('proc_bus_pci', 'esx') . ".");
}


########################################################################
#
# GetDevicesFromHostProbe --
#
#       Return a DeviceManager with all devices found by probing
#       through the host.
#
# Results:
#       A DeviceManager containing the found devices.
#
# Side effects:
#       Panics if no devices are found.
#
########################################################################

sub GetDevicesFromHostProbe
{
   my $pciMgr = ParseLSPCI(1);
   return $pciMgr if $pciMgr;
   Panic("Could not find PCI devices by probing with lspci.");
}


########################################################################
#
# ParseLSPCI --
#
#       Parses the output of lspci to detect PCI devices.
#
#       Depending on the $probe parameter, either probe for new devices,
#       (which may hang the system) or just read what we already know
#       we have.  In general, the COS needs to probe because it is not
#       automatically aware of all devices.  If we are booted into a
#       standard linux kernel, we can skip probing.
#
# Results:
#       A DeviceManager holding the detected devices, or undef
#       if no devices were detected.
#
# Side effects:
#       Runs lspci.  May hang the system if probing is requested.
#
########################################################################

sub ParseLSPCI
{
   my $probe = shift;  # IN: If true, probe for new devices (may hang system).
                       #     While there are two possible probe methods,
                       #     we have not encountered systems which need the
                       #     second, less reliable method.  So don't bother.

   my $lspciId = $probe ? 'probe_lspci' : 'safe_lspci';
   my $lspciNamesId = $probe ? 'probe_lspci_names' : 'safe_lspci_names';

   my $pciMgr = ParseLSPCIBasic($lspciId);
   return undef if not $pciMgr;

   unless (ParseLSPCINames($lspciNamesId, $pciMgr)) {
      LogWarn("Could not get the names of the PCI devices.");
   }
   return $pciMgr;
}


########################################################################
#
# ParseLSPCIBasic --
#
#       Parses the output from the lspci program which extracts
#       PCI information.  We find lspci and its options through
#       the StandardLocator package.
#
# Results:
#       A DeviceManager containing the found devices, or undef
#       if no devices were found.
#
# Side effects:
#       Runs lspci.  If probing options were requested (depends on
#       $lspciId), could hang the system, but this is unlikely.
#
########################################################################

sub ParseLSPCIBasic
{
   my $lspciId = shift;  # IN: The ID of the lspci command to look up.

   my $pciMgr = VMware::PCI::DeviceManager->new();


   my $lspciCommandList = StdCommand($lspciId, "esx");
   unless (defined($lspciCommandList)) {
      Panic("No lspci command defined!");
   }

   my $lspciCommand = join(' ', @$lspciCommandList);
   unless ($lspciCommand) {
      LogError("'$lspciCommand' is either not present or not executable.");
      return undef;
   }
   LogDebug("Entered ParseLSPCI with command '$lspciCommand'", 1, "pci");

   my @lines = split("\n", LogCommand($lspciCommand, 1));
   unless (@lines) {
      LogError("Could not run lspci!");
      return undef;
   }

   #
   # $dev is the current device we are parsing.  It is reconstructed
   # when we see the start of each new section.
   #
   my $dev = undef;
   my $numFound = 0;
   foreach my $line (@lines) {
      chomp($line);

      my $busSlotFuncRe = qr/([0-9a-f]+):([0-9a-f]+)\.([0-9a-f]+)/;
      my $classVendDevRe = qr/([0-9a-f]+):\s*([0-9a-f]+):([0-9a-f]+)/;
      if ($line =~ /^$busSlotFuncRe Class $classVendDevRe/) {
         #
         # Note that lspci uses hex numbers where we use decimal for
         # the internal representation of most numbers.
         #

         my $pciStr = MakePCIString(hex($1), hex($2), hex($3));
         my ($class, $vendor, $devid) = (hex($4), hex($5), hex($6));

         #
         # If we do not already have the device, add it.
         #

         $dev = $pciMgr->GetDevice($pciStr);
         unless ($dev) {
            $dev = new VMware::PCI::Device($pciStr);
            $pciMgr->AddDevice($dev);
            ++$numFound;
         }

         $dev->SetVendorID($vendor);
         $dev->SetDeviceID($devid);
         $dev->SetClass($class << 8);

      } elsif (defined($dev)) {
         #
         # Still working on the device $dev   
         # (subsequent line in section for the device)
         #

         if ($line =~ /IRQ\s*([0-9]+)/) {
            # Extract the IRQ information for this device
            $dev->SetIrq($1);
         }
         if ($line =~ /Subsystem: ([0-9a-f]+):([0-9a-f]+)/) {
            my $subsysVendor = hex($1);
            my $subsysDevice = hex($2);
            $dev->SetSubsysVendorID($subsysVendor);
            $dev->SetSubsysDeviceID($subsysDevice);
         }
      }
   }

   unless ($numFound) {
      LogDebug("Did not find any PCI devices from lspci.", 1, "pci");
      return undef;
   }

   return $pciMgr;
}


########################################################################
#
# ParseLSCPINames --
#
#       Parses the output from the lspci program extracting PCI device
#       names.  Set the names on the devices in the supplied DeviceManager.
#       We expect this DeviceManager to have been constructed by a previous
#       run of lspci so we warn about and ignore any devices not already
#       present in the manager.
#
# Results:
#       True on success, false otherwise.
#       Sets device names (not VMK device names) in the manger.
#
# Side effects:
#       Calls SetName() on all devices found that are already in the manager.
#
########################################################################

sub ParseLSPCINames
{
   my $lspciNamesId = shift;  # IN: The ID to use to find the command to run.
   my $pciMgr = shift;        # IN/OUT: The manager where we add the names.

   my $lspciCommandList = StdCommand($lspciNamesId, "esx");
   unless (defined($lspciCommandList)) {
      Panic("No lspci command defined!");
   }

   my $lspciCommand = join(' ', @$lspciCommandList);
   unless ($lspciCommand) {
      LogError("'$lspciCommand' is either not present or not executable.");
      return undef;
   }

   my @lines = split("\n", LogCommand($lspciCommand, 1));
   unless (@lines) {
      LogError("Could not run lspci!");
      return undef;
   }

   #
   # $dev is the current device we are parsing.  It is reconstructed
   # when we see the start of each new section.
   #
   my $dev = undef;
   my $numFound = 0;
   foreach my $line (@lines) {
      chomp($line);

      if ($line =~ /^([0-9a-f]+):([0-9a-f]+)\.([0-9a-f]+) [^:]*: (.+)/) {
         #
         # Note that lspci uses hex numbers where we use decimal for
         # the internal representation of most numbers.
         #

         my $pciString = MakePCIString(hex($1), hex($2), hex($3));
         my $name = $4;

         #
         # Check for existence of this device in the list.  If the device does
         # not exist, warn and move on.
         #
         $dev = $pciMgr->GetDevice($pciString);
         unless ($dev) {
            LogWarn("Found unexpected device at '$pciString' while " .
                    "reading names.");
            next;
         }
         ++$numFound;
         $dev->SetName($name);
      }
   }

   unless ($numFound) {
      LogDebug("Did not find any expected PCI devices from lspci (names).",
               1, "pci");
      return undef;
   }

   return 1;
}


########################################################################
#
# ParseProcFile --
#
#       XXX This function is deprecated, but is kept for last-ditch
#       efforts when lspci fails.
#
#       Parses the /proc/pci file to extract PCI information and build
#       a DeviceManager representing the devices found.
#
# Results:
#       A DeviceManager instance or undef on error.
#
# Side effects:
#       Reads from /proc.
#
########################################################################

sub ParseProcFile
{
   my $pciMgr = new VMware::PCI::DeviceManager::();

   my $proc = StdFile('proc_pci', 'esx');
   unless (defined($proc)) {
      Panic("No path to the proc node returned!");
   }
   LogDebug("Entered ParseProcFile with proc file '$proc'.");

   unless ($proc) {
      LogError("'$proc' is not present.");
      return undef;
   }

   my $lines = FileSlurpArray($proc);
   unless ($lines) {
      LogError("Could not run read from '$proc'!");
      return undef;
   }

   #
   # $dev is the current device we are parsing.  It is reconstructed
   # when we see the start of each new section.
   #
   my $dev = undef;
   my $numFound = 0;
   my $expectingName = 0;
   foreach my $line (@$lines) {
      chomp($line);

      if ($line =~
          /Bus[ ]*([0-9]+),[ ]*device[ ]*([0-9]+),[ ]*function[ ]*([0-9]+):/) {
         my $pciStr = MakePCIString($1, $2, $3);

         #
         # If we do not already have the device, add it.
         #

         $dev = $pciMgr->GetDevice($pciStr);
         unless ($dev) {
            $dev = new VMware::PCI::Device($pciStr);
            $pciMgr->AddDevice($dev);
            ++$numFound;
         }

         #
         # If it's in the /proc/pci file, the cos has it.
         #
         # TODO-handrews: This was only done on the non-new device
         #                path before.  Why?
         #

         $dev->SetOwner(VMware::PCI::Device::COS);

         #
         # The name of the device always follows the identification of the
         # device's bus, slot (device), and function numbers, so mark
         # that we are expecting it.
         #
         $expectingName = 1;

      } elsif (defined($dev) && $expectingName) {
         my $name = $line;
         $name =~ s/^\s*(.+)\s*$/$1/;
         $dev->SetName($name);
         $expectingName = 0;

      } else {
         if (defined($dev)) {
            if ($line =~ /IRQ[ ]*([0-9]+)/) {
               $dev->SetIrq($1);
            }
         }
      }
   } 

   unless ($numFound) {
      LogDebug("Found 0 PCI devices from /proc.", 1, "pci");
      return undef;
   }

   return $pciMgr;
}


########################################################################
#
# HostInterface::MakeCPCILine --
#
#       Make a cpci line based on the configuration of the devices
#       in the given PCI DeviceManager.
#
#       Builds a CPCI line of the form:
#          bus0:slot0[.function0],s1[.f1],...sn[.fn];bus1:...;...
#       from the PCI Devices that are currently held by the object.
#
#       Rules for CPCI line:
#
#       1. A blank CPCI setting means all devices go to host.
#       2. Any device that is listed must have its bus and slot number
#          passed to the host even if it shares the bus and slot with
#          a device allocated to the VMs.
#       3. If a bus is not mentioned, the host is given all the devices
#          on that bus.
#          (We do this by default to shorten kernel command line. 
#          The grub version we're using has a 256 char limit. 
#          Because of this, new cards on that bus will to the COS, but
#          our hardware change detection/reconf mechanism will fix that.)
#
#       4. If the bus is mentioned but no slots listed, all devices
#          are given to the VMs.
#       5. If the bus and slot are mentioned but no functions listed,
#          all functions are given to the host.
#
#
########################################################################

sub MakeCPCILine
{
   my $pciMgr = shift;    # IN: The DeviceManger holding the devices.
   my $cpciLine = "";

   #
   # Devices come sorted Bus:Slot:Function
   #
   my @devs = @{$pciMgr->GetDevices()};
   my @busDevs;
   my @slotDevs;
   my $vmOwner = VMware::PCI::Device::VMKERNEL;

   while (@devs) {
      my $addSemi = 1;
      my $bus = $devs[0]->GetBus();
      @busDevs = grep { $_->GetBus() == $bus; } @devs;
      @devs = grep { $_->GetBus() != $bus; } @devs;

      # Remove leading zeroes.
      $bus = $bus + 0;

      if ((grep { $_->GetOwner() ne $vmOwner } @busDevs) == 0) {
         # "B:" -- all devices on this bus go to the VMkernel
         $cpciLine .= "$bus:";

      } elsif ((grep { $_->GetOwner() eq $vmOwner } @busDevs) == 0) {
         # Nothing -- all devices on this bus go to the host
         $addSemi = 0;

      } else {
         # "B:S[.F], ... ,S[.F]" -- comma-separated list of slot[.function]
         $cpciLine .= "$bus:";
         my @slotFuncList = ();
         while (@busDevs) {
            my $slot = $busDevs[0]->GetSlot();
            @slotDevs = grep { $_->GetSlot() == $slot; } @busDevs;
            @busDevs = grep { $_->GetSlot() != $slot; } @busDevs;

            # Remove leading zeroes.
            $slot = $slot + 0;

            if ((grep { $_->GetOwner() eq $vmOwner } @slotDevs) == 0) {
               # "B:S" -- all functions on this device go to the host
               push @slotFuncList, "$slot";

            } else {
               # "B:S.F" -- this slot/function goes to the host
               while (@slotDevs) {
                  my $dev = shift(@slotDevs);
                  my $function = $dev->GetFunction() + 0; # Remove leading zeroes.
                  if ($dev->GetOwner() ne $vmOwner) {
                     push @slotFuncList, "$slot.$function";
                  }
               }
            }
         }
         $cpciLine .= join(",", @slotFuncList);
      }
      if ($addSemi) {
         $cpciLine .= ";";
      }
   }
   return $cpciLine;
}


########################################################################
#
# HostInterface::FromCPCILine --
#
#       Sets the devices to a given state based on the CPCI line.
#       The "inverse" function of the MakeCPCILine to a limited extent.
#
#       Notes:
#          1. Ignore the - character.  Not necessary in this case.
#
#          2. If more than one device uses the same bus and device number,
#             we'll just allocate it to the host even if previously only
#             some of the devices were allocated to the host.  We have no
#             way to remember the previous setting unless we create a
#             separate host configuration file.
#
#          3. Algorithm: (needed to match semantics of CPCI)
#             - Set all the devices as being owned by the host.
#             - When a bus is listed, set all devices on the bus as owned
#               by the VMs.
#             - If the bus and slot is listed, give it to the host.
#
# Results:
#       Sets ownership of the devices in the DeviceManager to match the
#       given CPCI line.
#       True (1) on success, false (0) on failure.
#
# Side effects:
#       PCI Devices may have their ownership changed.
#
########################################################################

sub FromCPCILine
{
   my $cpci = shift;    # IN: The CPCI line to parse.
   my $pciMgr = shift;  # IN/OUT: The manager whose devices need ownership.

   my $rc = 1;
   LogDebug("Entered FromPCILine with line '$cpci'.", 1, "pci");

   #
   # We can safely ignore any whitespace and '-' in the CPCI string
   # Initialization: Devices on unlisted buses are managed by the host.
   # Then split the CPCI line by bus for further processing.
   #

   $cpci =~ s/[\s-]//g;
   my @devs = @{$pciMgr->GetDevices()};
   foreach my $dev (@devs) {
      $dev->SetOwner(VMware::PCI::Device::COS);
   }

   my @busSpecs = split(";", $cpci);
   foreach my $busSpec (@busSpecs) {
      if ($busSpec eq "") {
         next;
      } elsif ($busSpec =~ /^(\w+):([\w.,*]*)$/) {
         my $bus = $1;
         my $slotStr = $2;
         my @busDevs = grep { $_->GetBus() == $bus; } @devs;

         #
         # Initialization: unlisted devices on this bus are managed by
         # the VMkernel.
         #

         foreach my $dev (@busDevs) {
            $dev->SetOwner(VMware::PCI::Device::VMKERNEL);
         }

         if ($slotStr eq "") {
            #
            # "B:" -- all devices on this bus are managed by the VMkernel.
            # So do nothing further.
            #

         } elsif ($slotStr eq "*") {
            #
            # "B:*" -- all devices on this bus are managed by the COS.
            #

            foreach my $dev (@busDevs) {
               $dev->SetOwner(VMware::PCI::Device::COS);
            }

         } else {
            #
            # "B:S[.F], ... ,D[.F]" -- comma-separated list of
            # slot[.function] which requires further parsing.
            #

            $rc = FromCPCILineSlotSpec($bus, $slotStr, @busDevs);
         }
      } else {
         LogError("Invalid CPCI entry '$busSpec'.");
         $rc = 0;
      }
   }

   unless ($rc) {
      #
      # We had a problem.  Set everything back to the default.
      #

      foreach my $dev (@devs) {
         $dev->SetOwner(VMware::PCI::Device::COS);
      }
   }
   return $rc;
}


########################################################################
#
# HostInterface::FromCPCILineSlotSpec --
#
#       Helper function for FromCPCILine to handle the most complex
#       case, where there is a comma-separated list of slot[.function]
#       following the bus (i.e. B:S[.F], ... ,S[.F] form).
#
# Results:
#       Sets ownership of the devices in the DeviceManager to match the
#       given CPCI line.  Should only be called from FromCPCILine().
#       True (1) on success, false (0) on failure.
#
# Side effects:
#       PCI Devices may have their ownership changed.
#
########################################################################

sub FromCPCILineSlotSpec
{
   my $bus = shift;     # IN: The bus whose slot spec we are parsing.
   my $slotStr = shift; # IN: The slot spec portion for a particular
                        #     bus from the CPI line.
   my @busDevs = @_;    # IN: The devices matching $bus.

   my $rc = 1;
   my @slotSpecs = split(",", $slotStr);
   foreach my $slotSpec (@slotSpecs) {
      my @slotDevs;

      if ($slotSpec eq "") {
         @slotDevs = ();

      } elsif ($slotSpec =~ /^(\w+)$/) {
         #
         # "S" -- match slot with any function
         #

         my $slot = $1;
         @slotDevs = grep { $_->GetSlot() == $slot; } @busDevs;

      } elsif ($slotSpec =~ /^(\w+)\.(\w+)$/) {
         #
         # "S.F" -- match slot and function
         #

         my $slot = $1;
         my $function = $2;
         @slotDevs = grep {
            $_->GetSlot() == $slot && $_->GetFunction() == $function;
         } @busDevs;

      } else {
         LogError("Invalid CPCI entry '$slotSpec' for bus '$bus'.");
         $rc = 0;
      }

      #
      # Assign all matching devices to the console
      #

      foreach my $dev (@slotDevs) {
         $dev->SetOwner(VMware::PCI::Device::COS);
      }
   }
   return $rc;
}


1;
