##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# LinuxDevice.pm --
#
#       A basic class that contains information representing a device as
#       listed by kudzu in hwconf.
#       Information includes bus #, device/slot #, function #, name, class.
#

package VMware::PCI::LinuxDevice;

use VMware::Log qw(:log);
use VMware::PCI::PCIString qw(:all);

use strict;


########################################################################
#
# LinuxDevice::new --
#
#       Constructor for a VMware::PCI::LinuxDevice object.
#
# Results:
#       The new object, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;         # IN: Invoking class.
   my $entry = (shift || {}); # IN: Hash of hwconf entry keys/values.

   my $self = {
      entry => $entry
   };

   bless $self => $class;
   return $self;
}


########################################################################
#
# LinuxDevice::Copy --
#
#       Constructs a new LinuxDevice instance based on this instance.
#
# Results:
#       A new LinuxDevice identical to the original.
#
# Side effects:
#       None.
#
########################################################################

sub Copy
{
   my $self = shift;  # IN: Invoking instance (base for the copy).
   my $class = ref($self);

   my $copy = $class->new();
   %{$copy->{entry}} = %{$self->{entry}};

   return $copy;
}


#
# Trivial Accessor Functions
#

sub GetBus
{
   my $self = shift;
   return hex($self->{entry}->{pcibus});
}

sub GetSlot
{
   my $self = shift;
   return hex($self->{entry}->{pcidev});
}

sub GetFunction
{
  my $self = shift;
   return hex($self->{entry}->{pcifn});
}

sub GetClass
{
   my $self = shift;
   return $self->{entry}->{class};
}

sub GetDriver
{
   my $self = shift;
   return $self->{entry}->{driver};
}

sub GetDevice
{
   my $self = shift;
   return $self->{entry}->{device};
}

sub GetDescription
{
   my $self = shift;
   return $self->{entry}->{desc};
}

sub GetVendorID
{
   my $self = shift;
   return hex($self->{entry}->{vendorId});
}

sub GetDeviceID
{
   my $self = shift;
   return hex($self->{entry}->{deviceId});
}

sub GetSubsysVendorID
{
   my $self = shift;
   return hex($self->{entry}->{subVendorId});
}

sub GetSubsysDeviceID
{
   my $self = shift;
   return hex($self->{entry}->{subDeviceId});
}

sub GetPCIString
{
   my $self = shift;
   return MakePCIString($self->GetBus(),
                        $self->GetSlot(),
                        $self->GetFunction());
}

sub GetNetworkHWAddr
{
   my $self = shift;
   return $self->{entry}->{"network.hwaddr"};
}

#
# Trivial Query Functions
#

sub IsNetClassDevice
{
   my $self = shift;  # IN: Invoking instance.
   if ($self->GetClass() eq "NETWORK") {
      return 1;
   } else {
      return 0;
   }
}

1;
