##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# LinuxDeviceManager.pm --
#
#       This class holds and manages a set of VMware::PCI::LinuxDevice
#       objects as listed in a kudzu hwconf file (such as
#       /etc/sysconfig/hwconf.
#

package VMware::PCI::LinuxDeviceManager;

use VMware::Log qw(:log);
use VMware::FileSys::StandardLocator qw(StdCommand StdFile);
use VMware::PCI::LinuxDevice;
use VMware::System qw(LogCommand);

use strict;


########################################################################
#
# LinuxDeviceManager::new --
#
#       Constructor.  Optionally takes the filename of the hwconf file
#       to load; otherwise will load the system hwconf.
#
# Results:
#       The new instance, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;      # IN: Invoking class name.
   my $filename = shift;   # IN: The file to load; defaults to system file.

   my $self = {
      devices => [],
   };
   bless $self => $class;

   unless (defined $filename) {
      $filename = StdFile('hwconf', 'all');
   }

   unless ($self->LoadFile($filename)) {
      return undef;
   }

   return $self;
}


########################################################################
#
# DeviceManager::GetDevices --
#
#       This function returns a list of references to VMware::PCI::LinuxDevice
#       objects in the order they were encountered in the file.
#
#       NOTE: This list consists of references to the device objects
#       still held by the LinuxDeviceManager, so changes to the devices are
#       automatically reflected.
#
# Results:
#       An array ref of references to the device objects in the manager.
#
# Side effects:
#       None.
#
########################################################################

sub GetDevices
{
   my $self = shift;    # IN: Invoking instance.

   my @devices = @{$self->{devices}};
   return \@devices;
}


########################################################################
#
# DeviceManager::GetDevice --
#
#       Retrieve a reference to one PCIDevice object in the manager
#       with the same bus/slot/func as the specified PCIString.
#
# Results:
#       The requested Device, or undef if it is not present.
#
# Side effects:
#       None.
#
########################################################################

sub GetDevice
{
   my $self = shift;       # IN: Invoking instance.
   my $pciString = shift;  # IN: PCIString identifying the device.

   # Not all of the entries in the file have bus/slot/function numbers,
   # so we can't index on PCIString.
   foreach my $device (@{$self->{devices}}) {
      if ($device->GetPCIString() eq $pciString) {
         return $device;
      }
   }

   return undef;
}


########################################################################
#
# LoadFile --
#       Loads a linux hwconf file.
#
# Results:
#       On success, sets the device list to the devices found in the file
#       and returns 1. Otherwise returns 0.
#
# Side effects:
#       Reads hwconf
#
########################################################################

sub LoadFile
{
   my $self = shift;       # IN: Invoking instance.
   my $filename = shift;   # IN: File to load from.

   unless (-r $filename) {
      LogError("Hardware configuration file '$filename' is not readable or does not exist.");
      return 0;
   }

   my $fh;
   unless (open($fh, "<$filename")) {
      LogError("Could not open file $filename: $!");
      return 0;
   }

   my @devices = ();
   my $entry = undef;

   while (<$fh>) {
      my $line = $_;
      chomp($line);

      if ($line eq "-") {
         # New entry delimiter
         if (defined $entry and $entry->{bus} eq "PCI") {
            push @devices, new VMware::PCI::LinuxDevice($entry);
         }

         $entry = {};
      } elsif ($line =~ /^([\w\.]+):\s+([^"]+)$/) {
         # Entry without quotes
         $entry->{$1} = $2;
      } elsif ($line =~ /^([\w\.]+):\s+\"(.*)\"$/) {
         # Entry with quoted values
         $entry->{$1} = $2;
      }
   }

   if (defined $entry and $entry->{bus} eq "PCI") {
      push @devices, new VMware::PCI::LinuxDevice($entry);
   }

   unless (close($fh)) {
      LogError("Could not close file $filename: $!");
      return 0;
   }

   $self->{devices} = \@devices;

   return 1;
}


########################################################################
#
# LinuxDeviceManager::UpdateConfig --
#       Updates the standard linux hwconf file using kudzu.
#
# Results:
#       On success, updates hwconf and returns 1. Otherwise returns 0.
#
# Side effects:
#       Runs kudzu
#
########################################################################

sub UpdateConfig {
   my $cmdList = StdCommand('kudzu', 'all');
   unless (defined $cmdList) {
      LogError("Could not get kudzu helper.");
      return 0;
   }

   my $cmd = join(' ', @$cmdList);
   unless (LogCommand($cmd)) {
      LogError("Execution of '$cmd' failed.  See log messages above.");
      return 0;
   }

   return 1;
}

1;
