##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# PCIInfo.pm --
#

package VMware::PCI::PCIInfo;

require Exporter;

use VMware::Log qw(:log);
use VMware::System qw(LogCommand);

use VMware::PCI::PCIString qw(:all);
use VMware::PCI::Device;
use VMware::PCI::DeviceManager;
use VMware::PCI::HostInterface qw(:all);

use strict;

@VMware::PCI::PCIInfo::ISA = qw(Exporter);
@VMware::PCI::PCIInfo::EXPORT_OK =
   qw(GetPCIDeviceManagerFromConfig SetPCIDeviceDataInConfig
      FindPCIDeviceManager);


########################################################################
#
# PCIInfo::GetPCIDeviceManagerFromConfig --
# 
#       Construct a VMware::PCI::DeviceManager from the esx.conf
#       configuration object.  Exists to prevent other packages
#       (like VMware::Boot::*) from needing to know where in the config
#       file the dev mgr info lives.
#
# Results:
#       A DeviceManager, or undef if the constructor failed or no
#       devices were found.
#
# Side effects:
#       None.
#
########################################################################

sub GetPCIDeviceManagerFromConfig
{
   my $config = shift;  # IN: The esx.conf configuration object.

   my $devMgr = new VMware::PCI::DeviceManager($config->GetTree(['device']));
   unless (defined($devMgr) && @{$devMgr->GetDevices()}) {
      return undef;
   }
   return $devMgr;
}


########################################################################
#
# PCIInfo::SetPCIDeviceDataInConfig --
#
#       Inverse of GetPCIDeviceManagerFromConfig, again existing
#       to prevent other packages from needing to know the config path.
#
# Results:
#       True if $config data changed, false if there was an error.
#
# Side effects:
#       Replaces the current device information in $config with
#       the information from $devMgr.
#
########################################################################

sub SetPCIDeviceDataInConfig
{
   my $config = shift;  # IN/OUT: The esx.conf configuration object.
   my $devMgr = shift;  # IN: The DeviceManager with the new data.

   return $config->SetTree(['device'], $devMgr->GetConfigData());
}


########################################################################
#
# PCIInfo::FindPCIDeviceManager --
#
#       This function constructs a view of the PCI devices and their
#       drivers based on both VMware configuration files and on
#       standard COS interfaces.  It produces a single view based on
#       the following logic:
#
#       If we have config files but cannot get information from the COS,
#       log a warning but go ahead and accept the config file
#       information as valid.
#
#       If we have information from the COS but no config file, assume
#       we must be doing a new installation and simply use the COS info.
#
#       If we have information from both, compare the two.  Remove all
#       device information from the config file for
#       devices which are different or not present in the COS info.
#       Then use the COS info for changed or added devices.
#
#       Once the DeviceManager is set up, load driver information
#       into it and run some sanity checks before returning it.
#
# Results:
#       A VMware::PCI::DeviceManager object representing the devices
#       in the system.  Changes made during the construction of
#       this object are *not* reflected in the configuration object.
#
#       undef on error.
#
# Side effects:
#       The configuration object passed in is updated to match the
#       device configuration in the DeviceManager.
#
########################################################################

sub FindPCIDeviceManager
{
   my $config = shift;  # IN: ESX Config instance to use.  Not modified
                        #     in this function- caller must save the
                        #     changes from the returned DeviceManager
                        #     if that is desired.
   my $noDetect = shift;  # IN: If true, just accept whatever's currently
                          #     in the config file and don't try to detect
                          #     actual hardware and compare.
   my $oldCPCI = shift; # IN: The previously configured CPCI line.

   my $devMgr = GetPCIDeviceManagerFromConfig($config);
   if ($noDetect) {
      unless ($devMgr->LoadDriverInfo()) {
         LogError("Could not load driver information for PCI Devices");
         return undef;
      }
      return $devMgr;
   }

   my $cosDevMgr = GetKnownHostDevices();

   #
   # If we could not detect PCI devices from the COS, warn and use the
   # previously configured devices if we have them.  If we don't have
   # them, then return undef because there's nothing else to do.
   #
   # If we were successful in detecting PCI devices from the COS.
   # If we have both, figure out the right merged set.  Otherwise,
   # take the COS set and figure out ownership if we can.
   #

   if (!defined($cosDevMgr)) {
      if (!defined($devMgr)) {
         LogError("Could not detect PCI devices and could not find any prior " .
                  "configuration.  Cannot proceed because nothing about the " .
                  "PCI devices could be determined.");
         return undef;

      } else {
         LogWarn("Could not detect PCI devices, but a prior configuration " .
                 "was found in the database.\nCould not determine if any new " .
                 "PCI devices were added to the machine.\nHowever, if " .
                 "nothing has changed on the machine, this does not matter.");
         return $devMgr;
      }

   } elsif (defined($devMgr)) {
      #
      # Driver information must be loaded before a Reconcile is done
      # so that VMkernel device names can be preserved across certain kinds
      # of configuration changes.
      #

      unless ($cosDevMgr->LoadDriverInfo() && $devMgr->LoadDriverInfo()) {
         LogError("Could not load driver information for PCI Devices");
         return undef;
      }
      ReconcileDeviceManagers($devMgr, $cosDevMgr);
   } else {
      $devMgr = $cosDevMgr;
      FindDeviceOwnership($devMgr, $config, $oldCPCI);
      unless ($devMgr->LoadDriverInfo()) {
         LogError("Could not load driver information for PCI Devices");
         return undef;
      }
   }

   CheckPCIDevicesWithCurrent($devMgr, 0);
   return $devMgr;
}


########################################################################
#
# PCIInfo::FindDeviceOwnership --
#
#       Given a DeviceManager lacking ownership information,
#       use the CPCI line from the boot information to figure
#       out some reasonable values.  If we are booted into VMnix,
#       the CPCI line will tesll us what VMnix knows about which
#       tells us that anything else belongs to the VMkernel.
#
# Results:
#       No return code.  Devices in the manager parameter may get owners.
#
# Side effects:
#       Devices in the manager may get owners.
#
########################################################################

sub FindDeviceOwnership
{
   my $devMgr = shift;  # IN/OUT: The device manager to work with.
   my $config = shift;  # IN: The esx.conf configuration object.
   my $cpci = shift; # IN: The previously configured CPCI line.

   LogDebug("Could not find a prior configuration.  Attempting to " .
            "generate information about the PCI devices by detecting the " .
            "current configuration information.", 1, "pci");

   unless (defined($cpci)) {
      LogWarn("Could not recover CPCI information.");
      LogWarn("Could not gather ownership information from the CPCI line.");
      LogWarn("Using default ownership.");
      return;
   }

   unless (FromCPCILine($cpci, $devMgr)) {
      LogWarn("Could not gather ownership information from the CPCI line.");
      LogWarn("Using default ownership.");
   }
}


########################################################################
#
# PCIInfo::ReconcileDeviceManagers --
#
#       In the case where we have device information from two sources
#       we need to produce one to go forward with.  This method treats
#       one as our primary and merges in new or changed devices from
#       the other.
#
# Results:
#       No return code.  1st parameter contains the updated
#       device manager.
#
# Side effects:
#       1st DeviceManager is updated.
#
########################################################################

sub ReconcileDeviceManagers
{
   my $devMgr = shift;     # IN/OUT: The original device manager to update
                           #         with new or changed devices.
   my $newDevMgr = shift;  # IN: Device manager with possible new or
                           #     changed devices.
   #
   # These map statements create a two-level hash table for each
   # of the two device sets we are dealing with.  The first level
   # maps PCIStrings to the 2nd level.  The second level contains
   # the device and a flag to indicate if we've touched the
   # entry in the processing algorithm below.
   #

   my %devSet = map {
      $_->GetPCIString() => {
         dev => $_,
         touched => 0,
      };
   } @{$devMgr->GetDevices()};
   my %newDevSet = map {
      $_->GetPCIString() => {
         dev => $_,
         touched => 0,
      };
   } @{$newDevMgr->GetDevices()};

   foreach my $str (keys %devSet) {
      if (defined $newDevSet{$str}) {
         $newDevSet{$str}->{touched} = 1;
         unless ($newDevSet{$str}->{dev}->Equals($devSet{$str}->{dev})) {
            #
            # Use the COS version of the device instead.
            # AddDevice() will replace the existing Device with the new one.
            # If the devices are of the same type (network vs. storage),
            # preserve the vmkernel name.
            #

            my $newDev = $newDevSet{$str}->{dev};
            my $oldDev = $devSet{$str}->{dev};
            $devMgr->AddDevice($newDev);
            if ($newDev->IsNicEquiv() && $oldDev->IsNicEquiv() ||
                ($newDev->IsScsiEquiv() || $newDev->IsFcEquiv() || $newDev->IsIdeEquiv()) &&
                ($oldDev->IsScsiEquiv() || $oldDev->IsFcEquiv() || $oldDev->IsIdeEquiv())) {
               #
               # Note that SetVMKDeviceName() must only be called through
               # the DeviceManager, as calling it directly on the Device
               # will cause the DeviceManager to lose track of the name,
               # and there is a possibility it will get assigned to another
               # Device later on.
               #

               $devMgr->SetVMKDeviceName($str, $oldDev->GetVMKDeviceName());
            }
         }
      } else {
         $devMgr->RemoveDevice($str);
      }
   }

   #
   # All COS devices that we did *not* see during the above algorithm
   # are new devices and should be added to the primary manager.
   #
   my @newList = map { $_->{dev} } grep { !$_->{touched} } values %newDevSet;
   foreach my $dev (@newList) {
      $devMgr->AddDevice($dev);
   }

   LogDebug("The configuration had to be updated because a change in " .
            "the hardware configuration was detected.", 2, "pci");
}


########################################################################
#
# PCIInfo::CheckPCIDevicesWithCurrent --
#
#       Perform some sanity checks before returning the device list.
#       These are not the best checks, but it's an imperfect world.
#       Returns the number of warnings found.
#
#       Turn on the fix flag to have the function change the ownership
#       settings to match that of the current setup.
#
# Results:
#       The number of warnings issued.
#
# Side effects:
#       If fixing, may change ownership on devices in the manager.
#
########################################################################

sub CheckPCIDevicesWithCurrent
{
   my $devMgr = shift;  # IN/OUT: Device manager.  Only OUT if $fix.
   my $fix = shift;     # IN: If true, fix up the errors.

   my $warnings = 0;

   my %deviceTouched;
   my %deviceManageable;

   my $devList = $devMgr->GetDevices();
   foreach my $dev (@$devList) {
      if ($dev->IsVMwareManageable()) {
         #
         # We want to determine if the whole card is manageable,
         # which is true if any function on the card is manageable.
         # This is why we have to do a complete pass over the devices
         # before running any of the checks.
         #

         my $slotStr = MakePCIString($dev->GetBus(), $dev->GetSlot(), 0);
         $deviceManageable{$slotStr} = 1;
      }
      $deviceTouched{$dev->GetPCIString()} = 0;
   }

   #
   # Check to make sure that devices that are being managed
   # by the VMkernel are really manageable.
   #

   foreach my $dev (@$devList) {

      #
      # A device can only be owned by VM or shared if we can manage it
      #
      if ($dev->GetOwner() eq VMware::PCI::Device::VMKERNEL and
          !$dev->IsVMwareManageable()) {
         #
         # Make sure that that there is not another function on the card that
         # is manageable.
         #
         # TODO-handrews:  Why is it OK for this function to be marked
         # VMkernel-owned if it is not manageable, even if some other
         # function on the card is?
         #

         my $slotStr = MakePCIString($dev->GetBus(), $dev->GetSlot(), 0);
         if (!defined($deviceManageable{$slotStr}) ||
             $deviceManageable{$slotStr} != 1) {

            LogWarn("Device (" . $dev->GetPCIString() . ") " .
                    $dev->GetClassName() . " is owned " .
                    $dev->GetOwner() . " but it should not be " .
                    "because it is an unsupported device.");
            $warnings++;

            if ($fix) {
               $dev->SetOwner(VMware::PCI::Device::COS);
            }
         }
      }
   }

   #
   # Check to make sure that devices owned by the host are listed as being
   # owned by the host
   #

   my $hostDevMgr = GetKnownHostDevices();
   my $hostDevList = $hostDevMgr->GetDevices();
   foreach my $hdev (@$hostDevList) {
      my $dev = $devMgr->GetDevice($hdev->GetPCIString());
      unless ($dev) {
         LogWarn("Host reports that it has a PCI device that was not " .
                 "found in a prior configuration nor during the probe " .
                 "of the devices.  Device is located at " .
                 $hdev->GetPCIString() . ".  Ignoring this device.");
         $warnings++;
         next;
      }

      # Mark each device as being touched
      $deviceTouched{$dev->GetPCIString()} = 1;
   }

   # Each device that was not touched should not be set to HOST or SHARED
   foreach my $dev (@$devList) {
      if ($deviceTouched{$dev->GetPCIString()} == 0) {
         if ($dev->GetOwner() eq VMware::PCI::Device::COS) {
            LogWarn("Device (" . $dev->GetPCIString() . ") " .
                    $dev->GetClassName() .  " is owned by " .
                    $dev->GetOwner() . " but is not detected by the hosts.");
            $warnings++;

            if ($fix) {
               $dev->SetOwner(VMware::PCI::Device::VMKERNEL);
            }
         }
      }
   }

   return($warnings);
}


1;
