##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# PCIString.pm --
#
#       This module provides translation to and from a string representation
#       of a PCI device's bus, slot and function numbers.  This string
#       is suitable for use in configuration files and as an identification
#       string.
#


package VMware::PCI::PCIString;

require Exporter;

use VMware::Log qw(:log);
use VMware::Panic qw(Panic);

use strict;

@VMware::PCI::PCIString::ISA = qw(Exporter);

@VMware::PCI::PCIString::EXPORT_OK =
   qw(PCI_STRING_REGEXP MakePCIString SplitPCIString);
%VMware::PCI::PCIString::EXPORT_TAGS =
   (all => [qw(PCI_STRING_REGEXP MakePCIString SplitPCIString)]);


#
# This can be used to match the string in a config file,
# which is useful when filtering or merging configs.
# Because Perl constants are rather limited, you have to
# assign this off into a variable to interpolate it.
#

use constant PCI_STRING_REGEXP => qr/\d{3}:\d{2}\.\d/;


########################################################################
#
# PCIString::MakePCIString --
#
#       Given bus, slot and func numbers, produce a string for
#       use in a config file or as a convenient ID.
#
# Results:
#       The string, or undef if not everything was supplied.
#
# Side effects:
#       If requested, will Panic() if not everything was supplied.
#
########################################################################

sub MakePCIString
{
   my $bus = shift;    # IN: The bus number of the device.
   my $slot = shift;   # IN: The device number.
   my $func = shift;   # IN: The function number of the device.
   my $fatal = shift;  # IN: If true, Panic() if there is a problem.

   unless (defined($bus) && defined($slot) && defined($func)) {
      #
      # Quiesce warnings about using undefined variables in strings.
      #

      map { $_ = '<undef>' if !defined($_) } ($bus, $slot, $func);
      LogError("One of bus '$bus', slot '$slot' or function '$func' " .
               "is undefined.\nCannot create PCI string.");
      Panic() if $fatal;
      return undef;
   }
   {
      #
      # Briefly turn off warnings since we expect one if any of $bus, $slot
      # or $func is not a number.
      #

      local $^W = 0;
      foreach my $n ($bus, $slot, $func) {
         if ((0 == $n) && ($n !~ /0+/)) {
            LogError("One of bus '$bus', slot '$slot' or function '$func' " .
                     "is not a number.\nCannot create PCI string.");
            Panic() if $fatal;
            return undef;
         }
      }
   }

   return sprintf("%03d:%02d.%1d", $bus, $slot, $func);
}


########################################################################
#
# PCIString::SplitPCIString --
#
#       Given a string produced by MakePCIString, split it back into
#       its components.
#
# Results:
#       A list of (bus, slot, function) numbers, or an empty list
#       if the string was not of the correct format.
#
# Side effects:
#       If requested, will Panic() if the string was not of the
#       correct format.
#
########################################################################

sub SplitPCIString
{
   my $string = shift;  # IN: The string to split.
   my $fatal = shift;   # IN: If true, Panic() if there is a problem.

   if (not defined $string) {
      LogError("PCI string is undefined.");
      Panic() if $fatal;
      return ();
   }
   if ($string =~ /^(\d{3}):(\d{2})\.(\d)$/) {
      return ($1,  # Bus number.
              $2,  # Slot number.
              $3); # Function number.
   }
   LogError("PCI string '$string' is not in the expected format.");
   Panic() if $fatal;
   return ();
}


1;
