##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# Panic.pm --
#
#       This module provides a Panic function that will terminate the
#       current process, calling any registered callbacks before doing so.
#


package VMware::Panic;

require Exporter;

use VMware::Log qw(:log);

use strict;

@VMware::Panic::ISA = qw(Exporter);

@VMware::Panic::EXPORT_OK =
   qw(Panic RegisterPanicCallback RemovePanicCallback DumpStackTrace);

%VMware::Panic::EXPORT_TAGS = (
   all => [qw(Panic RegisterPanicCallback RemovePanicCallback DumpStackTrace)],
);


{
   #
   # Holds the callbacks needed to be made before actually exiting.
   #
   my @callbacks = ();

   ########################################################################
   #
   # Panic::Panic --
   #
   #       Kill the current process, with logging, stack trace and
   #       callbacks on the way.
   #
   # Results:
   #       Does not return.
   #
   # Side effects:
   #       Termination.
   #
   ########################################################################

   sub Panic
   {
      my $string = shift;  # IN: Diagnostic message.
      my $exitCode = (shift || -1); # IN: Custom exit code.

      LogError($string) if defined $string;
      LogError("Panic!  Stack trace follows:");
      DumpStackTrace(VMware::Log::ID_ERROR);
      LogError("Making panic callbacks...");
      foreach my $cb (@callbacks) {
         if (ref($cb->{callback}) eq "ARRAY") {
            my $object = $cb->{callback}->[0];
            my $method = $cb->{callback}->[1];
            $object->$method(@{$cb->{args}});

         } else {
            $cb->{callback}->(@{$cb->{args}});
         }
      }
      LogError("Done, exiting with code $exitCode.  Goodbye!");
      exit($exitCode);
   }


   ########################################################################
   #
   # Panic::RegisterPanicCallback --
   #
   #       Adds a callback function or object/method pair and its arguments
   #       to the list to be called when we Panic.
   #
   # Results:
   #       None.
   #
   # Side effects:
   #       Adds the callback to the list.
   #
   ########################################################################

   sub RegisterPanicCallback
   {
      my $callback = shift;  # IN: Sub ref or array ref where the first
                             #     entry is an object and the second is
                             #     the method name.
      my $args = shift;      # IN: Arguments to be passed to the callback. (array ref)
      my $tag = shift;       # IN: Tag that can be used to remove the callback later.
                             #     Need not be unique.

      $args = [] unless defined $args;
      unshift(@callbacks, { callback => $callback, args => $args, tag => $tag });
   }


   ########################################################################
   #
   # Panic::RemovePanicCallback --
   #
   #       Remove all callbacks associated with the given tag from the
   #       callback list.  This can remove callbacks from the middle
   #       or either end of the list, not just the most recently registered.
   #
   # Results:
   #       None.
   #
   # Side effects:
   #       Removes the identified callbacks.
   #
   ########################################################################

   sub RemovePanicCallback
   {
      my $tag = shift;  # IN: The identifier of the callbacks to remove.

      my $i = 0;
      while ($i < @callbacks) {
         if (defined($callbacks[$i]->{tag}) && ($tag eq $callbacks[$i]->{tag})) {
            splice(@callbacks, $i, 1);
         } else {
            ++$i;
         }
      }
   }

}



########################################################################
#
# Panic::DumpStackTrace --
#
#       Log a stack trace to the given channel (defaults to LogInfo).
#
# Results:
#       None.
#
# Side effects:
#       Logs the stack trace.
#
########################################################################

sub DumpStackTrace
{
   my $channel = shift;

   $channel = VMware::Log::ID_INFO unless defined $channel;
   my $trace = "";
   my $i;
   while (my ($package, $filename, $line, $subroutine) = caller($i++)) {
      $trace .= "$package $subroutine in $filename line $line\n";
   }
   Log($channel, $trace);
}


1;
