##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# Prompt.pm --
# 
#       Utilities for interactive scripting.
#

package VMware::Prompt;

require Exporter;

@VMware::Prompt::ISA = qw(Exporter);
@VMware::Prompt::EXPORT_OK = qw(PromptUser CheckValidSet);
%VMware::Prompt::EXPORT_TAGS = (all => [(@VMware::Prompt::EXPORT_OK)]);

use strict;


########################################################################
#
# Prompt::PromptUser --
#       Prompts the user for input allowing a validator callback to
#       process the results.
#
# Results:
#       The user's input.
#
# Side effects:
#       Terminal I/O.
#
########################################################################

sub PromptUser
{
   my $prompt = shift;     # IN: The string to print as prompt, not including
                           #     the characters ': ' which will be appended.
   my $error = shift;      # IN: The string to print if an unexpected response
                           #     is passed (according to the validator).
   my $default = shift;    # IN: Default response (optional).
   my $validator = shift;  # IN: Callback for validating response.
   my @validatorArgs = @_; # IN: Extra args for callback.

   $prompt .= ": ";
   print $prompt;

   my $res = "";
   while ($res = <STDIN>) {
      $res =~ s/^\s*(.*)\s*$/$1/;

      if (($res eq "") && defined($default)) {
         $res = $default;
      }
      if (&$validator($res, @validatorArgs)) {
         last;
      } else {
         print $error;
      }

      print $prompt;
   } 
   return $res;
}


########################################################################
#
# Prompt::CheckValidSet --
#
#       Callback for use with PromptUser().  Expects the result
#       followed by an array of legal values, and verifies that
#       the result is in the array.  Case sensitive.
#
# Results:
#       True if the result is in the array, false otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub CheckValidSet
{
   my $val = shift;  # IN: Value to check.
   my $set = shift;  # IN: Set (array ref) in which to check.

   return grep { $_ eq $val; } @$set;
}


1;
