##########################################################################
# Copyright 2003 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# System.pm --
#
#	Routines for interacting with the operating system,
#	with abstractions for variations between platforms.
#

package VMware::System;

use strict;

use Sys::Hostname;
use File::Copy;

use VMware::Log qw(:log);
use VMware::FileManip qw(:filemanip);
use VMware::Config qw(:config);
use VMware::FileSys::Tmp qw(:tmp);


require Exporter;

@VMware::System::ISA = qw(Exporter);

@VMware::System::EXPORT_OK =
   qw(LogCommand LogCommandGetOutput LogCommandGetOutputArray
      LogCommandErr LogCommandErrWithFiles SetRunOnBoot ClearRunOnBoot Reboot
      GetMachineName GetPlatform CheckForRH30Cos GetOSName);

%VMware::System::EXPORT_TAGS = (
	command => [qw(LogCommand LogCommandGetOutput LogCommandGetOutputArray
                  LogCommandErr LogCommandErrWithFiles)],
   platform => [qw(GetMachineName GetPlatform CheckForRH30Cos GetOSName)],
   reboot => [qw(SetRunOnBoot ClearRunOnBoot Reboot)],
);


###########################################################################
#
# System::SetWindowsBoot --
#
#       For Windows, sets or clears a command to run on the next boot.
#       If $bootCmd is defined, sets a command to run that command with
#       any options passed in $options.  Otherwise, clears any existing
#       command.
#
# Results:
#       True on success, undef on failure.
#
# Side effects:
#       As described above.
#
###########################################################################

sub SetWindowsBoot
{
   my $bootCmd = shift;         # IN: The program to execute, or undef to
                                #     remove any existing command.
   my $options = shift;         # IN: options for the program.

   eval {
      require Win32::Registry;
   };
   if ($@) {
      LogError("Couldn't load Win32::Registry: $@");
      return undef;
   }

   my $hkey;
   my $ret = $main::HKEY_CURRENT_USER->Create(
      "Software\\Microsoft\\Windows\\CurrentVersion\\RunOnce", $hkey);

   unless ($ret) {
      LogError("Couldn't access RunOnce registry key: $!");
      return undef;
   }

   if ($bootCmd) {

      # REG_SZ = 1
      $ret = $hkey->SetValueEx("VMWARE_BOOT_CMD", undef, 1, $bootCmd);
      unless ($ret) {
         LogError("SetBootCommand: Couldn't add VMQA value to RunOnce key: $!");
         return undef;
      }
   } else {
      # Check whether VMWARE_BOOT_CMD value is present
      my %values;
      $hkey->GetValues(\%values);

      if (exists $values{"VMWARE_BOOT_CMD"}) {
	 $ret = $hkey->DeleteValue("VMWARE_BOOT_CMD");
	 unless ($ret) {
	    LogError("SetBootCommand: Couldn't remove value in RunOnce: $!");
	    return undef;
	 }
      }
   }

   $hkey->Close();

   return 1;
}


###########################################################################
#
# System::SetLinuxBoot --
#
#       For Linux, sets or clears a command to run on the next boot.
#       If $bootCmd is defined, sets a command to run that command with
#       any options passed in $options.  Otherwise, clears any existing
#       command.
#
# Results:
#       True on success, undef on failure.
#
# Side effects:
#       As described above.
#
###########################################################################

sub SetLinuxBoot
{
   my $bootCmd = shift;         # IN: The program to execute, or undef to
                                #     remove any existing command.
   my $options = shift;         # IN: options for the program.

   unless ((GetPlatform() eq "vmnix") ||
           (GetOSName() =~ /Red Hat Linux/) ||
           (GetOSName() =~ /Red Hat Enterprise Linux/)) { 
      LogError("Host ". GetOSName() . 
               " not supported for vmqa boot invocation.");
      return undef;
   }

   my $marker = "####VMQA####";
   my $rcLocal = "/etc/rc.d/rc.local";
   my $backupFile = "/etc/rc.d/rc.local.save";

   #
   # Read in existing file
   #
   unless (open(RCLOCAL, "<$rcLocal")) {
      LogError("Can't read $rcLocal: $!");
      return undef;
   }
   my @lines = <RCLOCAL>;
   unless (close(RCLOCAL)) {
      LogError("Can't close $rcLocal: $!");
      return undef;
   }

   #
   # Check for an existing reboot callback.
   #
   my $existing = 0;
   for (my $i=0; $i < @lines; $i++) {
      if ($lines[$i] =~ /$marker/) {
         $existing = 1;
         splice(@lines, $i);
         last;
      }
   }

   if ($bootCmd) {
      $bootCmd = "$bootCmd $options" if $options;

      #
      # Make sure $HOME is set when we start back up
      #
      if (defined($ENV{HOME})) { 
         $bootCmd = "HOME=$ENV{HOME} $bootCmd";
      }

      #
      # If no existing callback, make a backup
      #
      unless ($existing) {
         if (!copy($rcLocal, $backupFile)) {
            LogError("Can't copy $rcLocal to $backupFile: $!");
            return undef;
         }
      }

      #
      # Append the needed lines
      #
      push @lines, "$marker\n";
      push @lines, "$bootCmd &\n";
      push @lines, "cp -f $backupFile $rcLocal\n";
      push @lines, "rm -f $backupFile\n";

   } else {
      #
      # If removing the callback, and none was found, we're done.
      #
      return 1 unless ($existing);
   }

   #
   # Write the new file.
   #
   unless (open(RCLOCAL, ">$rcLocal")) {
      LogError("Can't write $rcLocal: $!");
      return undef;
   }
   print RCLOCAL @lines;
   unless (close(RCLOCAL)) {
      LogError("Can't close $rcLocal: $!");
      return undef;
   }

   return 1;
}


###########################################################################
#
# System::SetRunOnBoot --
#
# 	Sets up a program to be executed on the next restart for Windows
# 	or Linux.
#
# Results:
#       As with SetWindowsBoot or SetLinuxBoot, depending on $^O.
#
# Side effects:
#       As with SetWindowsBoot or SetLinuxBoot, depending on $^O.
#
###########################################################################

sub SetRunOnBoot
{
   my $bootCmd = shift;  # IN: The command to execute, or undef to clear.
   my $options = shift;  # IN: The options for the command.

   if ($^O eq "MSWin32") {
      return SetWindowsBoot($bootCmd, $options);
   } else {
      return SetLinuxBoot($bootCmd, $options);
   }
}


###########################################################################
#
# System::ClearRunOnBoot --
#
#       Removes any existing OS callback to re-invoke a program on the
#       next reboot.
#
# Results:
#       As with SetWindowsBoot(undef) or SetLinuxBoot(undef),
#       depending on $^O.
#
# Side effects:
#       As with SetWindowsBoot(undef) or SetLinuxBoot(undef),
#       depending on $^O.
#
###########################################################################

sub ClearRunOnBoot
{
   if ($^O eq "MSWin32") {
      return SetWindowsBoot(undef);
   } else {
      return SetLinuxBoot(undef);
   }

   return 1;
}


###########################################################################
#
# System::Reboot --
#
# 	Initiates a graceful reboot of the system (Windows or Linux)
#
# Results:
#       Does not return.
#
# Side effects:
# 	Exits the perl process immediately after initiating reboot.
#
###########################################################################

sub Reboot
{
   LogInfo("Rebooting in 10 seconds...");
   sleep 10;
   if ($^O eq "MSWin32") {

      eval {
	 require Win32;
      };
      if ($@) {
	 LogError("Reboot: Couldn't load Win32: $@");
      }

      my $machine = Win32::NodeName();
      my $ret = Win32::InitiateSystemShutdown($machine, undef, 3, 1, 1);

      unless ($ret) {
	 LogError("Failed to start system shutdown: $!");
      }

   } else {
      unless (LogCommand("/sbin/shutdown -r now")) {
	 LogError("Error shutting down system: $!");
      }
   }

   # This should ensure that open filehandles are closed.
   exit(0);
}


###########################################################################
#
# System::GetMachineName --
#
# 	Returns this machine's network node name.
#
# Results:
#       The machine's network node name.
#
# Side effects:
#       None.
#
###########################################################################

sub GetMachineName
{
   return (split /\./, Sys::Hostname::hostname)[0];
}


###########################################################################
#
# System::GetPlatform --
#
# 	Return a string representing this machine's platform
#
# 	windows - 	All MSWin32 operating systems.
# 	linux -		All Linux operating systems
# 	vmnix -		ESX
#
# Results:
#       The platform string as described above.
#
# Side effects:
#       Reads /proc/version on Linux.
#
###########################################################################

sub GetPlatform
{
   if ($^O eq "MSWin32") {
      return "windows";
   } else {
      if (-f "/proc/version") {
         my $kversion = FileSlurp("/proc/version");
         if ($kversion && $$kversion =~ /(?:vmnix|2\.4\.9-(?:13|34))/) {
            return "vmnix";
         } else {
            return "linux";
         }
      }
   }
}


########################################################################
#
# System::CheckForRH30Cos --
#
#    Check whether we have a RH3 EL based COS or not
#
# Results:
#    Returns 1 if yes; undef otherwise.
#
# Side effects:
#    None
#
########################################################################

sub CheckForRH30Cos
{
   if ($^O eq "MSWin32") {
      return undef;
   } 

   #
   # Check under different proc nodes
   #
   if (-f "/proc/sys/kernel/osrelease") {
      my $osrelease = FileSlurp("/proc/sys/kernel/osrelease");
      chomp $$osrelease;
      if ($$osrelease =~ /2\.4\.21-4\.ELvmnix/) {
         return 1;
      }
   } 
   
   #
   # Try  best to determine the version
   #
   if (-f "/proc/version") {
      my $kversion = FileSlurp("/proc/version");
      if ($kversion && $$kversion =~ /2\.4\.21-4\.ELvmnix/) {
         return 1;
      } 
   }
   return undef;
}


###########################################################################
#
# System::GetOSName --
#
#       Return a string describing this machine's operating system,
#       or "Unknown" if it cannot be determined. On Windows if we can we use 
#       a 3rd party utility to return a accurate description of the OS.
#       If that utility is not installed we use cmd.
#
# Results:
#       The string as described above.
#
# Side effects:
#       Runs a command on Windows, examines files in /etc on Linux.
#
###########################################################################

sub GetOSName
{
   if ($^O eq "MSWin32") {
      my $hostOS = "Windows ";
      my $psinfoCmd = File::Spec->catfile($ENV{TESTWAREWINROOT},
                                          "bin",
                                          "psinfo");
      unless (open(CMD, "$psinfoCmd|")) {
         LogWarn("Cannot determine Host OS: psinfo.exe not in \%PATH\%");
         unless (open(CMD, "cmd /C ver|")) {
            LogError("Can't run cmd to get version: $!");
            return "Unknown Windows";
         }
         my @lines = <CMD>;
         unless (close(CMD)) {
            if ($!) {
               LogWarn("Could not close pipe to cmd.exe: $!");
            } else {
               LogWarn("Version command failed: $@");
            }
            return "Unknown Windows"
            }
         chomp @lines;
         return join("", @lines);
      }          
      my @lines = <CMD>;
      unless (close(CMD)) { 
         if ($!) { 
            LogWarn("Could not close pipe to psinfo.exe")
               and return "Unknown Windows";
         }
      }
      foreach my $line (@lines) {
         if ($line =~ /Kernel version/) { 
            $line =~ /Microsoft Windows (\w+)\,/; 
            # Needed for MS Windows Server editions
            if (!defined $1) {
               $line =~ /Microsoft Windows Server (\w+)\,/;
               $hostOS .= $1;
            } else {
               # Note I need an else here as the $1
               # are in different scopes.
               $hostOS .= $1;
            }
         }
         if ($line =~ /Product type/) {
            my @tmp = split(/:/, $line);  
            $tmp[1] =~ s/(\s+)/ /;
            chomp($tmp[1]);
            $hostOS .= $tmp[1];
            
         }
         if ($line =~ /Service pack/) { 
            my @tmp = split(/:/, $line);
            $hostOS .= " w/SP ".int($tmp[1]);
         }
      }
      return $hostOS;
   } else {
      if (-e "/usr/sbin/vmkfstools") {
         return "VMware ESX Server";
      } else {
         foreach my $osFile ("/etc/redhat-release", "/etc/SuSE-release") {
            if (-e $osFile) {
               unless (open(OSFILE, "<$osFile")) {
                  LogWarn("Couldn't open $osFile: $!");
                  return undef;
               }
               my $osLine = <OSFILE>;
               close(OSFILE);
               chomp $osLine;
               return $osLine;
            }
         }
      }
      LogError("Unknown operating system");
      return "Unknown";
   }
}


###########################################################################
#
# System::LogCommand --
#
#	Wrap system calls to log commands and output.  If a second
#	argument is present, returns collected output from the command,
#	otherwise, returns 1. An optional third argument allows for an
#       expected non-zero exit status. An optional fourth argument prevents
#       redirection of stderr, which can bother windows.
#
# Results:
#       Output string if requested, true on success otherwise.
#       Returns undef on failure.
#
# Side effects:
#       Runs the command.
#
###########################################################################

sub LogCommand
{
   my $cmd = shift;                # IN: Command to run.
   my $returnOutput = shift;       # IN: If true, return the output.
   my $expectedResult = shift;     # IN: Expected return value.
   my $dontRedirectStdErr = shift; # IN: Ignore stderr, windows can bug out.
   my $arrayOutput = shift;        # IN: Return an array ref, not a big string.

   LogInfo("\"$cmd\"");

   if (!defined($expectedResult)) {
      $expectedResult = 0;
   }
   
   my $openArgs = " |";
   unless ($dontRedirectStdErr) {
      $openArgs = "2>&1 |";
   }

   local *CMD;
   unless (open(CMD, "$cmd $openArgs")) {
      LogInfo("Error: $!");
      return undef;
   }
   
   my $output = $arrayOutput ? [] : "";

   my $outputLine;
   while($outputLine = <CMD>) {
      LogInfo($outputLine);

      if ($arrayOutput) {
         chomp($outputLine);
         push(@$output, $outputLine);
      } else {
         $output .= $outputLine;
      }
   }

   #
   # Clear errno because close can fail but not set errno, which
   # should indicate that we should check $?.  But errno is never
   # guaranteed to be cleared if it is not set, so we may accidentally
   # see the errno of the last failed system call if we don't clear it.
   #

   $! = 0;
   unless (close(CMD)) {
      if ($!) {
         LogInfo("Could not close pipe to $cmd: $!\n");
         return undef;
      } elsif ($? >> 8 != $expectedResult) {
	 my $msg = "Command returned with non-zero status: exit status " .
                   ($? >> 8);
	 $msg .= ", signal ".($? & 127) if ($? & 127);
	 $msg .= "(dumped core)" if ($? & 128);
         
         LogInfo("$msg\n");
         return undef;
      }
   }

   return $returnOutput ? $output : 1;
}


###########################################################################
#
# LogCommandGetOutput --
#
#       Equivalent to LogCommand($cmd, 1, 0, $dontRedirectStdErr);
#
# Results:
#       Output of the command, or undef on non-command failure.
#
# Side effects:
#       Runs the command.
#
###########################################################################

sub LogCommandGetOutput
{
   my $cmd = shift;                 # IN: The command to be executed.
   my $dontRedirectStdErr = shift;  # IN: Optional. See param for LogCommand.

   return LogCommand($cmd, 1, 0, $dontRedirectStdErr);
}


###########################################################################
#
# LogCommandGetOutputArray --
#
#       Equivalent to LogCommand($cmd, 1, 0, $dontRedirectStdErr, 1);
#
# Results:
#       Output of the command as an array containing one line of
#       output per entry (without the end-of-line character(s)).
###########################################################################

sub LogCommandGetOutputArray
{
   my $cmd = shift;                 # IN: The command to be executed.
   my $dontRedirectStdErr = shift;  # IN: Optional. See param for LogCommand.

   return LogCommand($cmd, 1, 0, $dontRedirectStdErr, 1);
}

###########################################################################
#
# LogCommandErrWithFiles --
#
#       Similar to LogCommand, but instead separately redirects stdout
#       and stderr to temporary files.  By default, it then reads these
#       files in and both logs their contents and returns them in the
#       output variables, deleting the temp files.
#
#       Alternatively, it can simply return the file names in which case
#       the caller is responsible for deleting the files and doing any
#       desired logging of their contents.
#
#       Note that it will not be possible to tell when a given character of
#       stdout text was written relative to a given character of stderr
#       text if this method is used.
#
#       NOTE:  Yes, this is a rather clumsy way to do things, and one could
#              do a better job of mixing stdout and stderr back together
#              using IPC::Open3().  One can also do a great job of hanging
#              the process up while trying to figure out which channel
#              is going to produce output next.  This can be added (or
#              found online) later if it is needed.  This simple file
#              approach works well for many current cases.
#
# Results:
#       True on success, undef if the system calls failed or if the
#       command returned an unexpected exit status or was terminated
#       by a signal.
#
#       The $outRef and $errRef variables will hold either the contents
#       of stdout and stderr or file names for those contents depending
#       on the value of $returnFileNames.
#
# Side effects:
#       Runs the command.
#
#       Temp files are created and possibly deleted.  The files are
#       created using the VMware::FileSys::Tmp module and can be
#       removed either manually or with its global cleanup functions.
#       If $returnFileNames is set to true, the caller is responsible
#       for cleanup, including in cases of error.
#
#       If content is being returned (the default), then all of the
#       stdout and stderr is sucked into memory.  Twice.  Don't do this
#       with huge files.
#
###########################################################################

sub LogCommandErrWithFiles
{
   my $cmd = shift;               # IN: The command to execute.
   my $outRef = shift;            # OUT: The stdout of the cmd, as one string,
                                  #      *or* the name of the file holding it.
   my $errRef = shift;            # OUT: The stderr of the cmd, as one string.
                                  #      *or* the name of the file holding it.
   my $expectedResult = shift;    # IN: Expected return code.
   my $returnFileNames = shift;   # IN: If true, return the names of the files
                                  #     instead of the output directly.
   my $useErrForErr = shift;      # IN: If true, log cmd's stderr to the
                                  #     ERROR channel instead of INFO.

   if (!defined($expectedResult)) {
      $expectedResult = 0;
   }

   my $success = 0;

   LogInfo("\"$cmd\"");
   if ($returnFileNames) {
      LogDebug("stdout and stderr from the above command will not be logged.");
   }
   
   my $outfile = TmpFile(1, "vmware.out.");
   my $errfile = TmpFile(1, "vmware.err.");

   if ($returnFileNames) {
      LogDebug("stdout and stderr from the above command will not be logged.");
      LogInfo("stdout from the above command will be placed in $outfile");
      LogInfo("stderr from the above command will be placed in $errfile");
   }
   
   #
   # Set output params even if there is an error and they are undef.
   # Error checking is below.
   #

   if ($returnFileNames) {
      $$outRef = $outfile;
      $$errRef = $errfile;
   }

   if (!defined($outfile) || !defined($errfile)) {
      LogError("Could not create files for command I/O redirection.");
      goto CLEANUP;
   }

   #
   # Clear errno because system can fail but not set errno, which
   # should indicate that we should check $?.  But errno is never
   # guaranteed to be cleared if it is not set, so we may accidentally
   # see the errno of the last failed system call if we don't clear it.
   #

   $! = 0;
   unless (system("$cmd 1>$outfile 2>$errfile")) {
      my $msg;
      if ($!) {
         LogInfo("Could not execute command '$cmd' (errno from syscall): $!\n");
      } elsif ($? >> 8 != $expectedResult) {
         $msg = "Command returned with unexpected status: "
      } elsif ($? & 127) {
         $msg = "Command exit status OK but exited due to signal: "
      }
      if ($msg) {
         $msg .= "exit status " . ($? >> 8);
         $msg .= ", signal " . ($? & 127) if ($? & 127);
         $msg .= "(dumped core)" if ($? & 128);
         
         LogInfo("$msg\n");
      }
      goto CLEANUP;
   }

   if ($returnFileNames) {
      $success = 1;

   } else {
      #
      # Copying this string is not super-efficient.  Maybe should make
      # a FileSlurp that will take a reference and avoid this.  But this
      # is a bad function to use if you're expecting huge output/error
      # files anyway.
      #

      my $ref = FileSlurp($outfile, 1);
      $$outRef = $$ref;
      $ref = FileSlurp($errfile, 1);
      $$errRef = $$ref;

      if (!defined($$outRef) || !defined($$errRef)) {
         LogError("Could not read output and error files for above command.");
         goto CLEANUP;
      }

      LogInfo("STDOUT for the above command follows:");
      LogInfo($$outRef);

      my $channel = $useErrForErr ? VMware::Log::ID_ERROR :
                                    VMware::Log::ID_INFO;
      Log($channel, "STDERR for the above command follows:");
      Log($channel, $$errRef);
      $success = 1;
   }


CLEANUP:
   unless ($returnFileNames) {
      foreach my $file ($outfile, $errfile) {
         RemoveFileFromTempCleanup($file);
         unless (unlink($file)) {
            $success = 0;
            LogError("Could not remove temp file $file: $!\n");
         }
      }
   }
   return 1 if $success;
   return undef;
}

1;
