##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# CustomerLog.pm --
#
#       This class provides a channel ID for the special logging
#       statements the upgrade code uses for messages that the
#       customer must see, along with a method to initialize
#       that custom channel.
#

package VMware::Upgrade::CustomerLog;

require Exporter;

use constant ID_CUST => "CUSTOMER WARNING";
use VMware::Log qw(:log :manip);
use VMware::FileSys::StandardLocator qw(StdFile);

use strict;

#
# Don't export Init.  Should call as VMware::Upgrade::CustomerLog::Init();
#

@VMware::Upgrade::CustomerLog::ISA = qw(Exporter);
@VMware::Upgrade::CustomerLog::EXPORT_OK = qw(ID_CUST);


########################################################################
#
# CustomerLog::Init --
#
#       Set up the special log.  Duplicate the standard WARN channel
#       and *add* the special destination.  Leave it logging to
#       the standard location as well (that's why we duplicate rather
#       than create a new channel).
#
#       Once this is called the new log channel can be used like this:
#
#       Log(ID_CUST, "your message here");
# 
#       As always with perl's shaky constants implementation, you
#       may need to write ID_CUST() if the parser gets confused.
#
# Results:
#       None.
#
# Side effects:
#       Creates log.
#
########################################################################

sub Init
{
   my $log = LogPeek();
   $log->DuplicateChannel(VMware::Log::ID_WARN, ID_CUST(), 0);
   my $custFile = StdFile('customer_upgrade_warnings', 'esx');

   #
   # We can't tell if we'll be able to write to this log without
   # writing to it (adding a destination does not create the file).
   # We don't want the file created unless we *need* to write
   # something to it, because it will needlessly scare the customer.
   # So there's not much to be done in terms of error handling here.
   #

   $log->AddDestination($custFile, $custFile, ID_CUST());
}


1;

