##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# DevNames.pm --
#
#       This class reads the ESX 2.x format /etc/vmware/devnames.conf
#       file.
#

package VMware::Upgrade::DevNames;

use VMware::FileManip qw(FileSlurpArray);
use VMware::Log qw(:log);
use VMware::Upgrade::Locator qw(PreUpgradeFile);

use warnings;
use strict;


########################################################################
#
# DevNames::new --
#
#       Read an /etc/vmware/devnames.conf file and do basic processing
#       on it to produce an internal representation.
# 
# Results:
#       A new instance or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift; # IN: Invoking class name.

   my $self = {
      entries => [],
   };
   bless $self => $class;

   my $path = PreUpgradeFile('devnames', 'esx');
   my $lines = FileSlurpArray($path);
   unless (defined $lines) {
      LogError("Could not read file '$path'");
      return undef;
   }

   foreach my $line (@$lines) {
      # Skip comments
      if ($line =~ /^\s*#/) {
         next;
      }

      unless ($line =~ /^\s*
                        (\d{3}:\d{2}\.\d)    # BSF number
                        \s+
                        (\S+)                # Equivalence class
                        \s+
                        (\S+)                # Name
                        \s*$/x) {
         LogError("Unexpected devnames.conf entry '$line'");
         next;
      }

      my ($pciStr, $equiv, $name) = ($1, $2, $3);
      my $entry = [ $pciStr, $equiv, $name ];
      push @{$self->{entries}}, $entry;
   }

   return $self;
}


########################################################################
#
# DevNames::GetEntries --
#
#       Return an anonymous list of device name entries. Each entry
#       is an anonymous tuple consisting of the PCI string, equivalence
#       string, and device name.
#
# Results:
#       The list described above.
#
# Side effects:
#       None.
#
########################################################################

sub GetEntries 
{
   my $self = shift; # IN: Invoking instance.

   return $self->{entries};
}

1;
