##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# HWConfig.pm --
#
#       This class reads the ESX 2.x format /etc/vmware/hwconfig
#       file and parses it into an internal representation suitable
#       for translating to the ESX 3+ /etc/vmware/esx.conf format.
#

package VMware::Upgrade::HWConfig;

use VMware::Log qw(:log);
use VMware::Config::ConfigObj;
use VMware::Config::VmacoreConfigObj;

use VMware::PCI::Device;
use VMware::PCI::DeviceManager;
use VMware::PCI::LinuxDeviceManager;
use VMware::PCI::DriverManager;
use VMware::PCI::PCIString qw(MakePCIString);
use VMware::PCI::PCIInfo qw(SetPCIDeviceDataInConfig);

use VMware::Net::VSwitch;
use VMware::Net::VMKNic;
use VMware::Net::Interface;
use VMware::Net::Uplink;
use VMware::Net::PortGroup;
use VMware::Net::TeamPolicy;
use VMware::Net::NetworkManager;

use VMware::Upgrade::DevNames;
use VMware::Upgrade::Netmap;
use VMware::Upgrade::VMKModule;
use VMware::Upgrade::LegacyEthMap;
use VMware::Upgrade::Locator qw(PreUpgradeFile);
use VMware::Upgrade::RCLocal;

use strict;


########################################################################
#
# HWConfig::new --
#
#       Read an /etc/vmware/hwconfig file and do basic processing
#       on it to produce an internal representation.
# 
# Results:
#       A new instance or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;  # IN: Invoking class name.

   my $path = PreUpgradeFile('hwconfig', 'esx');
   my $hwconfig = new VMware::Config::ConfigObj();
   return undef unless $hwconfig->LoadFromFile($path);

   #
   # Now break the config file up into logically related areas.
   #

   my $devCfg = $hwconfig->Filter(qr/^device\.\d+\.\d+\.\d+\.\D/, 1);
   my $devOwnerCfg = $hwconfig->Filter(qr/^device\..*?\.\d+\.\d+\.\d+/, 1);
   my $nicTeamCfg = $hwconfig->Filter(qr/^nicteam/, 1);
   my $tcpipCfg = $hwconfig->Filter(qr/^tcpip/, 1);

   #
   # The following sections are not handled here but we filter them
   # out to check for unexpected entries.  Multipathing is handled
   # during a later stage of upgrade, swap is no longer relevant
   # due to per-VM swap files, and hyperthreading is now in hostd.
   # Boot is handled elsewhere due to the LiLo/GRUB change.
   # Device names are loaded from devnames.conf.
   #

   my $boot = $hwconfig->Filter(qr/^boot/, 1);
   my $multipathCfg = $hwconfig->Filter(qr/^multipath/, 1);
   my $swapCfg = $hwconfig->Filter(qr/^swapfile/, 1),
   my $hyperCfg = $hwconfig->Filter(qr/^hyperthreading/, 1),
   my $devNamesCfg = $hwconfig->Filter(qr/^devicenames/, 1);

   my $remainder = $hwconfig->ToHash();
   if (keys %$remainder) {
      LogWarn("Unexpected hwconfig entries:");
      foreach my $key (keys %$remainder) {
         LogWarn("   $key = \"$remainder->{$key}\"");
      }
   }

   my $self = {};
   bless $self => $class;

   $self->BuildDeviceManager($devCfg);
   $self->ParseDevOwnership($devOwnerCfg);
   $self->CheckManualVMKModule();
   $self->ParseDevVMKNames();
   $self->PlumbNetwork($nicTeamCfg, $tcpipCfg);
   $self->CheckLegacyVmnics();
   $self->CreateVswifs();
   
   if ($hyperCfg->{hyperthreading}) {
      if ($hyperCfg->{hyperthreading} eq "false") {
         $self->{hyperthreading} = "false";
      }
      else {
         $self->{hyperthreading} = "true";
      }
   }
   
   return $self;
}


########################################################################
#
# HWConfig::GetConfigData --
#
#       Return the root-level tree structure for the esx.conf file
#       produced by hwconfig and its supporting config files.
#
# Results:
#       A reference suitable for passing to VmacoreConfigObj::SetTree().
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigData
{
   my $self = shift;  # IN: Invoking instance.

   my $config = new VMware::Config::VmacoreConfigObj();
   SetPCIDeviceDataInConfig($config, $self->{devMgr});
   my $driverMgr = new VMware::PCI::DriverManager::($self->{devMgr},
      $config->GetTree(['vmkernel', 'module']));
   $config->SetTree(['vmkernel', 'module'], $driverMgr->GetConfigData());
   $config->SetTree(['net'], $self->{netMgr}->GetConfigData());

   return $config->GetTree([]);
}


########################################################################
#
# HWConfig::CommitCosNetworking --
#
#       Passthrough to our NetworkManager's CommitInterfaces and
#       CommitNetworkFile methods.
#
# Results:
#       1 for success, 0 otherwise.
#
# Side effects:
#       See VMware::NetworkManager::CommitInterfaces.
#
########################################################################

sub CommitCosNetworking
{
   my $self = shift;  # IN: Invoking instance.

   my $rc = $self->{netMgr}->CommitInterfaces();
   $rc &&= $self->{netMgr}->CommitNetworkFile();
   return $rc;
}


########################################################################
#
# HWConfig::BuildDeviceManager --
#
#       Read the basic PCI device configuration from the hwconfig file
#       and produce a corresponding VMware::PCI::DeviceManager object.
#       Ignore auxilliary information such as ownership and VMkernel
#       device names, which have a different key format in hwconfig.
#
# Results:
#       None.
#
# Side effects:
#       $self->{devMgr} is constructed.
#
########################################################################

sub BuildDeviceManager
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $devCfg = shift; # IN: Basic device configuration.

   my $devMgr = new VMware::PCI::DeviceManager();

   foreach my $devKey (keys %$devCfg) {
      my ($bus,
          $slot,
          $function,
          $field) = $devKey =~ /device\.(\d+)\.(\d+)\.(\d+)\.(\S+)/;

      my $value = $devCfg->{$devKey};
      my $pciStr = MakePCIString($bus, $slot, $function);

      my $dev = $devMgr->GetDevice($pciStr);
      unless ($dev) {
         $dev = new VMware::PCI::Device($pciStr);
         $devMgr->AddDevice($dev);
      }
      if ($field eq "class") {
         $dev->SetClass(hex $value);
      } elsif ($field eq "devID") {
         $dev->SetDeviceID(hex $value);
      } elsif ($field eq "vendor") {
         $dev->SetVendorID(hex $value);
      } elsif ($field eq "name") {
         # XXX: did we get rid of this?
         $dev->SetName($value);
      } elsif ($field eq "subsys_devID") {
         $dev->SetSubsysDeviceID(hex $value);
      } elsif ($field eq "subsys_vendor") {
         $dev->SetSubsysVendorID(hex $value);
      } elsif ($field eq "irq") {
         # XXX: weren't we going to get rid of this?  Still in HostInterface.pm
         $dev->SetIrq($value);
      }
   }

   unless ($devMgr->LoadDriverInfo()) {
      LogWarn("Could not load driver information for devices.  " .
              "Some upgrade steps will likely fail.");
   }

   $self->{devMgr} = $devMgr;
}


########################################################################
#
# HWConfig::CheckManualVMKModule --
#
#       Get any manual overrides for vmkernel modules and merge them
#       in with the device settings we have from other files.
#
#       Reassign any devices that we earlier detected should be
#       moved to the VMkernel as we no longer need to preserve
#       the pre-upgrade ordering.
#
# Results:
#       None.
#
# Side effects:
#       Device reassignment, possible option setting.
#
########################################################################

sub CheckManualVMKModule
{
   my $self = shift;  # IN/OUT: Invoking instance.

   my $vmkmod = new VMware::Upgrade::VMKModule::();
   
   #
   # If we had trouble reading an existing vmkmodule file,
   # just warn and move on.
   #

   my $modules = undef;
   if (defined $vmkmod) {
      $modules = $vmkmod->GetModules();
   } else {
      LogWarn("/etc/vmkmodule.conf is present but could not be read.");
      LogWarn("This is probably OK as vmkmodule.conf is only used by " .
              "advanced users, and even then only rarely.");
   }

   if (defined $modules) {
      #
      # This works because GetDevices() returns devices in bus:slot.function
      # order, which is the same order options are passed for a given driver.
      # So sort the devices by driver and store the options to compare with
      # those in the vmkmodule.conf.
      #

      my $mgrDriverInfo = {};
      foreach my $dev (@{$self->{devMgr}->GetDevices()}) {
         if ($dev->GetOwner() eq VMware::PCI::Device::VMKERNEL()) {
            my $driver = $dev->GetDrivers()->[0];
            unless (defined $mgrDriverInfo->{$driver}) {
               $mgrDriverInfo->{$driver} = [];
            }
            push(@{$mgrDriverInfo->{$driver}}, $dev);
         }
      }

      my $confDriverInfo = {};
      foreach my $modInfo (@$modules) {
         $confDriverInfo->{$modInfo->{driver}} = $modInfo->{options};
      }

      #
      # We only loop over the drivers from the device manager because
      # those are the ones that will still be loaded post-upgrade.  If
      # vmkmodule.conf has a driver that is no longer in use, its options
      # likewise do not matter.
      #

      foreach my $driver (keys %$mgrDriverInfo) {
         if (defined $confDriverInfo->{$driver}) {
            my @options = keys %{$confDriverInfo->{$driver}};
            for (my $i = 0; $i < @{$mgrDriverInfo->{$driver}}; ++$i) {
               my $optstring = "";
               foreach my $option (@options) {
                  $optstring .= " " unless $optstring eq "";

                  my $value = $confDriverInfo->{$driver}->{$option}->[$i];
                  $value = "" unless defined $value;
                  $optstring .= "$option=$value";
                     
                  $mgrDriverInfo->{$driver}->[$i]->SetOptions($optstring);
               }
            }
         } else {
            foreach my $dev (@{$mgrDriverInfo->{$driver}}) {
               $dev->SetOptions("");
            }
         }
      }
   }

   #
   # Now fix the ownership that we'd kept in the old state
   # so we could track options.  This puts everything into
   # the proper post-upgrade state.
   #

   foreach my $dev (@{$self->{assignToVMkernel}}) {
      $dev->SetOwner(VMware::PCI::Device::VMKERNEL);
   }

   return 1;
}


########################################################################
#
# HWConfig::ParseDevOwnership --
#
#       Look through the device ownership information and find the
#       right boot configuration to use.  This should be the one
#       that was currently in use at the time of the upgrade.
#       Record which VMkernel-manageable devices were previously
#       assigned to the COS or shared, and then assign all VMkernel-
#       manageable devices to the VMkernel.  Do not name them yet.
#
#       Also bring over the driver options as they are set in the
#       same area that ownership is set.
#
# Results:
#       None.
#
# Side effects:
#       Add ownership to $self->{devMgr} and populate
#       $self->{formerlyCOS} and $self->{formerlyShared}.
#       
########################################################################

sub ParseDevOwnership
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $devOwnerCfg = shift;  # IN: Device ownership configuration.

   my $bootCfgs = {};
   foreach my $ownerKey (keys %$devOwnerCfg) {
      my ($cfgName,
          $bus,
          $slot,
          $function,
          $field) = $ownerKey =~ /device\.(.*?)\.(\d+)\.(\d+)\.(\d+)\.(\w+)/;
      my $value = $devOwnerCfg->{$ownerKey};

      unless (exists $bootCfgs->{$cfgName}) {
         $bootCfgs->{$cfgName} = {};
      }
      my $pciStr = MakePCIString($bus, $slot, $function);
      $bootCfgs->{$cfgName}->{$pciStr}->{$field} = $value;
   }

   my $currentCfg = $self->FindCurrentBootConfig();

   $self->{formerlyCOS} = [];
   $self->{formerlyShared} = [];
   $self->{formerlyVMkernel} = [];
   my $found;
   $self->{assignToVMkernel} = [];
   foreach my $cfgName (keys %$bootCfgs) {
      if ($cfgName eq $currentCfg) {
         $found = 1;
         foreach my $pciStr (keys %{$bootCfgs->{$cfgName}}) {

            my $dev = $self->{devMgr}->GetDevice($pciStr);
            if (not defined $dev) {
               #
               # Move on silently.  This is likely just an entry for
               # a device that was moved or removed.
               #
               next;
            }

            #
            # The usage of the "assignToVMkernel" array is as follows:
            # A later step (vmkmodule.conf) requires that devices
            # previously assigned to the COS still be assigned to the
            # COS even if they should now go to the VMkernel.  This is
            # to track the pre-upgrade ordering of VMkernel devices.
            # Therefore, such devices are given to the COS, but marked
            # as needing to be assigned to the VMkernel later.
            #
            # Devices also already being assigned to the VMkernel
            # may also be so marked, which is harmless.
            #

            if ($dev->IsVMwareManageable()) {
               if ($bootCfgs->{$cfgName}->{$pciStr}->{owner} eq "host") {
                  push @{$self->{formerlyCOS}}, $dev;
                  $dev->SetOwner(VMware::PCI::Device::COS);
               } elsif ($bootCfgs->{$cfgName}->{$pciStr}->{owner} eq "shared") {
                  push @{$self->{formerlyShared}}, $dev;
                  $dev->SetOwner(VMware::PCI::Device::VMKERNEL);
               } else {
                  push @{$self->{formerlyVMkernel}}, $dev;
                  $dev->SetOwner(VMware::PCI::Device::VMKERNEL);
               }
               push @{$self->{assignToVMkernel}}, $dev;
            } else {
               $dev->SetOwner(VMware::PCI::Device::COS);
            }

            if (defined $bootCfgs->{$cfgName}->{$pciStr}->{options}) {
               $dev->SetOptions($bootCfgs->{$cfgName}->{$pciStr}->{options});
            }
         }
      } else {
         LogInfo("Dropping inactive boot configuration '$cfgName'.");
      }
   }
   if (not $found) {
      LogWarn("No boot configurations found, device ownership will be " .
              "determined from scratch.");
   }
   
   #
   # Load the driver information with this device assignment so we
   # can match the options in /etc/vmware/vmkmodule.conf (if it is used).
   #

   $self->{devMgr}->LoadDriverInfo();
}


########################################################################
#
# HWConfig::FindCurrentBootConfig --
#
#       Find (or guess, if necessary) the boot configuration that
#       the system is currently using.  We will use this for upgrade
#       and drop the rest as multiple boot configs are no longer
#       supported.
#
#       Find the config as follows:
#         1. If there is a lilo default entry with a vmnix kernel, use it.
#         2. Take the vmnix kernel entry named "esx" (default from 2.5.x).
#         3. Take the vmnix kernel entry named "vmnix" (default from 2.1.x).
#         4. Take the first vmnix kernel entry in the file.
#         5. Guess "esx" and hope it works.
#
# Results:
#       The name of the current boot config.
#
# Side effects:
#       Reads /etc/lilo.conf
# 
########################################################################

sub FindCurrentBootConfig
{
   my $self = shift;  # IN: Invoking instance.

   my $lilo = new VMware::Upgrade::LiloConfig::();

   unless ($lilo->ReadLiloConfig()) {
      LogWarn("Could not read lilo.conf to determine current boot label.");
      LogWarn("Guessing 'esx' as the current boot label.");
      return "esx";
   }

   my $label = $lilo->GetDefaultBootLabel();
   my $esxLabels = $lilo->GetESXBootLabels();
   if (defined($label) && grep { $_ eq $label; } @$esxLabels) {
      LogInfo("Choosing the default label '$label'.");
      return $label;

   } elsif (grep { $_ eq "esx"; } @$esxLabels) {
      LogInfo("Choosing the 'esx' label as there is no explicit " .
              "vmnix default.");
      return "esx";

   } elsif (grep { $_ eq "vmnix"; } @$esxLabels) {
      LogInfo("Choosing the 'vmnix' label as there is no explicit " .
              "default and no 'esx' label.");
      return "vmnix";

   } elsif (@$esxLabels) {
      LogInfo("Choosing the first vmnix boot label ('$esxLabels->[0]') " .
              "as there is no explicit default 'esx' or 'vmnix' label.");
      return $esxLabels->[0];

   } else {
      LogWarn("Could not find any ESX boot labels.  Guessing 'esx'.");
      return "esx";
   }
}


########################################################################
#
# HWConfig::ParseDevVMKNames --
#
#       Go through the device naming information and pick the most
#       relevant name for each device and set it through the
#       DeviceManager.
#
#       We only keep one device name even if multiple names were being
#       held for different kinds of devices at the same address.
#
# Results:
#       None.
#
# Side effects:
#       Names devices.
#       May increment logged warnings count.
#
########################################################################

sub ParseDevVMKNames
{
   my $self = shift;  # IN/OUT: The invoking instance.

   my $devNames = new VMware::Upgrade::DevNames();
   unless ($devNames) {
      LogError("Failed to load device naming information.");
      LogError("This may result in device renaming and/or");
      LogError("broken networking or storage settings");
      return;
   }

   foreach my $entry (@{$devNames->GetEntries()}) {
      my ($pciStr, $equiv, $name) = @$entry;

      my $dev = $self->{devMgr}->GetDevice($pciStr);
      if (not defined $dev) {
         #
         # Silently drop names with no device as they are almost certainly
         # remnants of devices since removed (or just moved).
         #
         next;
      }

      #
      # Past releases (2.1.x and earlier) treated each scsi
      # device as its own equivalence class.  In 2.5.x and later,
      # we treat them all as the scsi equivalence class.
      #
         
      my $devEquiv = $dev->GetEquiv();
      if (($devEquiv eq $equiv) ||
          (($devEquiv eq VMware::PCI::Device::EQUIV_SCSI()) &&
           ($equiv ne VMware::PCI::Device::EQUIV_FC()) &&
           ($equiv ne VMware::PCI::Device::EQUIV_NIC()))) {
         $self->{devMgr}->SetVMKDeviceName($pciStr, $name);
      } else {
         # Silently drop alternate names.
      }
   }

   $self->{devMgr}->GenerateVMKDeviceNames();
}


########################################################################
#
# HWConfig::ParseNicTeams --
#
#       Parse the nic teaming information into a hash keyed by bond
#       names, where each entry is a hash with two keys:
#          nics:  A list of network devices in the team (vmnicN form).
#          loadBalance: The load balancing policy, possibly a vmnic name
#                       if a preferred nic was specified.
#
# Results:
#       Returns the nic teaming structure defined above.
#
# Side effects:
#       None.
#       May increment logged warnings count.
#
########################################################################

sub ParseNicTeams
{
   my $self = shift;  # IN: Invoking instance.
   my $nicTeamCfg = shift;  # IN: hwconfig info about nic teams.

   my $teams = {};
   foreach my $teamKey (keys %$nicTeamCfg) {
      if ($teamKey =~ /nicteam\.(vmnic\d+)\.team/) {
         my ($vmnic) = $1;
         my $bond = $nicTeamCfg->{$teamKey};

         unless (exists $teams->{$bond}) {
            #
            # It simplifies code to ensure that these are always defined.
            #

            $teams->{$bond}->{nics} = [];
            $teams->{$bond}->{loadBalance} = "";
         }
         push(@{$teams->{$bond}->{nics}}, $vmnic);

      } elsif ($teamKey =~ /nicteam.(bond\d+).load_balance_mode/) {
         #
         # If we've already found a home_link entry and used it
         # to set the loadBalance policy, then ignore this entry.
         # home_link always overrides load_balance_mode.  Note
         # than an unset loadBalance holds a defined empty string.
         #

         my $bond = $1;
         unless ($teams->{$bond}->{loadBalance}) {
            $teams->{$bond}->{loadBalance} = $nicTeamCfg->{$teamKey};
         }

      } elsif ($teamKey =~ /nicteam.(bond\d+).home_link/) {
         my $bond = $1;
         $teams->{$bond}->{loadBalance} = $nicTeamCfg->{$teamKey};

      } elsif ($teamKey =~ /nicteam.bond\d+.switch_failover_threshold/) {
         #
         # No direct translation, so we explicitly drop this setting.
         #
      } else {
         LogWarn("Unexpected nic teaming option '$teamKey' = " .
                 "'$nicTeamCfg->{$teamKey}'.");
      }
   }
   return $teams;
}


########################################################################
#
# HWConfig::PlumbNetwork --
#
#       Parse out the configured NIC teams and any associated
#       information.  We can use this to determine which actual NICs
#       back networks in netmap.conf that refer to bondN devices
#       (which are the names given to the nic teams here).
#
#       Then pull in netmap.conf and build up the new esx.conf-style
#       network configuration using this and the nic team info.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Creates $self->{netMgr} and adds stuff to it.
#       May increment logged warnings count.
#
########################################################################

sub PlumbNetwork
{
   my $self = shift;  # IN/OUT: The invoking instance.
   my $nicTeamCfg = shift; # IN: Nic teaming configuration.
   my $tcpipCfg = shift;   # IN: VMkernel network configuration.

   my $teams = $self->ParseNicTeams($nicTeamCfg);

   my $netmap = new VMware::Upgrade::Netmap();
   unless ($netmap) {
      LogError("Could not build network map.");
      return 0;
   }

   $self->{netMgr} = new VMware::Net::NetworkManager();

   my $switches = {};
   my $networks = $netmap->GetNetworks();
   foreach my $network (@$networks) {
      my $vswitch;
      if (defined $switches->{$network->{device}}) {
         $vswitch = $switches->{$network->{device}};
      } else {
         $vswitch =
            new VMware::Net::VSwitch( {
               name => $self->{netMgr}->GetNextVSwitchName()
            } );

         $switches->{$network->{device}} = $vswitch;
      }

      #
      # wal says a name conflict between (old) switches and vlans
      # is possible when setting names here.  
      # XXX: Figure this out for sure and deal with it.
      #

      my $pg = new VMware::Net::PortGroup();
      $pg->SetName($network->{name});
      $pg->SetVlanId($network->{vlanId});
      $vswitch->AddPortGroup($pg);

      #
      # Only deal with uplinks for the vswitch proper, not vlans.
      # XXX: This is not foolproof.  While the UI will not permit it,
      #      there could be additional VLANs with ID 0 that do not (properly)
      #      represent the vswitch.  However, in this case, that just means
      #      we'll assign the uplinks twice which should be harmless.
      # XXX: Is that really true with a VSwitch object instead of just a hash?
      #

      next unless $network->{vlanId} == 0;

      $self->ConvertDevToUplink($network->{device}, $teams, $vswitch);
   }

   $self->AddVMKNic($tcpipCfg, $teams, $networks, $switches);

   foreach my $vs (sort { $a->GetName() cmp $b->GetName(); }
                        (values %$switches)) {
      $self->{netMgr}->AddVSwitch($vs);
   }

   return 1;
}


########################################################################
#
# HWConfig::ConvertDevToUplink --
#
#       Convert a vmnic|vmnet|bond device and its configuration into the
#       appropriate vswitch uplink configuration.
#
# Results:
#       None.
#
# Side effects:
#       Vswitch parameter modified.
#       May increment logged warnings count.
#
########################################################################

sub ConvertDevToUplink
{
   my $self = shift;     # IN/OUT: Invoking instance;
   my $device = shift;   # IN: Device backing old network.
   my $teams = shift;    # IN: Nic teaming information.
   my $vswitch = shift;  # IN/OUT: The new vswitch representing the old team.

   #
   # Track used vmnic devices so we can later determine if there are
   # any vmnics that were not part of virtual switches that need
   # special handling.
   #

   unless (defined $self->{legacyVmnics}) {
      $self->{legacyVmnics} = [];
   }

   my $pg = new VMware::Net::PortGroup();
   $pg->SetName("Legacy $device");
   $pg->SetVlanId(0);
   $vswitch->AddPortGroup($pg);

   my $bond;
   if ($device =~ /vmnic/) {
      my $uplink = new VMware::Net::Uplink();
      $uplink->SetPnic($device);
      $vswitch->AddUplink($uplink);

      my $teamPolicy = new VMware::Net::TeamPolicy();
      $teamPolicy->AddUplink($uplink);
      $vswitch->SetTeamPolicy($teamPolicy);
      push(@{$self->{legacyVmnics}}, $device);
      return;

   } elsif ($device =~ /vmnet/) {

      # Do nothing, as there are no uplinks.
      return;

   } elsif ($device =~ /bond/) {
      $bond = $device;

   } else {
      LogWarn("Unknown device type '$device'.  This device will be dropped " .
              "so VirtualSwitch " . $vswitch->GetName() . " may not have " .
              "any uplinks.");
      return;
   }

   my $teamPolicy = new VMware::Net::TeamPolicy();

   unless (defined $teams->{$bond}) {
      LogWarn("VirtualSwitch " . $vswitch->GetName() . " references unknown " .
              "nic team '$device'.\nThis device will be dropped so the " .
              "may not have any uplinks.");
      return;
   }

   foreach my $vmnic (@{$teams->{$bond}->{nics}}) {
      #
      # If a preferred vmnic has been set in loadBalance, that should
      # be the first listed uplink to set the proper failover order.
      #

      my $vmnicUplink = new VMware::Net::Uplink();
      $vmnicUplink->SetPnic($vmnic);
      my $preferred = $vmnic eq $teams->{$bond}->{loadBalance};

      $vswitch->AddUplink($vmnicUplink, $preferred);
      $teamPolicy->AddUplink($vmnicUplink, $preferred);

      #
      # Mark the vmnic as used for the purpose of tracking
      # legacy vmnics because vmnics that were part of bonds
      # could not be used in isolation.
      #

      push(@{$self->{legacyVmnics}}, $vmnic);
   }

   if ($teams->{$bond}->{loadBalance}) {
      if ($teams->{$bond}->{loadBalance} =~ /vmnic/) {
         $teamPolicy->SetTeam(VMware::Net::TeamPolicy::TEAM_FO_EXPLICIT);

      } elsif ($teams->{$bond}->{loadBalance} eq "out-mac") {
         $teamPolicy->SetTeam(VMware::Net::TeamPolicy::TEAM_LB_SRCID);

      } elsif ($teams->{$bond}->{loadBalance} eq "out-ip") {
         $teamPolicy->SetTeam(VMware::Net::TeamPolicy::TEAM_LB_IP);

      } else {
         LogWarn("Network team '$bond' used unknown load balancing " .
                 "method '$teams->{$bond}->{loadBalance}'.");
         #
         # Go ahead and keep this vswitch and risk surprising
         # the customer with the load-balancing policy.  This
         # is better than dropping the connection entirely.
         #
      }
   }
   $vswitch->SetTeamPolicy($teamPolicy);
}


########################################################################
#
# HWConfig::AddVMKNic --
#
#       Converts the tcpip config into esx.conf structures
#       on top of the vswitches constructed earlier.
#
# Results:
#       None.
#
# Side effects:
#       May increment logged warnings count.
#
########################################################################

sub AddVMKNic
{
   my $self = shift;      # IN/OUT: Invoking instance.
   my $tcpipCfg = shift;  # IN: VMkernel networking config info.
   my $teams = shift;     # IN: Nic team info from hwconfig.
   my $networks = shift;  # IN: List of info from netmap.conf
   my $switches = shift;  # IN/OUT: Hash of vswitch configurations
                          #         keyed on device.


   return unless defined $tcpipCfg->{"tcpip.nicName"};

   my $device;
   my $vlanId;
   if ($tcpipCfg->{"tcpip.nicName"} =~ /^(.*)\.(\d+)$/) {
      $device = $1;
      $vlanId = $2;
   } else {
      $device = $tcpipCfg->{"tcpip.nicName"};
      $vlanId = 0;
   }

   unless ($device =~ /(?:vmnic|vmnet|bond)/) {
      #
      # If this doesn't match exactly one element, we're in bizzaro-world
      # because we are connected to a network name rather than a device,
      # but the name is not present in our list of networks.  And two
      # networks may not have the exact same name.
      #
      my $oldDev = $device;
      $device =
         (map { $_->{device} } grep { $_->{name} eq $device } @$networks)[0];
      unless (defined $device) {
         LogWarn("VMkernel nic attached to non-existent network '$oldDev'.");
         LogWarn("VMkernel network configuration is being dropped.");
         return;
      }
   }

   my $vswitch;
   if (defined $switches->{$device}) {
      $vswitch = $switches->{$device};
   } else {
      $vswitch = new VMware::Net::VSwitch();
      $vswitch->SetName($self->{netMgr}->GetNextVSwitchName());
      $self->ConvertDevToUplink($device, $teams, $vswitch);
      $switches->{$device} = $vswitch;
   }

   my $pg = new VMware::Net::PortGroup();
   my $pgName = "Legacy VMotion";
   $pg->SetName($pgName);
   $pg->SetVlanId($vlanId);
   
   my $vmknic = new VMware::Net::VMKNic();
   $vmknic->SetPortGroup($pgName);
   $vmknic->SetEnable("true");
   $vmknic->SetIpV4Address($tcpipCfg->{"tcpip.ipAddr"});
   $vmknic->SetIpV4Netmask("255.255.255.0");
   $vswitch->AddPortGroup($pg);

   $self->{netMgr}->AddVMKNic($vmknic);

   $self->{netMgr}->SetVMKernelGateway($tcpipCfg->{"tcpip.gatewayAddr"});
}


########################################################################
#
# HWConfig::CheckLegacyVmnics --
#
#       Check the list of legacy vmnics found against the list of
#       named vmnic devices.  Create legacy portgroups for all vmnics
#       that have not already been found.
#
# Results:
#       None.
#
# Side effects:
#       More virtual switches, uplinks, porgtroups, etc. added.
#
########################################################################

sub CheckLegacyVmnics
{
   my $self = shift;  # IN/OUT: Invoking instance.

   my @vmnics = map {
      $_->GetVMKDeviceName();
   } grep {
      $_->IsNicEquiv();
   } @{$self->{formerlyVMkernel}};

   my @shared = map {
      $_->GetVMKDeviceName();
   } grep {
      $_->IsNicEquiv();
   } @{$self->{formerlyShared}};

   push(@vmnics, @shared);
   foreach my $vmnic (@vmnics) {
      unless (grep { $vmnic eq $_; } @{$self->{legacyVmnics}}) {
         my $pg = new VMware::Net::PortGroup();
         $pg->SetName("Legacy $vmnic");
         $pg->SetVlanId(0);

         my $uplink = new VMware::Net::Uplink();
         $uplink->SetPnic($vmnic);

         my $teamPolicy = new VMware::Net::TeamPolicy();
         $teamPolicy->AddUplink($uplink);

         my $vswitch = new VMware::Net::VSwitch();
         $vswitch->SetName($self->{netMgr}->GetNextVSwitchName());
         $vswitch->AddPortGroup($pg);
         $vswitch->AddUplink($uplink);
         $vswitch->SetTeamPolicy($teamPolicy);

         $self->{netMgr}->AddVSwitch($vswitch);
         push(@{$self->{legacyVmnics}}, $vmnic);
      }
   }
}


########################################################################
#
# HWConfig::CreateVswifs --
#
#       Examine the network devices previously assigned to the COS
#       and create a vswif and all dependant network objects for it.
#
#       If no devices were assigned to the COS and there is only
#       one vswitch, attach vswif0 to that vswitch as it was most
#       likely a shared with the COS through vmxnet_console.
#       If there are multiple vswitches, give up because we can't
#       know what the user intended and the vmxnet_console configuration
#       is not available to us.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       $self->{netMgr} gets VMware::Net::Interfaces added.
#       Old network interfaces are removed.
#       May increment logged warnings count.
#
########################################################################

sub CreateVswifs
{
   my $self = shift; # IN/OUT: Invoking instance.

   my $linuxMgr = new VMware::PCI::LinuxDeviceManager();
   unless (defined $linuxMgr) {
      LogWarn("Could not get linux device info, no COS networking will " .
              "be available (no vswifs).");
      return 0;
   }

   #
   # Load legacy interface information: Which interfaces
   # were used for COS networking and what their hwaddrs were.
   # We load the list of legacy eths from two seperate locations
   # because the legacy-eth.map file is not guaranteed to exist.
   # The information from NetworkManager is used as a fallback.
   #

   my @legacyInterfaces = grep {
      $_->GetOnBoot() =~ /^YES$/i && $_->GetName() =~ /^eth/
   } $self->{netMgr}->GetInterfaces();
   
   my $legacyEthMap = new VMware::Upgrade::LegacyEthMap();
   unless (defined $legacyEthMap) {
      LogWarn("Could not build legacy eth map.");
      return 0;
   }

   my @cosnics = grep { $_->IsNicEquiv(); } @{$self->{formerlyCOS}};
   foreach my $cosnic (@cosnics) {
      my $pciStr = $cosnic->GetPCIString();
      my $nicdev = $linuxMgr->GetDevice($pciStr);
      unless (defined($nicdev)) {
         LogWarn("No nic where expected at '$pciStr'.  At least one " .
                 "COS network device may be inactive after upgrade.");
         next;
      }

      my $hwaddr = $nicdev->GetNetworkHWAddr();
      my $eth = $legacyEthMap->GetEthByHWAddr($hwaddr);

      if ($eth) {
         LogInfo("Found legacy eth from map: $eth ($hwaddr)");
      } else {
         # We can't find a legacy eth. Try to see if the user
         # only had one COS eth, and if so use that instead.
         if (scalar(@legacyInterfaces) == 1) {
            $eth = $legacyInterfaces[0]->GetName();
            LogInfo("Could not find legacy ethN mapping for $pciStr," .
                    " but found $eth through process of elimination.");
         } else {
            # No idea. Use the eth from kudzu, but warn that this
            # is probably incorrect.
            $eth = $nicdev->GetDevice();
            LogWarn("Could not determine the ethN for device $pciStr." .
                    " This may cause problems. Guessing $eth instead.");
         }
      }

      my $vswitch = new VMware::Net::VSwitch();
      $vswitch->SetName($self->{netMgr}->GetNextVSwitchName());
      $self->{netMgr}->AddVSwitch($vswitch);

      my $uplink = new VMware::Net::Uplink();
      $uplink->SetPnic($cosnic->GetVMKDeviceName());
      $vswitch->AddUplink($uplink);

      my $teamPolicy = new VMware::Net::TeamPolicy();
      $teamPolicy->AddUplink($uplink);
      $vswitch->SetTeamPolicy($teamPolicy);

      my $pg = new VMware::Net::PortGroup();
      $pg->SetName("Legacy $eth");
      $vswitch->AddPortGroup($pg);

      $self->AddLegacyVswif($eth, $pg);
   }

   #
   # Add vmxnet_console devices.
   #
   my $vmxnetDevices = [];
   my $eths = {};
   my $ethCount = scalar(@cosnics);

   my $rclocal = new VMware::Upgrade::RCLocal();
   if (defined $rclocal) {
      $vmxnetDevices = $rclocal->GetVMXNetDevices();
      $eths = $rclocal->GetEths();
   }

   # As of ESX 2.1, vmxnet_console devices are brought up by
   # the sharenic.pl script, and not rc.local.  Need to get the
   # extra device names based on shared devices in hwconfig.
   foreach my $sharednic (@{$self->{formerlyShared}}) {
      if ($sharednic->IsNicEquiv()) {
         push(@$vmxnetDevices, $sharednic->GetVMKDeviceName());
      }
   }

   foreach my $device (@$vmxnetDevices) {
      my $eth = "eth" . $ethCount++;
      
      # Check for a manual interface. If the user has not specified one,
      # $if will be undef and AddLegacyEth will look for an ifcfg
      # file.
      my $if = $eths->{$eth};

      LogInfo("Processing vmxnet_console device $device ($eth)");
      if ($if) {
         LogInfo("Found manually-configured interface: $eth " . $if->GetIPAddr());
      }

      # Find the portgroup associated with this device.
      # XXX: What if the user specified a VLAN ID?
      my $pgname = "Legacy $device";
      my $pg = $self->{netMgr}->GetPortGroupByName($pgname);
      unless ($pg) {
         if ($pgname =~ /^vmnet_\d+$/) {
            # We discovered a vmnet_N not referenced in netmap.conf.
            # Go ahead and create a vswitch for it.
            LogInfo("Creating vswitch for discovered network $pgname");

            my $vswitch = new VMware::Net::VSwitch();
            $pg = new VMware::Net::PortGroup();
            $pg->SetName("Legacy $device");
            $pg->SetVlanId(0);
            $vswitch->AddPortGroup($pg);
            $self->{netMgr}->AddVSwitch($vswitch);
         } else {
            # It's possible this is a network name. Try to find
            # the network portgroup.
            $pg = $self->{netMgr}->GetPortGroupByName($device);
         }
      }

      if ($pg) {
         my $name = $pg->GetName();
         LogInfo("Found portgroup \"$name\" " .
                 "for vmxnet_console device $device");
      } else {
         LogWarn("Cannot find portgroup for vmxnet_console " .
                 "device $device; skipping.");
         next;
      }

      $self->AddLegacyVswif($eth, $pg, $if);
   }

   my $network = $self->{netMgr}->GetNetworkFile();
   unless (defined $network) {
      LogWarn("Couldn't load network settings file; you may need to set " .
              "a gateway device.");
      return 0;
   }
   my $gwDev = $network->GetGatewayDev();
   if (defined($gwDev) && $gwDev =~ /^eth\d+$/) {
      $gwDev =~ s/eth/vswif/;
      $network->SetGatewayDev($gwDev);
   } elsif (defined $self->{leastVswifNum}) {
      $network->SetGatewayDev("vswif" . $self->{leastVswifNum});
   }

   return 1;
}


########################################################################
#
# HWConfig::AddLegacyVswif --
#
#       Create a vswif for the specified legacy eth.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       $self->{netMgr} gets VMware::Net::Interface added.
#       Old network interface is removed.
#       May increment logged warnings count.
#
########################################################################

sub AddLegacyVswif {
   my $self = shift;    # IN/OUT: Invoking instance.
   my $eth = shift;     # IN: The legacy ethN as a string.
   my $pg = shift;      # IN: The portgroup to attach the vswif to
   my $if = shift;      # IN: (Optional) The interface for the ethN.

   unless ($if) {
      # Try to find an ifcfg file for this ethN.
      my $origIf = $self->{netMgr}->GetInterface($eth);
      unless (defined $origIf) {
         LogWarn("Could not get interface file for $eth, skipping.");
         return 0;
      }
      $if = $origIf->Copy();
   }

   # Remove any old interfaces for this ethN as we're replacing it
   # with a vswif.
   if ($self->{netMgr}->GetInterface($eth)) {
      unless ($self->{netMgr}->RemoveInterface($eth)) {
         LogWarn("Could not remove interface '$eth', this may cause problems.");
      }
   }

   my $vswif = $eth;
   $vswif =~ s/.*?(\d+)/vswif$1/;
   my $vswifNum = $1;

   if (defined $self->{leastVswifNum}) {
      $self->{leastVswifNum} = $vswifNum if $vswifNum < $self->{leastVswifNum};
   } else {
      $self->{leastVswifNum} = $vswifNum;
   }

   $if->SetName($vswif);
   $if->SetPortGroup($pg->GetName());

   unless ($self->{netMgr}->AddVswif($if)) {
      LogWarn("Could not add vswif '$vswif' corresponding to old '$eth'.");
      return 0;
   }

   return 1;
}



1;
