##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# LegacyEthMap.pm --
#
#       This class reads the /etc/vmware/legacy-eth.map file (as
#       generated by the pre-upgrade script) and parses it into
#       an internal representation suitable for use in 2.x->3.x
#       network upgrade.
#

package VMware::Upgrade::LegacyEthMap;

use VMware::Log qw(:log);
use VMware::Config::ConfigObj;
use VMware::Upgrade::Locator qw(PreUpgradeFile);

use strict;


########################################################################
#
# LegacyEthMap::new --
#
#       Opens and parses the legacy-eth.map file.
#
# Results:
#       The new instance or undef on failure.
#
# Side effects:
#       Reads the file.
#
########################################################################

sub new
{
   my $class = shift; # IN: Invoking class name.

   my $self = {
      eths => {},       # Hash mapping hwaddrs to eths
   }; 

   bless $self => $class;

   my $path = PreUpgradeFile('legacyethmap', 'esx');
   return $self unless -e $path;

   my $config = new VMware::Config::ConfigObj();
   unless ($config->LoadFromFile($path)) {
      return undef;
   }

   LogInfo("Loading legacy eth map");

   my $entries = $config->ToHash();
   foreach my $setting (keys %$entries) {
      unless ($setting =~ /^eth\.([^\.]+)\.hwaddr$/) {
         LogWarn("Unknown setting in legacy eth map: $setting");
         next;
      }

      my $eth = $1;

      #
      # Uppercase hwaddr so as to ensure one standard hex format.
      #
      my $hwaddr = uc($entries->{$setting});

      LogInfo("Legacy eth $eth has hwaddr $hwaddr");
      $self->{eths}->{$hwaddr} = $eth;
   }

   return $self;
}


########################################################################
#
# LegacyEthMap::GetEthByHWAddr --
#
#       Returns the legacy eth name which corresponds to the given hwaddr
#       string.
#
# Results:
#       The legacy eth as a string "ethN", or undef if none found.
#
# Side effects:
#       None.
#
########################################################################

sub GetEthByHWAddr
{
   my $self = shift; # IN: Invoking instance.
   my $hwaddr = shift; # IN: HWaddr string to search for.

   $hwaddr = uc($hwaddr);
   return $self->{eths}->{$hwaddr};
}

1;
