##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# LiloConfig.pm --
#
#       This class reads the ESX 2.x format /etc/lilo.conf
#       file and parses the boot information into an internal
#       representation suitable for translating to the ESX 3+
#       /boot/grub/grub.conf format.
#

package VMware::Upgrade::LiloConfig;

use VMware::Boot::GRUBConfig;
use VMware::Boot::GRUBDeviceMap;
use VMware::Log qw(:log);
use VMware::FileManip qw(FileSlurpArray);
use VMware::FileSys::MountInfo qw(ParseDeviceFilename);
use VMware::FileSys::StandardLocator qw(StdDir);
use VMware::Upgrade::Locator qw(PreUpgradeFile);
use strict;

########################################################################
#
# LiloConfig::new --
#
#       Read an /etc/lilo.conf file and do basic processing
#       on it to produce an internal representation.
# 
# Results:
#       A new instance or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift; # IN: Invoking class name.

   my $self = {
      cfgTree => {},
      rootDevices => [],
      grubConfig => undef,
      grubDeviceMap => undef,
   };
   bless $self => $class;

   $self->ReadLiloConfig() or return undef;
   $self->BuildGrubConfig();
   $self->BuildGrubDeviceMap();

   return $self;
}


########################################################################
#
# LiloConfig::ReadLiloConfig --
#
#       Extract data from /etc/lilo.conf and construct an internal
#       representation of it.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       Reads /etc/lilo.conf.
#
########################################################################

sub ReadLiloConfig
{
   my $self = shift; # IN/OUT: Invoking instance.

   my $path = PreUpgradeFile('lilo', 'esx');
   my $lines = FileSlurpArray($path);
   unless (defined $lines) {
      LogError("Could not read file '$path'");
      return undef;
   }

   #
   # Look for all referenced devices (boot and root)
   # and for the default label.
   # Also look for all vmnix (ESX) boot labels.  Do this by
   # examining the "label" and "kernel" entries.  If a kernel
   # entry includes "vmnix", save its label in the order it
   # was found in the file.  Otherwise ignore it.
   #
   # $labelSeen tells us if we've seen a label yet at all, or if
   # we've seen a label since finding a new kernel image line
   # (the corresponding label follows the image line).
   #
   # $nextLabelNotESX tells us that we've seen an image line
   # but know that it does not boot a vmnix kernel, so we should
   # ignore the next label (i.e. the label for this image line.
   #

   my $nextLabelNotESX = 0;
   my $labelSeen = 0;
   $self->{esxLabels} = [];
   foreach my $line (@$lines) {
      if ($line =~ /^\s*boot\=(\S+)/) {
         $self->{cfgTree}->{legacyBootDevice} = $1;

      } elsif ($line =~ /^\s*default\=(\S+)/) {
         $self->{defaultLabel} = $1;

      } elsif ($line =~ /^\s*label\=(\S+)/) {
         unless ($nextLabelNotESX) {
            push(@{$self->{esxLabels}}, $1);
            $labelSeen = 1;
         }

      } elsif ($line =~ /^\s*root\=(\S+)/) {
         unless ($nextLabelNotESX) {
            my $device = $1;
            my ($baseDevice, $partNum) = ParseDeviceFilename($device);

            if ($baseDevice) {
               push(@{$self->{rootDevices}}, $baseDevice);
            } else {
               push(@{$self->{rootDevices}}, $device);
            }
         }

      } elsif ($line =~ /^\s*image\=/) {
         $labelSeen = 0;
         if ($line =~ /vmnix/) {
            $nextLabelNotESX = 0;
         } else {
            $nextLabelNotESX = 1;
         }
      }
   }

   return 1 if defined $self->{cfgTree}->{legacyBootDevice};
   LogError("Cannot find boot device in lilo.conf");
   return 0;
}


########################################################################
#
# LiloConfig::GetDefaultBootLabel --
#
#       Get the default boot label, if any is defined.
#
# Results:
#       The default label, or undef if none was specified in the file.
#
# Side effects:
#       None.
#
########################################################################

sub GetDefaultBootLabel
{
   my $self = shift;  # IN: Invoking instance.
   return $self->{defaultLabel};
}


########################################################################
#
# LiloConfig::GetESXBootLabels --
#
#       Get a list of ESX boot labels (i.e. ones with a vmnix kernel)
#       in the order in which they appeared in lilo.conf.
#
# Results:
#       A reference to the internal list of such boot labels.
#
# Side effects:
#       None.
#
########################################################################

sub GetESXBootLabels
{
   my $self = shift;  # IN: Invoking instance.
   return $self->{esxLabels};
}


########################################################################
#
# LiloConfig::BuildGrubConfig --
#
#       Build a GRUB configuration based on data read from LILO.
#
# Results:
#       None.
#
# Side effects:
#       Initializes $self->{grubConfig}.
#
########################################################################

sub BuildGrubConfig 
{
   my $self = shift; # IN/OUT: Invoking instance.

   $self->{grubConfig} = new VMware::Boot::GRUBConfig();

   my $globals = "";
   $globals .= "# Automatically generated by $0\n";
   $globals .= "timeout=10\n";
   $self->{grubConfig}->SetGlobals($globals);
}


########################################################################
#
# LiloConfig::BuildGrubDeviceMap --
#
#       Build a GRUB device map based on data read from LILO.
#
# Results:
#       None.
#
# Side effects:
#       Initializes $self->{grubDeviceMap}.
#
########################################################################

sub BuildGrubDeviceMap 
{
   my $self = shift; # IN/OUT: Invoking instance.

   $self->{grubDeviceMap} = new VMware::Boot::GRUBDeviceMap();
   $self->{grubDeviceMap}->AddMapping("fd0" => "/dev/fd0");

   my @devices = ();
   push(@devices, $self->{cfgTree}->{legacyBootDevice});
   push(@devices, @{$self->{rootDevices}});

   # Filter out duplicates and sort the list
   my %devHash = map { $_ => 1} (@devices);
   @devices = sort(keys(%devHash));

   for (my $i = 0; $i < scalar(@devices); $i++) {
      $self->{grubDeviceMap}->AddMapping("hd$i" => $devices[$i]);
   }
}


########################################################################
#
# LiloConfig::Commit --
#
#       Commits the new GRUB configuration to disk. Note that this
#       will be an incomplete configuration and that esxcfg-boot -b
#       must be run afterwards to fill in the missing parts.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       Writes /boot/grub/grub.conf and /boot/grub/devices.map.
#
########################################################################

sub Commit
{
   my $self = shift; # IN: Invoking instance.

   my $grubdir = StdDir('boot_grub', 'all');
   unless (-d $grubdir) {
      unless (mkdir($grubdir)) {
         LogError("Failed to create $grubdir");
         return 0;
      }
   }

   unless ($self->{grubConfig}->Commit()) {
      LogError("Failed to write grub.conf");
      return 0;
   }

   unless ($self->{grubDeviceMap}->Commit()) {
      LogError("Failed to write grub devices.map");
      return 0;
   }
}


########################################################################
#
# LiloConfig::GetConfigData --
#
#       Return the vmacore config file data structure representing
#       this file's data for esx.conf.
#
# Results:
#       The data structure.
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigData
{
   my $self = shift; # IN: Invoking instance.
   return $self->{cfgTree};
}

1;
