##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# Locator.pm --
#
#       This module maps standard pre-upgrade names to pre-upgrade
#       filesystem locations and post-upgrade backup locations.
#

package VMware::Upgrade::Locator;

use File::Path;
use File::Copy;
use VMware::Log qw(:log);
use VMware::Config qw(:config);

require Exporter;

use strict;

@VMware::Upgrade::Locator::ISA = qw(Exporter);
@VMware::Upgrade::Locator::EXPORT_OK =
   qw(PreUpgradeFile BackUpFiles UnlinkPreUpgradeFiles);


my $etcDir = ConfigGet("test.upgrade.locator.etcDir", '/etc/vmware');
my $backupDir = ConfigGet("test.upgrade.locator.backupDir",
                          "$etcDir/upgrade-backup");
my $backupSuffix = ConfigGet("test.upgrade.locator.backupSuffix", '.esx2');
my $licenseGlob = ConfigGet("test.upgrade.locator.licenseGlob",
                            'license*esx*');

my $files = {
   esx => {
      # Hardware (and related )configuration.
      hwconfig => {
         pre => "$etcDir/hwconfig",
         backup => "$backupDir/hwconfig$backupSuffix",
         remove => 1,
      },
      # Virtual switch/VLAN configuration.
      netmap => {
         pre => "$etcDir/netmap.conf",
         backup => "$backupDir/netmap.conf$backupSuffix",
         remove => 1,
      },
      # FibreChannel persistent bindings.
      pbindings => {
         pre => "$etcDir/pbindings",
         backup => "$backupDir/pbindings$backupSuffix",
         remove => 1,
      },
      # General VMware (mostly vmx) config file.
      config => {
         pre => "$etcDir/config",
         backup => "$backupDir/config$backupSuffix",
         # Don't remove afterwards as it still has non-vmkernel entries.
         remove => 0,
      },
      # VMkernel config file for MUI "advanced options".
      vmkconfig => {
         pre => "$etcDir/vmkconfig",
         backup => "$backupDir/vmkconfig$backupSuffix",
         remove => 1,
      },
      # Module file for vmkernel.
      vmkmodule => {
         pre => "$etcDir/vmkmodule.conf",
         backup => "$backupDir/vmkmodule.conf$backupSuffix",
         remove => 1,
      },
      # Device names formatted for the vmkernel.
      devnames => {
         pre => "$etcDir/devnames.conf",
         backup => "$backupDir/devnames.conf$backupSuffix",
         remove => 1,
      },
      # Core dump partition.
      cdploc => {
         pre => "$etcDir/cdploc",
         backup => "$backupDir/cdploc$backupSuffix",
         remove => 1,
      },
      # Bootloader config for 2.x
      lilo => {
         pre => "/etc/lilo.conf",
         backup => "$backupDir/lilo.conf$backupSuffix",
         remove => 1,
      },
      # Legacy eth mappings.
      legacyethmap => {
         pre => "$etcDir/legacy-eth.map",
         backup => "$backupDir/legacy-eth.map$backupSuffix",
         remove => 1,
      },
      # rc.local init script
      rclocal => {
         pre => "/etc/rc.d/rc.local",
         backup => "$backupDir/rc.local$backupSuffix",
         remove => 0,
      },
   },

   all => {
      # Empty.
   },
};

########################################################################
#
# Locator::PreUpgradeFile --
#
#       Retrieve a standard file location before an ESX2.x->3.0 upgrade.
#
# Results:
#       The path to the file.
#
# Side effects:
#       None.
#
########################################################################

sub PreUpgradeFile
{
   my $fileId = shift;  # IN: The file ID to look up.
   my $product = shift; # IN: The product (optional).

   my $local = $files;
   my $testFiles = ConfigGet("test.upgrade.locator.files");
   if (defined $testFiles) {
      $local = Clone($files);
      foreach my $prodKey (keys %$testFiles) {
         foreach my $key (keys %{$testFiles->{$prodKey}}) {
            $local->{$prodKey}->{$key} = $testFiles->{$prodKey}->{$key};
         }
      }
   }

   my $file;
   if (defined($product)) {
      $file = $local->{$product}->{$fileId}->{pre};
   }
   return $file || $local->{all}->{$fileId}->{pre};
}


########################################################################
#
# Locator::BackUpFiles --
#
#       Copy all known config files from their pre-upgrade locations
#       to their backup locations.  This should be done at the very
#       beginning of the config file upgrade process.
#       Creates the backup directory if needed.
#
# Results:
#       True if successful, false otherwise.
#
# Side effects:
#       File/dir changes as described above.
#
########################################################################

sub BackUpFiles
{
   my $prod = shift;  # IN: Product whose files to back up.

   unless (-d $backupDir) {
      eval { mkpath($backupDir) };
      if ($@) {
         LogError("Could not create backup directory '$backupDir': $!.");
         return 0;
      }
   }

   foreach my $key (keys %{$files->{$prod}}) {
      my $pre = $files->{$prod}->{$key}->{pre};
      my $backup = $files->{$prod}->{$key}->{backup};

      unless (-f $pre) {
         if ($pre =~ /hwconfig/) {
            LogError("Main config file ($pre) not present, cannot upgrade.");
            return 0;
         } else {
            LogInfo("File '$pre' does not exists and so does not need " .
                    "to be backed up or upgraded.");
            next;
         }
      }
      unless (copy($pre, $backup)) {
         if ($pre =~ /hwconfig/) {
            LogError("Could not back up '$pre' to '$backup': $!.");
            return 0;
         }
      }
   }

   foreach my $license (glob("$etcDir/$licenseGlob")) {
      my $backup = $license . $backupSuffix;
      $backup =~ s/^\Q$etcDir/$backupDir/;
      unless (copy($license, $backup)) {
         LogError("Could not back up license '$license' to '$backup': $!.");
         return 0;
      }
   }
   return 1;
}


########################################################################
#
# Locator::UnlinkPreUpgradeFiles --
#
#       Unlink all pre-upgrade files.  This should be done at the
#       very end of the config file upgrade process, and must be done
#       *after* BackUpFiles() is called.  It does not check that the
#       files are backed up before unlinking.
#
# Results:
#       True if sucessful, false otherwise.
#       Failure to remove any file but hwconfig is considered OK, and
#       triggers only a warning, not failure.  Failure to remove
#       hwconfig is fatal because we use it to determine if an upgrade
#       has taken place or not.
#
# Side effects:
#       File removal.
#
########################################################################

sub UnlinkPreUpgradeFiles
{
   my $prod = shift; # IN: Product whose files to unlink.

   my $rc = 1;
   foreach my $key (keys %{$files->{$prod}}) {
      next unless $files->{$prod}->{$key}->{remove};

      my $pre = $files->{$prod}->{$key}->{pre};
      next unless -f $pre;
      unless (unlink($pre)) {
         if ($key eq 'hwconfig') {
            LogError("Could not remove '$pre': $!.");
            $rc = 0;
         } else {
            LogWarn("Could not remove '$pre': $!.");
         }
      }
   }

   foreach my $license (glob("$etcDir/$licenseGlob")) {
      unless (unlink($license)) {
         LogWarn("Could not remove license '$license': $!.");
      }
   }
   return $rc;
}


1;

