##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# MPath.pm --
#
#       This class reads the ESX 2.x format multipathing information
#       from /etc/vmware/hwconfig and parses it into an internal
#       representation suitable for translating to the ESX 3+
#       /etc/vmware/esx.conf format.  This is done outside of
#       the VMware::Upgrade::HWConfig class because it must be done
#       after the ESX 3+ VMkernel is loaded.
#

package VMware::Upgrade::MPath;

use VMware::Log qw(:log);
use VMware::Config::ConfigObj;
use VMware::Config::VmacoreConfigObj;
use VMware::FileSys::StandardLocator qw(StdFile StdCommand);
use VMware::System qw(LogCommandGetOutputArray);

use VMware::PCI::PCIString qw(MakePCIString);
use VMware::PCI::DeviceManager;
use VMware::Upgrade::PBindings;
use VMware::Upgrade::Locator qw(PreUpgradeFile);

use strict;

########################################################################
#
# MPath::new --
#
#       Read an /etc/vmware/hwconfig file and do basic processing
#       on it to produce an internal representation.
# 
# Results:
#       A new instance or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;  # IN: Invoking class name.

   my $hwconfigFile = PreUpgradeFile('hwconfig', 'esx');
   my $hwconfig = new VMware::Config::ConfigObj();
   return undef unless $hwconfig->LoadFromFile($hwconfigFile);
   my $self = {};
   bless $self => $class;

   #
   # It is irregular for esx.conf to be read in a module rather than a script,
   # but we're treating it more like an upgrade source config file
   # here.  So we don't ever write back to it in this module, and just
   # pretend that its not the ultimate destination of the data we're
   # mucking with.  The calling script will handle all write-backs.
   #

   my $esxConfFile = StdFile('esx_config', 'esx');
   my $esxConf = new VMware::Config::VmacoreConfigObj();
   $esxConf->LoadFromFile($esxConfFile);
   my $devMgr = new VMware::PCI::DeviceManager($esxConf->GetTree(['device']));

   my $paths = {};
   my $multipathCfg = $hwconfig->Filter(qr/^multipath/, 1);
   foreach my $key (keys %$multipathCfg) {
      my ($path, $option) = ($key =~ /^multipath\.(vmhba\d+:\d+:\d+)\.(.+)$/);

      unless (defined($path) && defined($option)) {
         LogWarn("Misformatted multipath config entry '$key'.");
         next;
      }

      unless (exists $paths->{$path}) {
         my ($adapter, $target, $lun) = ($path =~ /vmhba(\d+):(\d+):(\d+)/);
         $paths->{$path}->{adapter} = $adapter;
         $paths->{$path}->{target} = $target;
         $paths->{$path}->{lun} = $lun;
      }
      $paths->{$path}->{$option} = $multipathCfg->{$key};
   }

   my $data = $self->GetMultipathData();
   return undef unless defined $data;

   my $pbindingsObj = new VMware::Upgrade::PBindings();
   my $bindings = defined($pbindingsObj) ? $pbindingsObj->GetBindings() : {};

   foreach my $path (keys %$paths) {
      my ($newPath,
          $adaptWwpn,
          $tgtWwpn,
          $vml) = $self->GetNewPathInfo($path,
                                        $paths,
                                        $devMgr,
                                        $bindings,
                                        $data);
      next unless defined $newPath;

      my $state;
      if (defined($paths->{$path}->{disable}) &&
          ($paths->{$path}->{disable} eq "true")) {
         $state = "off";
      } else {
         $state = "on";
      }

      $self->{cfgTree}->{"lun[$vml]"}->{"adapter[$adaptWwpn]"}->{state} =
         $state;
      $self->{cfgTree}->{"lun[$vml]"}->{policy} = $paths->{$path}->{policy};

      next unless defined $paths->{$path}->{prefer};
      my ($newPrefPath, $newPrefAdaptWwpn, $newPrefTgtWwpn) =
         $self->GetNewPathInfo($paths->{$path}->{prefer},
                               $paths,
                               $devMgr,
                               $bindings,
                               $data);
      next unless defined $newPrefPath;

      $self->{cfgTree}->{"lun[$vml]"}->{preferredPath} = {
         adapter => $newPrefAdaptWwpn,
         target => $newPrefTgtWwpn,
         lun => ($newPrefPath =~ /vmhba\d+:\d+:(\d+)/)[0],
      };
   }

   return $self;
}


########################################################################
#
# GetMultipathData
#
#       Get the multipath data from the VMkernel and return it
#       indexed two different ways for the various lookups
#       we'll need later (see GetNewPathInfo()).
#
# Results:
#       Hashtable of data.
#
# Side effects:
#       None.
#
########################################################################

sub GetMultipathData
{
   my $self = shift; # IN: Invoking instance.

   my $cmdList = StdCommand('mpath_bulk', 'esx');
   unless (defined $cmdList) {
      LogError("Could not get multipath configuration tool location.");
      return undef;
   }

   my $data = {};
   my $mpathList = LogCommandGetOutputArray(join(" ", @$cmdList));
   my $byPCIString = {};
   my $byTgtWwpn ={};
   foreach my $mpathCfg (@$mpathList) {
      next if ($mpathCfg =~ /^\s*$/);

      my ($canonPath,
          $vml,
          $cosDev,
          $path,
          $type,
          $rawPCIString,
          $wwpns) = split(' ', $mpathCfg);
      my $adaptWwpn = "";
      my $tgtWwpn = "";

      unless ($cosDev and $cosDev =~ /^\/dev\//) {
         # esxcfg-mpath outputted a PseudoLUN without a cosDev.
         # Skip it since we don't need its path.
         LogWarn("Skipping path for PseudoLUN $canonPath");
         next;
      }

      my $pciString = MakePCIString(($rawPCIString =~ /(\d+):(\d+)\.(\d+)/));
      my $vmhbaAdaper = "";
      my $vmhbaTarget = "";
      my $lun = "";
      ($vmhbaAdaper, $vmhbaTarget, $lun) = split(":", $canonPath);

      #
      # FibreChannel devices may get their targets reordered, so
      # index by first pciString and then target wwpn (which stays the same)
      # to figure out the rest.
      #

      if (defined $wwpns and $wwpns ne '') {
         ($adaptWwpn, $tgtWwpn) = split("<->", $wwpns);

         $data->{byPCIString}->{$pciString}->{$tgtWwpn}->{type} = $type;
         $data->{byPCIString}->{$pciString}->{$tgtWwpn}->{$lun}->{vml} = $vml;
         $data->{byPCIString}->{$pciString}->{$tgtWwpn}->{adaptWwpn} 
            = $adaptWwpn;
         $data->{byPCIString}->{$pciString}->{$tgtWwpn}->{target} =
         ($path =~ /vmhba\d+:(\d+):\d+/)[0];
      }

      #
      # Indexing by PCIString does nothing for non-FibreChannel
      # devices, as we don't have any way to get the target
      # wwpn needed to sort out the right targets.  However,
      # non-FibreChannel devices don't change paths unless the actual
      # hardware changes, and we don't support hardware change
      # concurrent with upgrade.  So store everything by path so
      # we can do non-FibreChannel devices that way.
      #

      $data->{byPath}->{$path}->{type} = $type;
      $data->{byPath}->{$path}->{pciString} = $pciString;
      $data->{byPath}->{$path}->{adaptWwpn} = $adaptWwpn;
      $data->{byPath}->{$path}->{tgtWwpn} = $tgtWwpn;
   }
   return $data;
}


########################################################################
#
# MPath::GetNewPathInfo --
#
#       Determine the current path corresponding to a pre-upgrade path,
#       along with the World Wide Port Numbers of the adapter and target.
#
# Results:
#       A triplet of the new path (vmhbaX:Y:Z), the adapter WWPN,
#       and the WWPN.  If the original path referred to a FibreChannel
#       device and there are no persistent bindings for that target,
#       then an empty list will be returned.
#
# Side effects:
#       None.
#
########################################################################

sub GetNewPathInfo
{
   my $self = shift;  # IN: Invoking instance.
   my $path = shift;  # IN: The path to convert.
   my $paths = shift;  # IN: Hash of path information.
   my $devMgr = shift;  # IN: PCI::DeviceManager of post-upgrade devices.
   my $bindings = shift;  # IN: Persistent bindings for FibreChannel.
   my $data = shift;  # IN: Multipathing info from esxcfg-mpath --bulk.
   
   if (defined($data->{byPath}->{$path}->{type}) &&
       ($data->{byPath}->{$path}->{type} ne "FC")) {
      return ($path,
              $data->{byPath}->{adaptWwpn},
              $data->{byPath}->{tgtWwpn});
   }

   #
   # Either we *know* we're looking at FibreChannel, or the path
   # has changed.  In which case it had better be FibreChannel,
   # since no other type should allow changed paths.
   #

   my $devName = (VMware::PCI::DeviceManager::VMK_DEVICE_TYPE_HBA() .
                 ($path =~ /vmhba(\d+):\d+:\d+/)[0]);
   my $lun = ($path =~ /vmhba\d+:\d+:(\d+)/)[0];

   my $pciString = $devMgr->FindPCIStringByVMKName($devName);
   unless (defined $pciString) {
      LogWarn("Pre-upgrade device '$devName' seems to no longer exist.");
      return ();
   }

   #
   # If we have no persistent binding for a FibreChannel adapter,
   # drop the configuration as we have no way to know which
   # path to apply it to.
   #

   if (not defined $bindings) { LogWarn("Unable to upgrade path $path , no bindings found") ;}
   return () unless defined $bindings;
   my $tgtWwpn = $bindings->{$pciString}->{$paths->{$path}->{target}};
   if (not defined $tgtWwpn) { LogWarn("Unable to upgrade path  $path , no bindings found") ;}
   return () unless defined $tgtWwpn;

   unless (defined $data->{byPCIString}->{$pciString}) {
      LogWarn("No path config data for device '$devName' at '$pciString'.");
      return ();
   }

   my $newPath = $path;
   my $adaptWwpn = $data->{byPCIString}->{$pciString}->{$tgtWwpn}->{adaptWwpn};
   my $newTarget = $data->{byPCIString}->{$pciString}->{$tgtWwpn}->{target};
   my $vml = $data->{byPCIString}->{$pciString}->{$tgtWwpn}->{$lun}->{vml};

   LogInfo("$path is on $adaptWwpn at target $newTarget is $tgtWwpn LUN $lun with a vml name of $vml");

   $newPath =~ s/(vmhba\d+:)\d+(:\d+)/$1$newTarget$2/;

   $self->{reverseMap}->{$newPath} = {$path};
   return ($newPath, $adaptWwpn, $tgtWwpn, $vml);
}


########################################################################
#
# MPath::LookupOldFCPath --
#
#       Public method for looking up the pre-upgrade path corresponding
#       to a given post-upgrade path.  This is used by the VMHBADevs
#       module for FibreChannel adapters.
#
# Results:
#       The pre-upgrade path, or undef if there is no mapping.  This
#       means that either the requested path was for a FibreChannel
#       adapter with no persistent bindings, or the user messed with
#       the BIOS or did some other hardware change that they should
#       not do while upgrading.
#
#       Note that calling this on a path on a non-FC adapter will
#       also return undef.  In these cases old path == new path
#       so there should be no need to make this call.
#
# Side effects:
#       None.
#
########################################################################

sub LookupOldFCPath
{
   my $self = shift;  # IN: Invoking instance.
   my $newPath = shift;  # IN: The post-upgrade path (vmhbaX:Y:Z).

   return $self->{reverseMap}->{$newPath};
}


########################################################################
#
# MPath::GetConfigData --
#
#       Return the root-level tree structure for the esx.conf file
#       produce by the pre-upgrade multipathing configuration as
#       modified by post-upgrade information.
#
# Results:
#       A reference suitable for passing to VmacoreConfigObj::SetTree().
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigData
{
   my $self = shift;  # IN: Invoking instance.

   return $self->{cfgTree};
}


1;

