##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# Netmap.pm --
#
#       This class reads the ESX 2.x format /etc/vmware/netmap.conf
#       file and parses it into an internal representation suitable
#       for translating to the ESX 3+ /etc/vmware/esx.conf format.
#

package VMware::Upgrade::Netmap;

use VMware::Log qw(:log);
use VMware::Config::ConfigObj;
use VMware::Upgrade::Locator qw(PreUpgradeFile);

use strict;


########################################################################
#
# Netmap::new --
#
#       Read an /etc/vmware/netmap.conf file and parse it into
#       a useful internal representation.
#
# Result:
#       The constructed object, or false on failure.
#
# Side effects:
#       Constructs the object, reads netmap.conf.
#
########################################################################

sub new
{
   my $class = shift;  # IN: Invoking class.

   my $self = { networks => [] };
   bless $self => $class;

   my $file = PreUpgradeFile('netmap', 'esx');
   unless (-e $file) {
      LogWarn("No $file file, no virtual switches defined.");
      return $self;
   }

   my $cfg = new VMware::Config::ConfigObj;
   unless ($cfg->LoadFromFile($file)) {
      LogError("Could not load netmap.conf file!");
      return undef;
   }
   my $cfgHash = $cfg->ToHash();
   foreach my $key (keys %$cfgHash) {
      my ($netNum, $entryType) = $key =~ /network(\d+)\.(\S+)/;

      if ($entryType eq 'name') {
         $self->{networks}->[$netNum]->{name} = $cfgHash->{$key};

      } elsif ($entryType eq 'device') {
         if ($cfgHash->{$key} =~ /^([^.]+)\.(\d+)$/) {
            $self->{networks}->[$netNum]->{device} = $1;
            $self->{networks}->[$netNum]->{vlanId} = $2;

         } else {
            #
            # VLAN ID 0 indicates direct vswitch access.
            #

            $self->{networks}->[$netNum]->{device} = $cfgHash->{$key};
            $self->{networks}->[$netNum]->{vlanId} = 0;
         }
      }
   }

   foreach my $network (@{$self->{networks}}) {
      my $dev = $network->{device};
      unless ($network->{device} =~ /^(?:vmnic|vmnet_|bond)\d+$/) {
         #
         # We have a VLAN backed by a vswitch rather than
         # an explicit vmnic/vmnet/bond device.  That vswitch
         # is guaranted to be backed by a vmnic/vmnet/bond.
         #
         # Vswitch names are unique so if we find anything other
         # than exactly one match, it is an error.
         #

         my @vswitches = grep { $dev eq $_->{name}; } @{$self->{networks}};
         unless (scalar(@vswitches) == 1) {
            LogError("Multiple (or zero) vswitches matching device for " .
                     "network '$network->{name}'.");
            return undef;
         }

         $network->{device} = $vswitches[0]->{device};

         unless ($network->{device} =~ /^(?:vmnic|vmnet_|bond)\d+$/) {
            LogError("Expected vmnic, vmnet or bond device, got " .
                     "'$network->{device}' for network " .
                     "'$vswitches[0]->{name}'.");
            return undef;
         }
      }
   }
   return $self;
}


########################################################################
#
# Netmap::GetNetworks --
#
#       Returns the $self->{networks} array ref.
#       This is a list of hashes where each hash represents
#       a network and has the fields 'name', 'dev', and 'vlanId'.
#
#       The form of the 'dev' field is one of vmnicN, vmnet_N or bondN.
#
# Results:
#       As described above.
#
# Side effects:
#       None.
#
########################################################################

sub GetNetworks
{
   my $self = shift; # IN: Invoking instance.

   return $self->{networks};
}


1;

