##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# PBindings.pm --
#
#       This class reads the ESX 2.x format /etc/vmware/pbindings
#       file and parses it into an internal representation suitable
#       for translating to the ESX 3+ /etc/vmware/esx.conf format.
#

package VMware::Upgrade::PBindings;

use VMware::Log qw(:log);
use VMware::Config::ConfigObj;
use VMware::PCI::PCIString qw(MakePCIString);
use VMware::Upgrade::Locator qw(PreUpgradeFile);

use strict;


########################################################################
#
# PBindings::new --
#
#       Opens and parses the pbindings file.
#
# Results:
#       The new instance or undef on failure.
#
# Side effects:
#       Reads the file.
#
########################################################################

sub new
{
   my $class = shift; # IN: Invoking class name.

   my $self = { bindings => {}, };
   bless $self => $class;

   my $path = PreUpgradeFile('pbindings', 'esx');
   return $self unless -e $path;

   my $config = new VMware::Config::ConfigObj();
   unless ($config->LoadFromFile($path)) {
      return undef;
   }

   my $entries = $config->ToHash();
   foreach my $adapter (keys %$entries) {
      #
      # Adapter key starts with bus.slot.function, so build a PCIString.
      #

      my $pciStr = MakePCIString(($adapter =~ /^(\d+)\.(\d+)\.(\d+)/));
      LogDebug("Processing bindings for '$pciStr'.", 1, "upgrade");

      #
      # Parse qlogic or emulex format depending on the driver string in
      # the adapter key.  They have different formats and delimeters.
      #
      # The following uses map {} to construct a hash from the list of
      # target strings.  The regexp matches two items which it returns
      # in list context.  For QLogic, these are in the right order
      # to serve as key => value pairs, but for Emulex they must
      # be reversed using [1, 0] (target is the key, wwpn is the value).
      #

      if ($adapter =~ /qla/) {
         $self->{bindings}->{$pciStr} = { map {
            ($_ =~ /tgt-(\d+).*=([0-9a-f]+)$/);
         } split(';', $entries->{$adapter}) };
      } else {
         $self->{bindings}->{$pciStr} = { map {
            ($_ =~ /^([0-9a-f]+):lpfc\d+t(\d+)/)[1, 0];
         } split(',', $entries->{$adapter}) };
      }
   }
   return $self;
}

########################################################################
#
# PBindings::GetBindings --
#
#       Return a hash mapping PCIStrings to hashes which map
#       target numbers to World Wide Port Numbers.
#
# Results:
#       The mapping descripbed above.
#
# Side effects:
#       None.
#
########################################################################

sub GetBindings
{
   my $self = shift; # IN: Invoking instance.

   return $self->{bindings};
}

1;

