##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# RCLocal.pm --
#
#       This class reads the ESX 2.x format /etc/rc.d/rc.local
#       file and filters out the old VMWARE_RC_DOT_LOCAL section
#       of rc.local. It also provides a method to write out the
#       remainder of the rc.local file back to the original file.
#

package VMware::Upgrade::RCLocal;

use VMware::FileManip qw(FileSlurpArray);
use VMware::FileSys::Move qw(MoveFileKeepDestMode);
use VMware::FileSys::Tmp qw(:tmp);
use VMware::Log qw(:log);
use VMware::Net::Interface;
use VMware::Upgrade::Locator qw(PreUpgradeFile);

use strict;


########################################################################
#
# RCLocal::new --
#
#       Read an /etc/rc.d/rc.local file and do basic processing
#       on it to produce an internal representation.
# 
# Results:
#       A new instance or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift; # IN: Invoking class name.

   my $self = {
      lines => [],
      vmxnetDevices => [],
      eths => {},
   };
   bless $self => $class;

   $self->{path} = PreUpgradeFile('rclocal', 'esx');
   my $lines = FileSlurpArray($self->{path});
   unless (defined $lines) {
      LogError("Could not read file '$self->{path}'");
      return undef;
   }

   my $inVMwareSection = 0;
   foreach my $line (@$lines) {
      if ($line =~ /^# BEGINNING_OF_VMWARE_RC_DOT_LOCAL\s*$/) {
         $inVMwareSection = 1;
         next;
      } elsif ($inVMwareSection &&
               $line =~ /^# END_OF_VMWARE_RC_DOT_LOCAL\s*$/) {
         $inVMwareSection = 0;
         next;
      }

      if ($line =~ /^\s*
                     (?:\/sbin\/)?insmod\s+
                     vmxnet_console\s+
                     .*devName=(\S+)/x) {
         my $devName = $1;

         # Strip quotes
         $devName =~ s/^\"(.*)\"$/$1/;

         # Delimiter can be either semicolon or comma
         my @devices = split(/[;,]/, $devName);

         push(@{$self->{vmxnetDevices}}, @devices);

         # Leave this line out when we write the file back out
         next;
      } elsif ($self->ParseIfConfigLine($line)) {
         # Leave this line out when we write the file back out
         next;
      }

      push(@{$self->{lines}}, $line) unless $inVMwareSection;
   }

   return $self;
}


########################################################################
#
# RCLocal::GetVMXNetDevices --
#
#       Return an anonymous list of vmxnet_console devices.
#
# Results:
#       The list described above.
#
# Side effects:
#       None.
#
########################################################################

sub GetVMXNetDevices
{
   my $self = shift; # IN: Invoking instance.

   return $self->{vmxnetDevices};
}


########################################################################
#
# RCLocal::GetEths --
#
#       Return an anonymous hash mapping ethNs to Interface objects
#       as specified by manual ifconfig lines in rc.local.
#
# Results:
#       The hash described above.
#
# Side effects:
#       None.
#
########################################################################

sub GetEths
{
   my $self = shift; # IN: Invoking instance.

   return $self->{eths};
}


########################################################################
#
# RCLocal::WriteRemainder --
#
#       Write the remaining (non-vmware-specific) options back to
#       the file from which they came.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Writes the file.
#
########################################################################

sub WriteRemainder
{
   my $self = shift; # IN: Invoking instance.

   # Write to temporary file 
   my ($tmpfile, $tmpfh) = TmpFile();
   if (not (defined($tmpfile) && defined($tmpfh))) {
      return 0;
   }

   foreach my $line (@{$self->{lines}}) {
      print $tmpfh $line;
   }

   unless (close($tmpfh)) {
      LogError("Failed to close $self->{path}: $!");
      return 0;
   }
   
   # Change rc.local 
   unless (MoveFileKeepDestMode($tmpfile, $self->{path})) {
      LogError("Unable to create new rc.local file.  Reason $!");
      return 0;
   }

   RemoveFileFromTmpCleanup($tmpfile);

   return 1;
}


########################################################################
#
# RCLocal::ParseIP --
#
#       Private method to determine if the specified string is a valid
#       IP address.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       None.
#
########################################################################

sub ParseIP {
   my $line = shift;    # IN: The string to parse

   return $line =~ /^(?:\d+\.){3}\d+$/;
}


########################################################################
#
# RCLocal::ParseIfConfigLine --
#
#       Private method to parse a manually-specified ifconfig line
#       into an Interface object. If successful, saves the Interface
#       into the eth->Interface map.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Modifies $self->{eths} on success.
#
########################################################################

sub ParseIfConfigLine {
   my $self = shift;    # IN: Invoking instance.
   my $line = shift;    # IN: The line to parse

   # Strip out whitespace and comments
   $line =~ s/^\s+//;
   $line =~ s/\s+$//;
   $line =~ s/#.*//;

   # Is this a blank line?
   unless ($line) {
      return 0;
   }

   my @fields = split(/\s+/, $line);
   unless (scalar(@fields) >= 4) {
      return 0;
   }

   my $cmd = shift(@fields);
   unless ($cmd eq "ifconfig" or $cmd eq "/sbin/ifconfig") {
      return 0;
   }

   my $ethN = shift(@fields);
   unless ($ethN =~ /^eth\d+$/) {
      return 0;
   }

   my $enabled = 0;
   my $ipaddr = undef;
   my $netmask = undef;
   my $broadcast = undef;

   while (@fields) {
      my $opt = shift(@fields);
      if ($opt eq "up") {
         $enabled = 1;
      } elsif (ParseIP($opt)) {
         # Implicit address option.
         $ipaddr = $opt;
      } elsif ($opt eq "netmask") {
         my $val = shift(@fields);
         if (ParseIP($val)) {
            $netmask = $val;
         }
      } elsif ($opt eq "broadcast") {
         my $val = shift(@fields);
         if (ParseIP($val)) {
            $broadcast = $val;
         }
      }
   }

   # Require some basic parameters.
   unless ($enabled and $ipaddr) {
      return 0;
   }

   # If the user didn't specify a netmask, provide a default.
   unless ($netmask) {
      my @ip = split(/\./, $ipaddr);
      my $netclass = $ip[0];

      if ($netclass <= 127) {
         $netmask = "255.0.0.0";
      } elsif ($netclass <= 191) {
         $netmask = "255.255.0.0";
      } else {
         $netmask = "255.255.255.0";
      }
   }

   my $if = new VMware::Net::Interface();
   $if->SetOnBoot('yes');
   $if->SetBootProto(VMware::Net::Interface::BOOTPROTO_STATIC);
   $if->SetIPAddr($ipaddr);
   $if->SetNetmask($netmask);
   $if->SetBroadcast($broadcast);

   $self->{eths}->{$ethN} = $if;
   return 1;
}


1;
