##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# VMHBADevs.pm --
#
#

package VMware::Upgrade::VMHBADevs;

use VMware::Log qw(:log);
use VMware::Config::VmacoreConfigObj;
use VMware::FileSys::StandardLocator qw(StdFile StdCommand);
use VMware::System qw(LogCommandGetOutputArray);

use VMware::PCI::DeviceManager;
use VMware::Upgrade::PBindings;

use strict;


########################################################################
#
# VMHBADevs::new --
#
#       Get the UUIDs for the vmfs partitions and match them to old
#       vmhba devices from before an upgrade.
# 
# Results:
#       A new instance or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;  # IN: Invoking class name.
   my $devMgr = shift; # IN: Current (populated) device manager.
   my $mpath = shift;  # IN: Current (populated) mpath object.

   my $self = {
      partMap => {}, # Map from vmhbaW:X:Y:Z to uuid.
      cfgTree => {}, # Config data hash.
   };
   bless $self => $class;

   $self->LoadPartitionData();

   my @fcAdapters = map {
      $_->GetVMKDeviceName();
   } grep {
      $_->IsFcEquiv();
   } @{$devMgr->GetDevices()};

   foreach my $part (keys %{$self->{partMap}}) {
      my $adjustedPart;
      if (grep { $part =~ /^$_/; } @fcAdapters) {
         my ($path, $partNum) = ($part =~ /^(vmhba\d+:\d+:\d+):(\d+)$/);

         my $adjustedPath = $mpath->LookupOldFCPath($path);
         next unless defined $adjustedPath;

         $adjustedPart = "$adjustedPath:$partNum";

      } else {
         $adjustedPart = $part;
      }

      my $uuid = $self->{partMap}->{$adjustedPart};
      $self->{cfgTree}->{$part} = "/vmfs/volumes/$uuid";
   }

   return $self;
}


########################################################################
#
# VMHBADevs::LoadPartitionData --
#
#       Load the vmhbadevs data from the VMkernel.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Populates $self->{partMap}, the map from vmhbaW:X:Y:Z to uuid.
#
########################################################################

sub LoadPartitionData
{
   my $self = shift; # IN/OUT: Invoking instance.

   my $cmdList = StdCommand('vmhbadevs', 'esx');
   unless (defined $cmdList) {
      LogError("Could not get vmhbadevs configuration tool location.");
      return undef;
   }

   $self->{partMap} = {};
   my $lines = LogCommandGetOutputArray(join(" ", @$cmdList));
   foreach my $line (@$lines) {
      my ($vmhbaPath, $cosDev, $uuid) = split(" ", $line);
      $self->{partMap}->{$vmhbaPath} = $uuid;
   }
   return 1;
}


########################################################################
#
# VMHBADevs::GetConfigData --
#
#       Return the tree structure for vmhbadev upgrade mappings
#       in the esx.conf file.
#
# Results:
#       A reference suitable for passing to VmacoreConfigObj::SetTree().
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigData
{
   my $self = shift;  # IN: Invoking instance.

   return $self->{cfgTree};
}


1;

