
##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# VMKConfig.pm --
#
#       This class reads the ESX 2.x format /etc/vmware/vmkconfig
#       file and parses it into an internal representation suitable
#       for translating to the ESX 3+ /etc/vmware/esx.conf format.
#

package VMware::Upgrade::VMKConfig;

use VMware::Log qw(:log);
use VMware::Config::VMKConfigObj;
use VMware::Config::VmacoreConfigObj;
use VMware::Upgrade::Locator qw(PreUpgradeFile);

use strict;


########################################################################
#
# VMKConfig::new --
#
#       Read an /etc/vmware/vmkconfig file and do basic processing
#       on it to produce an internal representation.
# 
#       Shoving the data in and out of various ConfigObjs and hashes
#       like this is a bit roundabout, but very easy and we don't
#       really need to optimize this since its a one-time upgrade
#       cost and its not a huge file.
#
# Results:
#       A new instance or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift; # IN: Invoking class name.

   my $self = { cfgData => {} };
   bless $self => $class;

   # It is OK if this config file does not exist.
   my $path = PreUpgradeFile('vmkconfig', 'esx');
   return $self unless -e $path;

   my $config = new VMware::Config::VMKConfigObj();
   unless ($config->LoadFromFile($path)) {
      LogError("Could not read file '$path': $!");
      return undef;
   }
   my $hash = $config->ToHash();

   foreach my $key (%$hash) {
      $key =~ s:^:/:;
   }
   my $vmacoreConfig = new VMware::Config::VmacoreConfigObj();
   $vmacoreConfig->LoadFromHash($hash);
   $self->{cfgData} = $vmacoreConfig->GetTree([]);

   return $self;
}


########################################################################
#
# VMKConfig::GetConfigData --
#
#       Return the vmacore config file data structure representing
#       this file's data for esx.conf.
#
# Results:
#       The data structure.
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigData
{
   my $self = shift; # IN: Invoking instance.
   return $self->{cfgData};
}

1;

