
##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# VMKModule.pm --
#
#       This class reads the ESX 2.x format /etc/vmware/vmkmodule.conf
#       file and parses it into an internal representation suitable
#       for translating to the ESX 3+ /etc/vmware/esx.conf format.
#
#       /etc/vmware/vmkmodule.conf is only used if a special comment
#       line is present.  Otherwise modules and their options are
#       determined directly from /etc/vmware/hwconfig.
#

package VMware::Upgrade::VMKModule;

use VMware::Log qw(:log);
use VMware::Config::ConfigObj;
use VMware::Upgrade::Locator qw(PreUpgradeFile);
use VMware::FileManip qw(FileSlurpArray);

use strict;


########################################################################
#
# VMKModule::new --
#
#       Read an /etc/vmware/vmkmodule.conf file and do basic processing
#       on it to produce an internal representation.
# 
# Results:
#       A new instance or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift; # IN: Invoking class name.

   my $self = { modules => undef, };
   bless $self => $class;

   # It is OK if this config file does not exist.
   my $path = PreUpgradeFile('vmkmodule', 'esx');
   return $self unless -e $path;

   my $lines = FileSlurpArray($path);
   unless (defined $lines) {
      LogError("Could not read file '$path'");
      return undef;
   }

   #
   # Yes, the usage of this file really does hinge on the
   # presence or absence of a comment.  This is clearly
   # documented in the printed manuals.
   #
   # Note:  This grep does not require the keyword to be
   # in a comment, but neither does the ESX 2.x implementation
   # in services.sh (i.e. /etc/rc.d/init.d/vmware).  So this
   # just makes them match which seems like the correct action.
   #

   unless (grep(/MANUAL-CONFIG/, @$lines)) {
      return $self;
   }

   #
   # Now set up the modules array because "undef" means that
   # the file is irrelevant, while an empty array just means
   # that there are no modules/options in the file.  Which
   # would be wierd, but sometimes things are wierd.
   #

   $self->{modules} = [];

   foreach my $line (@$lines) {
      next if $line =~ /^\s*$/;
      next if $line =~ /\s*#/;
      $line =~ s/#.*//;

      my @fields = split(/\s+/, $line);
      my $module = {
         driver => $fields[0],
         module => $fields[1],
      };
      if (@fields > 2) {
         my @optstrings;
         if ($fields[2] eq '-d') {
            #
            # Skip sharing options in $fields[3]
            #

            @optstrings = @fields[(4 .. $#fields)];
         } else {
            @optstrings = @fields[(2 .. $#fields)];
         }
         $module->{options} = $self->ParseOptionString(@optstrings);
      }
      push(@{$self->{modules}}, $module);
   }
   return $self;
}


########################################################################
#
# VMKModule::ParseOptionString --
#
#       Parse the text form of the device options string into
#       a list of option settings for each option for the device.
#       Return the resulting structure.
#
# Results:
#       A hash mapping option names to a list of settings for the
#       devices in bus:slot.function order.
#
# Side effects:
#       None.
#
########################################################################

sub ParseOptionString
{
   my $self = shift;  # IN: Invoking instance.
   my @optstrings = @_;  # IN: Options string from vmkmodule.conf.

   my $options = {};
   foreach my $opt (@optstrings) {
      my ($name, $valstring) = split("=", $opt);
      $options->{$name} = [(split(",", $valstring))];
   }
   return $options;
}


########################################################################
#
# VMKModule::GetModules --
#
#       Return a list of modules with their drivers and options.
#       Each list is a hash with the keys 'driver', 'module'
#       and 'options', which should be self-explanatory.
#
#       Note that the position of values in the options string will
#       be relative to the *current* (i.e. pre-upgrade) PCI configuration,
#       and will need to be merged with hwconfig before the devices
#       are reassigned by Dali conventions.  At which point they will
#       almost certainly be wrong as most systems will gain a network
#       card under the vmkernel that used to be COS-exclusive.  And
#       options are primarily used for nic duplex settings.
#
# Results:
#       The data structure.
#
# Side effects:
#       None.
#
########################################################################

sub GetModules
{
   my $self = shift; # IN: Invoking instance.
   return $self->{modules};
}

1;

