##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# VMwareConfig.pm --
#
#       This class reads the ESX 2.x format /etc/vmware/config
#       file and parses the vmkernel-oriented parts into an internal
#       representation suitable for translating to the ESX 3+
#       /etc/vmware/esx.conf format.  It also provides a method
#       to write only the non-vmkernel parts back to the original file
#

package VMware::Upgrade::VMwareConfig;

use VMware::Log qw(:log);
use VMware::Config::ConfigObj;
use VMware::Config::VmacoreConfigObj;
use VMware::Upgrade::Locator qw(PreUpgradeFile);

use strict;


########################################################################
#
# VMwareConfig::new --
#
#       Read an /etc/vmware/config file and do basic processing
#       on it to produce an internal representation.
# 
# Results:
#       A new instance or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift; # IN: Invoking class name.

   my $self = { cfgData => {}, };
   bless $self => $class;

   $self->{path} = PreUpgradeFile('config', 'esx');
   $self->{vmwareCfg} = new VMware::Config::ConfigObj();
   unless ($self->{vmwareCfg}->ReadFile($self->{path})) {
      LogError("Could not read file '$self->{path}': $!");
      return undef;
   }
   my $config = $self->{vmwareCfg}->Filter(qr/^vmkernel\./, 1);
   my $newConfig = {};
   foreach my $key (%$config) {
      #
      # Trim vmkernel off as the code that assembles esx.conf
      # will supply that as the key under which all this is set.
      #
      my $newKey = $key;
      $newKey =~ s:^vmkernel\.:/:;
      $newKey =~ s:\.:/:;
      $newConfig->{newKey} = $config->{key};
   }

   my $vmacoreConfig = new VMware::Config::VmacoreConfigObj();
   $vmacoreConfig->LoadFromHash(%$newConfig);
   $self->{cfgData} = $vmacoreConfig->GetTree([]);

   #
   # XXX: Do we need to also handle security entries?
   #

   return $self;
}


########################################################################
#
# VMwareConfig::GetConfigData --
#
#       Get the esx.conf data in the appropriate data structure form.
#
# Results:
#       The data structure.
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigData
{
   my $self = shift; # IN: Invoking instance.
   return $self->{cfgData};
}


########################################################################
#
# VMwareConfig::WriteRemainder --
#
#       Write the remaining (non-vmkernel) options back to the file
#       from which they came.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Writes (and maybe backs up) the file.
#
########################################################################

sub WriteRemainder
{
   my $self = shift; # IN: Invoking instance.

   unless ($self->{vmwareCfg}->WriteFile($self->{path})) {
      LogError("Could not write config back to '$self->{path}': $!");
      return undef;
   }
   return 1;
}

1;
