use strict;

use File::Basename;
use File::Spec;
use Test;

BEGIN {
   plan tests => 130;

   my $path = File::Spec->catfile(dirname($0), "..", "..");
   push @INC, $path;
   push @INC, ".";
}

use VMware::Config::ConfigObj;

use VMware::Config qw(:config ConfigSet ConfigSetValidator);
use VMware::Log qw(:count);

#
# Squelch log messages- unit test diagnostics are
# sufficient, and this results in a cleaner-looking run.
#

my $log = VMware::Log::LogPeek();
$log->ClearDestinations();


########################################################################
# Create temporary files for reading
########################################################################

sub MakeTempFile {
   my $filename = shift;
   my $content = shift;

   open(FILE, ">$filename");
   print FILE $content;
   close(FILE);
}

# Test loading from hash

ok(!(VMware::Config::LoadFromHash(
	{
		"config.b", "YYY",
		" a b ==", "ZZZ"
	})),
	1,
	"LoadFromHash accepted bad values");

ok(!(VMware::Config::LoadFromHash(
	{
		"config.b", "YYY",
		"config.c", [ ]
	})),
	1,
	"LoadFromHash accepted bad values");

ok(VMware::Config::LoadFromHash({"config.b" => "b"}), 1);
ok(ConfigGet("config.b"), "b", "LoadFromHash corrupted original values");


# Create a config instance
my $cfg = VMware::Config::ConfigObj->new();
ok(defined($cfg), 1, "ConfigObj::new returned undef");
ok(ref($cfg), "VMware::Config::ConfigObj",
              "ConfigObj::new not blessed properly");

# Test set/get on instance
$cfg->Set("config.x", 123);
ok($cfg->Get("config.x"), 123, "Couldn't get/set same value on instance");
ok(!defined(ConfigGet("config.x")), 1, "Instance name reachable globally");

# Test set/unset on instance
$cfg->Set("config.y",234);
ok($cfg->Get("config.y"), 234, "Couldn't get/set same value on instance");
ok($cfg->UnSet("config.y"),234, "Couldn't unset value on instance");
ok(!defined($cfg->Get("config.y")), 1, "Unset didn't succeed for instance");
ok(!defined($cfg->UnSet("config.y")),1, "Unset didn't delete properly");

# Test set/get on global
ConfigSet("config.x", 456);
ok($cfg->Get("config.x"), 123, "Instance corrupted by global");
ok(ConfigGet("config.x"), 456, "Couldn't get/set same value on global");
ok($cfg->Get("config.x", 42), 123, "Instance defaulting incorrect");
ok(ConfigGet("config.x", 42), 456, "Global defaulting incorrect");
ok($cfg->Get("nothere", 88), 88, "Instance defaulting incorrect");
ok(ConfigGet("nothere", "default"), "default", "Global defaulting incorrect");

# Test ConfigGetBoolean
ConfigSet("config.bool", "TrUe");
ok(ConfigGetBoolean("config.bool"), 1, "Boolean value not recognized");
ok(ConfigGetBoolean("config.bool", "faLSe"),
                     1,
                     "Boolean defaulting not correct");
ConfigSet("config.bool", "0");
ok(ConfigGetBoolean("config.bool"), 0, "Boolean value not recognized");
ok(ConfigGetBoolean("config.nothere", 1), 1, "Boolean defaulting not correct");

# Test ConfigGetArray
ConfigSet("config.array", "abc, def ,hij,\tklm");
my @list = ConfigGetArray("config.array");
ok( ($list[0] eq "abc") &&
    ($list[1] eq "def") &&
    ($list[2] eq "hij") &&
    ($list[3] eq "klm"), 1, "Array not split correctly");
@list = ConfigGetArray("config.array", [1, 2, 3]);
ok( ($list[0] eq "abc") &&
    ($list[1] eq "def") &&
    ($list[2] eq "hij") &&
    ($list[3] eq "klm"), 1, "Array defaulting incorrect");
my $list = ConfigGetArray("config.array", [1, 2, 3]);
ok( ($list->[0] eq "abc") &&
    ($list->[1] eq "def") &&
    ($list->[2] eq "hij") &&
    ($list->[3] eq "klm"), 1, "Array references incorrect");
@list = ConfigGetArray("config.array", 1, 2, 3);
ok( ($list[0] eq "abc") &&
    ($list[1] eq "def") &&
    ($list[2] eq "hij") &&
    ($list[3] eq "klm"), 1, "Array defaulting incorrect");
$list = ConfigGetArray("config.array", 1, 2, 3);
ok( ($list->[0] eq "abc") &&
    ($list->[1] eq "def") &&
    ($list->[2] eq "hij") &&
    ($list->[3] eq "klm"), 1, "Array references incorrect");

ConfigSet("config.array", "");
@list = ConfigGetArray("config.array");
ok(@list, 0, "Array size not zero");
$list = ConfigGetArray("config.array");
ok(@$list, 0, "Array ref size not zero");
@list = ConfigGetArray("config.nothere", ["a", "b"]);
ok(("a" eq $list[0]) && ("b" eq $list[1]));
$list = ConfigGetArray("config.nothere", ["a", "b"]);
ok(("a" eq $list->[0]) && ("b" eq $list->[1]));
@list = ConfigGetArray("config.nothere", "a", "b");
ok(("a" eq $list[0]) && ("b" eq $list[1]));
$list = ConfigGetArray("config.nothere", "a", "b");
ok(("a" eq $list->[0]) && ("b" eq $list->[1]));
$list = ConfigGetArray("config.nothere", []);
ok(scalar(@$list), 0);

# Test ToHash
my %configCopy = %{$cfg->ToHash()};
$configCopy{"config.x"} = "888";
ok($cfg->Get("config.x"), "123", "Modifying hash copy corrupted object");

# Test reading files
ok(!defined($cfg->LoadFromFile()), 1, "File load not passed a file");

ok(!defined($cfg->LoadFromFile("not_exist.txt")), 1,
	"File load passed a nonexistant file");

# File with an invalid line
MakeTempFile("config_invalid.txt", << 'EOD');
invalid_line
EOD

ok(!defined($cfg->LoadFromFile("config_invalid.txt")), 1,
	"File with invalid line was accepted");

ok($cfg->LoadFromFile("config_invalid.txt", 1), 1,
	"File with invalid line was accepted, with ignore flag");

unlink "config_invalid.txt";

MakeTempFile("config_test.txt", << 'EOD');
#!/usr/bin/bash -x -y -z
#!/usr/bin/perl -w
# Comment
	# Comment
test.1=X
test:2 = X
test-3 = "Y" 
test.4 = "A B" 
test.5 = "A ""B""" 
test.6=  	  value # comment
test.7=#
test.8="a long string with spaces, # hashes, ""and quotes""" # comment
test.9="a value continued " \
       accross \
       "multiple lines"
EOD

ok($cfg->LoadFromFile("config_test.txt"), 1, "Valid file was not accepted");

unlink "config_test.txt";

ok($cfg->Get("test.1"), "X", "test.1 not parsed correctly");
ok($cfg->Get("test:2"), "X", "test.2 not parsed correctly");
ok($cfg->Get("test-3"), "Y", "test.3 not parsed correctly");
ok($cfg->Get("test.4"), "A B", "test.4 not parsed correctly");
ok($cfg->Get("test.5"), "A \"B\"", "test.5 not parsed correctly");
ok($cfg->Get("test.6"), "value", "test.6 not parsed correctly");
ok($cfg->Get("test.7"), "", "test.7 not parsed correctly");
ok($cfg->Get("test.8"),
	"a long string with spaces, # hashes, \"and quotes\"",
	"test.8 not parsed correctly");
ok($cfg->Get("test.9"),
        "a value continued accrossmultiple lines",
        "test.9 not parsed correctly");

ok($cfg->Get("config.binary"), "/usr/bin/bash", "config.binary not parsed");
ok($cfg->Get("config.opts"), "-x -y -z", "config.opts not parsed");


#
# Test for LoadFromHash
#

ok($cfg->LoadFromHash(
   {
      param1 => "419",
      param2 => "826",
      "test.6" => "RRV"
   }),
   1,
   "LoadFromHash failed");

ok($cfg->Get("param1"), "419", "LoadFromHash didn't set proper value");
ok($cfg->Get("test.6"), "RRV", "LoadFromHash didn't set proper value");
ok($cfg->Get("test.1"), "X", "Old values not preserved after LoadFromHash");

ok(defined($cfg->Clear()), 1, "Clear() returned undef");
ok(!defined($cfg->Get("test.1")), 1, "Value remains after clear");

# ToString and config.*
$cfg->Set("config.binary", "/usr/bin/vmware");
$cfg->Set("config.opts", "-G");
$cfg->Set("displayName", "(empty)");
ok($cfg->ToString(), "#!/usr/bin/vmware -G\ndisplayName=\"(empty)\"\n",
	"ToString() didn't form config.binary / config.opts correctly");


#
# Tests for validation.
#

my $validHash = {
   valid => "A valid config option.",
};

my $validator = VMware::Config::ConfigObj->new();
$validator->LoadFromHash($validHash);

my $vCfg = VMware::Config::ConfigObj->new(1,$validator);
ok($vCfg->Set("valid", 1), 1);
ok(not defined($vCfg->Set("invalid", 1)));

ok($vCfg->SetValidator(), $validator);
ok($vCfg->Set("invalid", 1));
ok($vCfg->Get("invalid"));

ok(not defined($vCfg->SetValidator($validator)));
ok(not defined($vCfg->Get("invalid")));

$validator->Set("array", "Valid array option");
$validator->Set("hash", "Valid hash option");
$validator->Set("bool", "Valid Boolean option");

ok(not defined(ConfigSetValidator($validator)));
ok(not defined(ConfigSet("foo", 1)));

ok(ConfigSet("array", undef));

my @array = ConfigGetArray("array", ["a", "b", "c"]);
ok(join("", @array), "abc");

ok(ConfigGetArray("badArray", ["a", "b", "c"]), undef);

my $warnings = LogGetWarnCount();
ok(ConfigSet("hash", undef));
ok(not defined(ConfigGetHash("hash")));
ok(LogGetWarnCount(), $warnings);

ok(ConfigSet("hash", "foo=bar, biz=baz"));
my $hash = ConfigGetHash("hash");
ok(scalar(keys %$hash), 2);
ok($hash->{foo}, "bar");
ok($hash->{biz}, "baz");

ok(ConfigSet("bool", "true"));
ok(ConfigGetBoolean("bool", 0), 1);

ok(ConfigSetValidator(), $validator);
ok(ConfigSet("foo", 1));


#
# Tests for read/write file, lock/unlock file and keys.
#

MakeTempFile("config_test2.txt", << 'EOD');
foo = 1
bar = 2
EOD

ok($cfg->ReadFile("config_test2.txt"), 1, "Valid file not accepted by ReadFile");

ok(scalar(keys(%{$cfg->{config}})), 2, "ReadFile did not replace contents");
ok($cfg->Get("foo"), 1, "Wrong value after ReadFile");
ok($cfg->Get("bar"), 2, "Wrong value after ReadFile");

my $preLockHash = {
   foo => "FOO!",
   bar => "BAR!",
   zippy => "The Pinhead!",
   newkey => "Its new and improved!",
};

my $preLockValidator = VMware::Config::ConfigObj->new();
$preLockValidator->LoadFromHash($preLockHash);
$cfg->SetValidator($preLockValidator);
ok(!defined($cfg->Set("InvalidKey", 42)), 1, "Invalid key set");

$cfg->LockKeys();
ok(!defined($cfg->UnSet("foo")), 1, "Keys locked but deletion allowed");
ok(!defined($cfg->Set("zippy")), 1, "Keys locked but otherwise valid add allowed");
ok($cfg->Set("bar", undef), 1, "Could not set key to undef, should be able to");
$cfg->UnlockKeys();
ok($cfg->UnSet("foo"), 1, "Could not unset foo after unlocking keys");
ok($cfg->Set("zippy", 88), 1, "Could not set valid key zippy");
ok(!defined($cfg->Set("InvalidKey", 42)), 1,
   "Invalid key accepted- key lock/unlock must have destroyed the old validator");

ok($cfg->WriteFile("config_test3.txt"), 1, "Could not write using WriteFile");
ok(-e "config_test3.txt", 1, "Written file does not exist");

ok($cfg->LockAndReadFile("config_test2.txt"), 1, "Could not lock/read file");
ok($cfg->LockFile("config_test2.txt"), 1, "Redundant lock file should be ignored");
ok(!defined($cfg->LockFile("config_test3.txt")), 1,
   "Should not be able to lock two files");

#
# Do not try this at home.  Fake out $$ so LockFile thinks we are
# another process.  Need to localize $$ or else Perl complains about
# modifying it (and yet obligingly modifies it anyway).
#

local $$ = $$;
my $oldPid = $$;
my $newPid = ++$$;

my $otherCfg = VMware::Config::ConfigObj->new();
ok($otherCfg->LockFile("config_test2.txt"), 0, "Should have been denied held lock");

$$ = $oldPid;
$cfg->Set("newkey", "newval");
ok($cfg->WriteAndUnlockFile(), 1, "Could not write/unlock file");

$$ = $newPid;
ok($otherCfg->LockFile("config_test2.txt"), 1, "Could not lock released file");
ok($otherCfg->ReadFile("config_test2.txt"), 1, "Could not read locked file");
ok($otherCfg->Get("newkey"), "newval", "New key did not get written to file");
ok($otherCfg->UnlockFile(), 1, "Could not unlock file");
$$ = $oldPid;

unlink("config_test2.txt");
unlink("config_test3.txt");


#
# Tests for Filter and Merge
#

$cfg->Clear();
$cfg->SetValidator();
$cfg->LoadFromHash( {
   foo => 1,
   fooboo => 2,
   oofoo => 3,
   foofoo => 4,
   bar => 5,
} );

my $filtered = $cfg->Filter(qr/foo/);
ok(defined $filtered);

my @fkeys = keys %$filtered;
ok(scalar(@fkeys), 4, "expected /foo/ to match foo, fooboo, foofoo and oofoo");
ok($filtered->{foo}, 1);
ok($filtered->{fooboo}, 2);
ok($filtered->{oofoo}, 3);
ok($filtered->{foofoo}, 4);

#
# Now test filter+delete.
#

my $cfg2 = new VMware::Config::ConfigObj();
$cfg2->LoadFromHash($cfg->ToHash());

my $filtered2 = $cfg2->Filter(qr/foo/, 1);
my @remainingKeys = keys %{$cfg2->ToHash()};

ok(scalar(@remainingKeys), 1,
   "Only one key should remain after filter+delete: (@remainingKeys).");
ok($cfg2->Get("bar"), 5, "bar = 5 should be remaining key/value pair.");

my @fkeys2 = keys %$filtered;
ok(scalar(@fkeys2), scalar(@fkeys), "Filter+delete should yield same number " .
                                    "of entries as plain Filter: (@fkeys2).");
ok($filtered2->{foo}, $filtered->{foo});
ok($filtered2->{fooboo}, $filtered->{fooboo});
ok($filtered2->{oofoo}, $filtered->{oofoo});
ok($filtered2->{foofoo}, $filtered->{foofoo});

#
# Note merge with /^foo/, not /foo/ which was used as filter.
#

delete $filtered->{fooboo};
$filtered->{foozle} = 6;
$filtered->{whatever} = 7;
$filtered->{foofoo} = 8;
$filtered->{oofoo} = 20;

ok($cfg->Merge(qr/^foo/, $filtered));
my @mkeys = keys %{$cfg->ToHash()};
ok(scalar(@mkeys), 5, "should be 5 entries after merge");
ok($cfg->Get("foo"), 1, "foo should not have been disturbed");
ok($cfg->Get("foozle"), 6, "foozle should have been added as 6");
ok($cfg->Get("foofoo"), 8, "foofoo should have been changed to 8");
ok(!defined($cfg->Get("fooboo")), 1, "fooboo should have been deleted");
ok(!defined($cfg->Get("whatever")), 1, "whatever should not have been added");
ok($cfg->Get("oofoo"), 3, "oofoo should be undisturbed (not a match for ^foo)");
ok($cfg->Get("bar"), 5, "bar should be undisturbed (not a match for ^foo)");

#
# Now test the exception argument to merge.
#

my $newData = {
   foo => 2,
   foozle => 42,
};
ok($cfg->Merge(qr/^foo/, $newData, qr/zle$/));
@mkeys = keys %{$cfg->ToHash()};
ok(scalar(@mkeys), 4, "should have 4 entries after merge with exceptions");
ok($cfg->Get("foo"), 2, "foo should have been changed to 2");
ok($cfg->Get("foozle"), 6, "foozle should have been undisturbed (exception)");
ok(!defined($cfg->Get("foofoo")), 1, "foofoo should have been deleted");

exit(0);
