
use strict;
use File::Basename;
use File::Spec;
use Test;

BEGIN {
   plan tests => 53;
   my $path = File::Spec->catfile(dirname($0), "..", "..");
   push @INC, $path;
}

use VMware::Log qw(:log :manip :count);

#
# Turn off all of the formatting options because they mess up comparisions
# and they should be covered by the Format unit tests anyway.
#

my $fmt = new VMware::Log::Format::();
$fmt->ClearDate();
$fmt->ClearTime();
$fmt->ClearCode();


#
# The basic idea is to write everything to the "write"
# ends of the pipes, set autoflush so we don't deadlock,
# and read the results out the read end.
#

pipe (R1, W1) || die "Could not make pipe 1: $!";
pipe (R2, W2) || die "Could not make pipe 2: $!";
pipe (R3, W3) || die "Could not make pipe 2: $!";

select W1;
$| = 1;
select W2;
$| = 1;
select W3;
$| = 1;
select STDOUT;

my $w1 = \*W1;
my $r1 = \*R1;
my $w2 = \*W2;
my $r2 = \*R2;
my $w3 = \*W3;
my $r3 = \*R3;

my $log1 = VMware::Log::LogObj->new($w1, "pipe 1", $fmt,
                                  VMware::Log::LogObj::ON_ERROR_FATAL);
my $log2 = VMware::Log::LogObj->new($w2, "pipe 2", $fmt,
                                  VMware::Log::LogObj::ON_ERROR_FATAL);
my $log3 = VMware::Log::LogObj->new($w3, "pipe 3", $fmt,
                                  VMware::Log::LogObj::ON_ERROR_FATAL);

ok(<$r1>, ($fmt->Description() . "\n"));
ok(<$r2>, ($fmt->Description() . "\n"));
ok(<$r3>, ($fmt->Description() . "\n"));

ok(LogGetInfoCount(), 0);
ok(LogGetWarnCount(), 0);
ok(LogGetErrorCount(), 0);
ok(LogGetDebugCount(), 0);
ok(LogGetCount(VMware::Log::ID_INFO), 0);

#
# Manipulation tests- pushing, popping, muting, etc.
#

my @priors = LogSet($log1);
ok(scalar(@priors), 1);
ok($priors[0] ne $log1);
ok(LogPeek() eq $log1);

ok(LogInfo("Testing info."), 1);
ok(<$r1>, (VMware::Log::ID_INFO . ":  Testing info.\n"));

ok(LogPush($log2) eq $log1);
ok(LogWarn("Testing warn."), 1);
ok(<$r2>, (VMware::Log::ID_WARN . ":  Testing warn.\n"));
ok(LogPop() eq $log2);
ok(LogError("Testing error."), 1);
ok(<$r1>, (VMware::Log::ID_ERROR . ":  Testing error.\n"));

#
# Retest push/pop with muting.
#

LogMute();
my %dests = LogPeek()->GetDestinations();
my $destCount = 0;
foreach my $channel (keys(%dests)) {
   $destCount += keys%{$dests{$channel}}
}
ok($destCount, 0);

ok(LogWarn("Should never see (warn) on log1."), 1);

ok(LogPush($log2) eq $log1);
ok(LogError("Sould never see (error) on log2."), 1);
LogUnmute();

ok(Log(VMware::Log::ID_ERROR, "Testing error through Log."), 1);
ok(<$r2>, (VMware::Log::ID_ERROR . ":  Testing error through Log.\n"));

LogMute();
ok(LogPop() eq $log2);
ok(LogInfo("Sould never see (info) on log1"), 1);
LogUnmute();

#
# If muting fails, we will see the "Should never see (warn)" line here,
# and then the "Should never see (info)" line.
#

ok(Log(VMware::Log::ID_WARN, "Testing warn through Log."), 1);
ok(<$r1>, (VMware::Log::ID_WARN . ":  Testing warn through Log.\n"));

ok(Log(VMware::Log::ID_INFO, "Testing info through Log."), 1);
ok(<$r1>, (VMware::Log::ID_INFO . ":  Testing info through Log.\n"));


#
# Tests for LogSet, both while muted and while not muted.
#

ok(LogPush($log2) eq $log1);
@priors = LogSet($log3);
ok(scalar(@priors), 2);
ok($priors[0] eq $log1);
ok($priors[1] eq $log2);

ok(LogDebug("Testing debug.", 0), 1);
ok(<$r3>, VMware::Log::ID_DEBUG . ":  Testing debug.\n");

ok(LogPush($log2) eq $log3);
LogMute();
my $old = LogSet($log1);
LogUnmute();
ok(scalar(@$old), 2);
ok($old->[0] eq $log3);
ok($old->[1] eq $log2);

ok(LogGetInfoCount(), 2);
ok(LogGetWarnCount(), 1);
ok(LogGetErrorCount(), 1);
ok(LogGetDebugCount(), 0);

LogPush($log2);
ok(LogGetWarnCount(), 1);
LogPop();
LogPush($log3);
ok(LogGetDebugCount(), 1);
LogPop();

ok(Log(VMware::Log::ID_INFO, "Testing log."), 1);
ok(<$r1>, VMware::Log::ID_INFO . ":  Testing log.\n");

ok(LogGetCount(VMware::Log::ID_INFO), 3);

close($w1) || die "Cannot close w1: $!\n";
close($w2) || die "Cannot close w2: $!\n";
close($w3) || die "Cannot close w2: $!\n";

ok(not defined <$r1>);
ok(not defined <$r2>);
ok(not defined <$r3>);

close($r1) || die "Cannot close r1: $!\n";
close($r2) || die "Cannot close r2: $!\n";
close($r3) || die "Cannot close reader: $!\n";

