#######################################################################
# Copyright (C) 2006 VMWare, Inc.
# All Rights Reserved
########################################################################
#
# File-based locking with PID
# Lock object destructor will delete the lock file -- this is
#   very convenient when programs exit due to unhandled exceptions.
#

import os
import errno
import logging


log = logging.getLogger('lock')

class Lock:
   def __init__(self, lockfile):
      self.lockfile = lockfile

   def __del__(self):
      #
      # Do not call log in destructor -- logging module may have unloaded by then
      #
      try:
         os.unlink(self.lockfile)
      except OSError:
         pass

   def Unlock(self):
      try:
         os.unlink(self.lockfile)
      except OSError:
         pass
      else:
         log.debug("Lock file %s deleted" % (self.lockfile))

   def CreateLockFile(self, pid, mode):
      """ Create the lock file and write the pid """
      try:
         lockfd = os.open(self.lockfile, os.O_EXCL|os.O_CREAT|os.O_WRONLY, mode)
      except OSError, e:
         if e.errno == errno.EEXIST:
            return 0
         else:
            raise
      else:
         os.write(lockfd, str(pid))
         os.close(lockfd)
         log.debug("Lock file %s created with PID %s" % (self.lockfile, pid))
         return 1
      
   def ReadPID(self):
      """ Read the PID from the lock file.
      Returns None if the file can't be opened or its contents are nonnumeric.
      """
      try:
         lock = open(self.lockfile, 'r')
         pid = int(lock.readline())
      except ValueError:
         log.debug('Non-numeric data in lock file %s' % (self.lockfile))
         return None
      except EnvironmentError, (enum, msg):
         log.debug('Ignoring error: %s (%d)' % (msg, enum) + \
                   ' while opening / reading %s' % (self.lockfile))
         return None
      else:
         return pid
      
   def Lock(self, mypid, mode=0644):
      """ Create the lock file.  Remove old lockfile if it has stale PID.
      LockError   - raised if lockfile owned by active PID
      """
      while not self.CreateLockFile(mypid, mode):
         pid = self.ReadPID()
         if not pid:
            log.debug('Bad data in lock file %s; deleting lock.' % (self.lockfile))
            self.Unlock()
            continue
         
         #
         # There is no signal 0.  Kill -0 checks to see if the process can
         # receive signals, ie is "active".  Zombied processes are inactive.  A process
         # will not become inactive and then later become active again, unless PIDs
         # are recycled.  Success (no exception) means the process is active.
         #
         try:
            os.kill(pid, 0)
         except OSError, (enum, msg):
            if enum == errno.ESRCH:
               log.debug('Process %d no longer seems to exist; deleting lock.' % (pid))
               self.Unlock()
               continue
            else:
               raise
         else:
            msg = 'Lock file %s is owned by active process %d' % (self.lockfile, pid)
            raise LockError(pid, msg)


class LockError(Exception):
   def __init__(self, pid, msg):
      self.pid = pid
      self.msg = msg
      self.args = (pid, msg)
