# -*- coding: UTF-8 -*-

"""
XXX

Active Directory uses non-standard ports for its Admin Server
The value may change again in the future, and the value for the
KDC may change as well.

In case Microsoft changes, change the ports in SetActiveDirectoryDomain
"""

import os
import sys
import re
import string
from vmware import Utility

class KerberosManager:
   krb5conf = '/etc/krb5.conf'
   get_header = re.compile(r'^\s*\[(\w+)\]\s*$')
   get_node = re.compile(r'^\s*([^\s]+)\s*=\s*\{\s*$')
   get_key_value = re.compile(r'^\s*([^\s]+)\s*=\s*([^{]*)$')
   is_node_end = re.compile(r'\s*}\s*$')

   krbconf = '/etc/krb.conf'
   krbconf_parser = re.compile(r'^\s*([^\s]+)\s+([^:\s]+):?(\d+)?\s*(.*)$')

   krbrealms = '/etc/krb.realms'
   krbrealms_parser = re.compile(r'([^\s]+)\s+(.*)$')

   # These should be the same across any instances of this Manager
   # as a change could be made to only one instance, but should the
   # config change, we wouldnt know about it and it would be lost
   config = { }
   servers = [ ]
   realms = [ ]

   """
   Constructor

   Arguments:
      kerb5conf: the krb5.conf file path
      kerbconf: the krb.conf file path
      kerbrealms: the krb.realms file path

   Returns:
      None

   Exceptions:
      None
   """
   def __init__(self, kerb5conf=krb5conf, kerbconf = krbconf, kerbrealms = krbrealms):
      self.krb5conf = kerb5conf
      self.krbconf = kerbconf
      self.krbrealms = kerbrealms
      self.Parse()
      self.dirty = False

   """
   Reads the krb5.conf file

   This method is responsible for reading the configuration
   tree for the krb5.conf file.  It is a recursive tree traversal
   method.  The tree is special in that the top level is different
   from the nodes and leaves, as such the code sees some repetition.
   Feel free to make this code better if possible.

   Arguments:
      stream: a stream to print to
      config: the dict structure which holds the configuration data

   Returns:
      None

   Exceptions:
      None
   """
   def __ParseTree(self, stream, config):
      read = stream.readline()
      current_header = None

      while read:
         if not (read.isspace() or read == '' or read.startswith('#')):
            read = read.split('\n')[0]

            if self.get_node.match(read):
               node = self.get_node.match(read).group(1)
               config[current_header][node] = { }
               self.__ParseTree(stream, config[current_header][node])
            elif self.is_node_end.match(read):
               return
            elif self.get_header.match(read):
               current_header = self.get_header.match(read).group(1)
               config[current_header] = { }
            elif self.get_key_value.match(read):
               matches = self.get_key_value.match(read)
               if current_header:
                  config[current_header][matches.group(1)] = matches.group(2)
               else:
                  config[matches.group(1)] = matches.group(2)

         read = stream.readline()

      return

   """
   Parse the configuration files for Kerberos.

   This method will parse three files:
      /etc/krb5.conf
      /etc/krb.conf
      /etc/krb.realms
   These three files define the kerberos client settings.

   Arguments:
      None

   Returns:
      None

   Exceptions:
      IOError
   """
   def Parse(self):
      # Parse Stage 1 (/etc/krb5.conf)

      try: 
          stream = open(self.krb5conf, 'r')
      except IOError:
         return  # missing files. Don't bother reading in anything else

      self.__ParseTree(stream, self.config)
      stream.close()

      # Parse Stage 2 (/etc/krb.conf)
      
      file = Utility.read_file(self.krb5conf)
      for line in file:
         matches = self.krbconf_parser.match(line)

         if matches is not None:
            if matches.group(4) != '':
               server = (matches.group(2), matches.group(3), True)
            else:
               server = (matches.group(2), matches.group(3), False)

            self.servers.append((matches.group(1), server))
      
      # Parse Stage 3 (/etc/krb.realms)

      file = Utility.read_file(self.krbrealms)
      for line in file:
         matches = self.krbrealms_parser.match(line)

         if matches is not None:
            self.realms.append((matches.group(1), matches.group(2)))
      
   """
   Write the configuration files to disk

   This method will rewrite the files back to the disk, however,
   it does not store comments or order.

   Arguments:
      None

   Returns:
      None

   Exceptions:
      None
   """
   def WriteConfig(self):
      stream = open(self.krb5conf, 'w')
      self.__WriteConfig(stream)
      stream.close()

      stream = open(self.krbconf, 'w')
      self.__WriteKerberosConfig(stream)
      stream.close()

      stream = open(self.krbrealms, 'w')
      self.__WriteRealmsConfig(stream)
      stream.close()

      self.dirty = False

   """
   Writes the configuration files to stdout

   This method will 'print' the files to stdout, however,
   it does not store comments or order.

   Arguments:
      None

   Returns:
      None

   Exceptions:
      None
   """
   def PrintConfig(self):
      stream = sys.stdout

      stream.write('%s\n' % self.krb5conf)
      self.__WriteConfig(stream)
      stream.write('\n')

      stream.write('%s\n' % self.krbconf)
      self.__WriteKerberosConfig(stream)
      stream.write('\n')

      stream.write('%s\n' % self.krbrealms)
      self.__WriteRealmsConfig(stream)

   """
   Writes the krb5.conf file

   This method is responsible for rewriting the configuration
   tree for the krb5.conf file.  It is a recursive tree traversal
   method.  The tree is special in that the top level is different
   from the nodes and leaves, as such the code sees some repetition.
   Feel free to make this code better if possible.

   Arguments:
      stream: a stream to write to
      config: dict containing values
      level: how far in the tree are we (needed for formatting)

   Returns:
      None

   Exceptions:
      None
   """
   def __WriteTree(self, stream, config, level):
      if level == 0:
         keys = config.keys()
         keys.sort()
         for key in keys:
            stream.write('[%s]\n' % key)

            for item in config[key]:
               if isinstance(config[key][item], dict):
                  stream.write(Utility.CreateIndent(level) + '%s = {\n' % item)
                  self.__WriteTree(stream, config[key][item], level + 1)
                  stream.write(Utility.CreateIndent(level) + '}\n')
               else:
                  stream.write(Utility.CreateIndent(level) + '%s = %s\n' % (item, config[key][item]))

            stream.write('\n')
      else:
         keys = config.keys()
         keys.sort()
         for item in keys:
            if isinstance(config[item], dict):
               stream.write(Utility.CreateIndent(level) + '%s = {\n' % item)
               self.__WriteTree(stream, config[item], level + 1)
               stream.write(Utility.CreateIndent(level) + '}\n')
            else:
               stream.write(Utility.CreateIndent(level) + '%s = %s\n' % (item, config[item]))
   
   def __WriteConfig(self, stream):
      stream.write('# Autogenerated by esxcfg-auth\n\n')
      self.__WriteTree(stream, self.config, 0)
   
   def __WriteKerberosConfig(self, stream):
      stream.write('# Autogenerated by esxcfg-auth\n\n')

      for (domain, server) in self.servers:
         if server[1] is None:
            address = server[0]
         else:
            address = ':'.join([server[0], server[1]])

         if server[2] is True:
            stream.write('%s\t%s\tadmin server\n' % (domain, address))
         else:
            stream.write('%s\t%s\n' % (domain, address))
 
   def __WriteRealmsConfig(self, stream):
      stream.write('# Autogenerated by esxcfg-auth\n\n')

      for tuple in self.realms:
         stream.write('%s\t%s\n' % tuple)

   def Reset(self):
      self.dirty = True
      self.config = { }
      self.servers = [ ]
      self.realms = [ ]

   def SetKerberosRealm(self, value):
      if not self.config.has_key('libdefaults'):
         self.config['libdefaults'] = { }

      if not self.config['libdefaults'].has_key('default_realm'):
         self.config['libdefaults']['default_realm'] = ''
      
      self.config['libdefaults']['default_realm'] = value.upper()

      if not self.config.has_key('domain_realm'):
         self.config['domain_realm'] = { }

      self.config['domain_realm'][''.join(['.',value.lower()])] = value.upper()
      self.config['domain_realm'][value.lower()] = value.upper()

      if not self.config.has_key('realms'):
         self.config['realms'] = { }

      self.config['realms'][value.upper()] = \
            {
              'kdc' : 'kdc:88' ,
              'admin_server' : 'admin_server:749' ,
              'default_domain' : value
            }

   def SetKerberosDistributionCenter(self, value):
      self.dirty = True
      if not self.config.has_key('realms'):
         self.config['realms'] = { }
      
      if not self.config.has_key('libdefaults'):
         self.config['libdefaults'] = { }

      if not self.config['libdefaults'].has_key('default_realm'):
         self.config['libdefaults']['default_realm'] = ''

      if not self.config['realms'].has_key(self.config['libdefaults']['default_realm']):
         self.config['realms'][self.config['libdefaults']['default_realm']] = { }

      self.config['realms'][self.config['libdefaults']['default_realm']]['kdc'] = \
            ':'.join([value, '88'])

   def SetKerberosAdminServer(self, value):
      self.dirty = True
      if not self.config.has_key('realms'):
         self.config['realms'] = { }

      if not self.config.has_key('libdefaults'):
         self.config['libdefaults'] = { }

      if not self.config['libdefaults'].has_key('default_realm'):
         self.config['libdefaults']['default_realm'] = ''

      if not self.config['realms'].has_key(self.config['libdefaults']['default_realm']):
         self.config['realms'][self.config['libdefaults']['default_realm']] = { }

      self.config['realms'][self.config['libdefaults']['default_realm']]['admin_server'] = \
            ':'.join([value, '749'])

   def SetActiveDirectoryDomain(self, value):
      self.dirty = True
      if not self.config.has_key('realms'):
         self.config['realms'] = { }

      if not self.config.has_key('libdefaults'):
         self.config['libdefaults'] = { }
      
      if not self.config['libdefaults'].has_key('default_realm'):
         self.config['libdefaults']['default_realm'] = ''

      if not self.config['realms'].has_key(self.config['libdefaults']['default_realm']):
         self.config['realms'][self.config['libdefaults']['default_realm']] = { }
      
      self.config['realms'][self.config['libdefaults']['default_realm']]['kdc'] = \
            ':'.join([value, '88'])
      self.config['realms'][self.config['libdefaults']['default_realm']]['admin_server'] = \
            ':'.join([value, '464'])
