# -*- coding: UTF-8 -*-

"""
XXX

There are two config files for ldap:
   /etc/ldap.conf
   /etc/openldap/ldap.conf

The contents of the files is similar \ (whitespace + comments)
This "Manager" holds two copies of the same configuration data
to be prepared for the case that the data is not actually the
same but only happens for these (currently used) keys
"""

import os
import sys
import re
from vmware import Utility

from Exceptions import InvalidCertificateException
from Exceptions import CertificatePermissionException

class LDAPManager:
   ldapconf = '/etc/ldap.conf'
   ldapconf_parser = re.compile(r'(\w+)\s+(.*)$')

   openldapconf = '/etc/openldap/ldap.conf'
   openldapconf_parser = re.compile(r'(\w+)\s+(.*)$')

   config = { } 
   open_config = { }

   def __init__(self, conf = ldapconf, open_conf = openldapconf):
      self.ldapconf = conf
      self.openldapconf = open_conf
      self.Parse()
      self.dirty = False



   """
   Parse the LDAP config files

   Currently, the files seem to be following a similar style,
   and we can use a single function to read the file into the Manager.

   Takes a stream, a parser and a config, parsing the stream
   using the parser, and writing to the config
   """
   def __Parse(self, fpath, parser, config):
      file = Utility.read_file(fpath)

      for line in file:
         matches = parser.match(line)

         if matches is not None:
            config[matches.group(1)] = matches.group(2)

   """
   Parse the configuration files for LDAP.

   This method will parse two files:
      /etc/ldap.conf
      /etc/openldap/ldap.conf
   The two files define the LDAP client settings.
   """
   def Parse(self):
      # Parse Stage 1 (/etc/ldap.conf)
      self.__Parse(self.ldapconf, self.ldapconf_parser, self.config)

      # Parse Stage 2 (/etc/openldap/ldap.conf)
      self.__Parse(self.openldapconf, self.openldapconf_parser, self.open_config)

   """
   Write the configuration files to disk

   This method will rewrite the files back to the disk, however,
   it does not store comments or order.
   """
   def WriteConfig(self):
      stream = open(self.ldapconf, 'w')
      self.__WriteConfig(stream)
      stream.close()

      stream = open(self.openldapconf, 'w')
      self.__WriteConfig(stream)
      stream.close()

      self.dirty = False

   """
   Writes the configuration files to stdout

   This method will 'print' the files to stdout, however,
   it does not store comments or order.
   """
   def PrintConfig(self):
      stream = sys.stdout

      stream.write('%s\n' % self.ldapconf)
      self.__WriteConfig(stream)

      stream.write('%s\n' % self.openldapconf)
      self.__WriteConfig(stream)

   """
   Writes the configuration file to a stream

   Currently both LDAP configuration files use a similar format, and so we can
   compress the writer to a single function.  In the future this may have to be
   changed.
   """
   def __WriteConfig(self, stream):
      stream.write('# Autogenerated by esxcfg-auth\n\n')

      keys = self.config.keys()
      keys.sort()
      for key in keys:
         stream.write('%s %s\n' % (key, self.config[key]))

   def EnableSSL(self):
      self.dirty = True
      self.config['ssl'] = "on"
      self.open_config['ssl'] = "on"

   def DisableSSL(self):
      self.dirty = True
      self.config['ssl'] = "no"
      self.open_config['ssl'] = "no"

   def EnableTLS(self):
      self.dirty = True
      self.config['ssl'] = "start_tls"
      self.open_config['ssl'] = "start_tls"

      # Ensure that the permission bits on the CA cert allow reading
      if self.open_config.has_key('TLS_CACERT'):
         certificate = self.open_config['TLS_CACERT']
      elif self.open_config.has_key('tls_cacert'):
         certificate = self.open_config['tls_cacert']
      else:
         return

      if not os.path.exists(certificate):
         raise InvalidCertificateException

      try:
         os.chmod(certificate, 0644)
      except:
         raise CertificatePermissionException

   def DisableTLS(self):
      self.dirty = True
      self.config['ssl'] = "no"
      self.open_config['ssl'] = "no"

   def SetBaseDN(self, value):
      self.dirty = True
      self.config['base'] = value
      self.open_config['base'] = value

   def SetLDAPServer(self, value):
      self.dirty = True
      self.config['host'] = value
      self.open_config['host'] = value

   def EnableMD5(self):
      self.dirty = True
      self.config['pam_password'] = "md5"
      self.open_config['pam_password'] = "md5"

   def DisableMD5(self):
      self.dirty = True
      self.config['pam_password'] = "crypt"
      self.open_config['pam_password'] = "crypt"
