# -*- coding: UTF-8 -*-

import os
import sys
import re
from vmware import Utility

class NISManager:
   ypconf = '/etc/yp.conf'
   ypconfig_parser = re.compile(r'(\w+)\s([^\s]+)\s?(\w+)?\s?(.*)$')
   networkconf = '/etc/sysconfig/network'
   networkconf_parser = re.compile(r'\s*NISDOMAIN')

   network = []
   ypservers = [ ]
   ypdomains = [ ]
   broadcast = False

   nis_domain = None
   nis_server = None
   
   def __init__(self, file = ypconf):
      self.ypconf = file
      self.Parse()
      self.dirty = False
   

   """
   Parse the configuration files for NIS.

   This method will parse the following files:
      /etc/yp.conf
      /etc/sysconfig/network
      
   This file defines the NIS client settings.
   """
   def Parse(self):

      # load /etc/sysconfig/network, and filter out nis domain 
      # (nis domain from yp.conf is authoritative)
      file = Utility.read_file(self.networkconf)
      for line in file:
         matches = self.networkconf_parser.match(line)
         if matches is None:
            self.network.append(line)
         
      file = Utility.read_file(self.ypconf)
      for line in file:
         matches = self.ypconfig_parser.match(line)

         if matches is not None:
            if matches.group(1) == 'ypserver':
               self.ypservers.append(matches.group(2))
            elif matches.group(1) == 'domain':
               if matches.group(4) != '':
                  self.ypdomains.append((matches.group(2), matches.group(3), matches.group(4)))
               else:
                  self.ypdomains.append((matches.group(2), matches.group(3)))
            elif matches.group(1) == 'broadcast':
               self.broadcast = True
      
      

   """
   Write the configuration files to disk

   This method will rewrite the files to the disk, however,
   it does not store comments or order.
   """
   def WriteConfig(self):
      if self.nis_domain and self.nis_server:
         # check the uniqueness of new ypdomain
         f_uniq = True
         for ypdomain in self.ypdomains:
            if ypdomain[0] == self.nis_domain and \
                   ypdomain[1] == 'server' and \
                   ypdomain[2] == self.nis_server:
               f_uniq = False
               break;
         if f_uniq:
            self.ypdomains.insert(0, (self.nis_domain, 'server', self.nis_server))

      stream = open(self.ypconf, 'w')
      self.__WriteConfig(stream)
      stream.close()

      stream = open(self.networkconf, 'w')
      for line in self.network:
         stream.write('%s\n' % line)
      if self.DefaultNISDomain() is not None:
         stream.write('NISDOMAIN=%s\n' % self.DefaultNISDomain())
      stream.close()

      self.dirty = False

   """
   Write the configuration files to stdout

   This method will 'print' the files to stdout, however,
   it does not store comments or order
   """
   def PrintConfig(self):
      stream = sys.stdout

      stream.write('%s\n' % self.ypconf)
      self.__WriteConfig(stream)

   def __WriteConfig(self, stream):
      stream.write('# Autogenerated by esxcfg-auth\n\n')

      for tuple in self.ypdomains:
         if len(tuple) is 3:
            stream.write('domain %s %s %s\n' % tuple)
         else:
            stream.write('domain %s %s\n' % tuple)

      for server in self.ypservers:
         stream.write('ypserver %s\n' % server)

      if self.broadcast is True:
         stream.write('broadcast\n')

   def NISDomain(self, value):
      self.dirty = True
      self.nis_domain = value

   def NISServer(self, value):
      self.dirty = True
      self.nis_server = value

   def DefaultNISDomain(self):
      if self.nis_domain is not None:
         return self.nis_domain
      elif len(self.ypdomains) > 0:
         return self.ypdomains[0][0]
      else:
         return None

   def Reset(self):
      self.dirty = True
      self.ypservers = [ ]
      self.ypdomains = [ ]
      self.broadcast = False

      self.nis_domain = None
      self.nis_server = None
