# -*- coding: UTF-8 -*-

import os
import sys
import string
import re
from vmware import Utility

class NSCDManager:
   nscd = '/etc/nscd.conf'
   nscd_parser = re.compile(r'^\s*([\w-]+)\s+([^\s]+)\s*(\w*)$')

   config = { }

   def __init__(self, file=nscd):
      self.nscd = file
      self.Parse()
      self.dirty = False


   """
   Parse the configuration file for NSCD

   This method will parse one file:
      /etc/nscd.conf
   This file defines the NSCD client settings.
   """
   def Parse(self):
      file = Utility.read_file(self.nscd)

      for line in file:
         matches = self.nscd_parser.match(line)

         if matches is not None:
            if matches.group(3) == '':
               # option : value
               self.config[matches.group(1)] = matches.group(2)
            else:
               # service : { option : value }
               if not self.config.has_key(matches.group(2)):
                  self.config[matches.group(2)] = { }

               self.config[matches.group(2)][matches.group(1)] = matches.group(3)

   """
   Write the configuration file to disk

   This method will rewrite the files back to the disk, however,
   it does not store comments or order.
   """
   def WriteConfig(self):
      stream = open(self.nscd, 'w')
      self.__WriteConfig(stream)
      stream.close()

      self.dirty = False

   """
   Writes the configuration files to stdout

   This method will 'print' th efiles to stdout, however,
   it does not store comments or order.
   """
   def PrintConfig(self):
      stream = sys.stdout

      stream.write('%s\n' % self.nscd)
      self.__WriteConfig(stream)

   def __WriteConfig(self, stream):
      stream.write('# Autogenerated by esxcfg-auth\n\n')

      services = self.config.keys()
      services.sort()

      for option in services:
         if not isinstance(self.config[option], dict):
            stream.write('\t%s\t%s\n' % (option, self.config[option]))

      stream.write('\n')

      for service in services:
         if isinstance(self.config[service], dict):
            options = self.config[service].keys()
            options.sort()
            for option in options:
               stream.write('\t%s\t%s\t%s\n' % (option, service, self.config[service][option]))
            stream.write('\n')
      
   """
   Set caching of a service

   This method will set a service's caching to on or off
   """
   def SetServiceCache(self, service, value):
      self.dirty = True
      if value:
         value = 'yes'
      elif not value:
         value = 'no'

      if not self.config.has_key(service):
         self.config[service] = { }
      
      self.config[service]['enable-cache'] = value
   
   """
   Enable the caching of data

   This method will enable the caching of login data
   for passwd, group, hosts
   """
   def EnableCache(self):
      self.dirty = True
      self.SetServiceCache('passwd', True)
      self.SetServiceCache('group', True)
      self.SetServiceCache('hosts', True)

   def DisableCache(self):
      self.dirty = True
      self.SetServiceCache('passwd', False)
      self.SetServiceCache('group', False)
      self.SetServiceCache('hosts', False)
