#######################################################################
# Copyright (C) 2005 VMWare, Inc.
# All Rights Reserved
########################################################################
#
# Descriptor.py
#     classes to read and write release descriptors and its components
#
# Terminology:
#   release  - A full set of RPMs like on the install CD.
#   patch    - A partial set of RPMs to patch an installed release.
#
##
# On XML Serialization:
#   XML is serialized into an object in the constructors, which is able
#   to take an Element instance, construct an object out of it, and
#   construct child nodes as needed, following the tree.
#
#   Our descriptor object tree is serialized into XML via the ToXml method
#   of each class.  ToXml is intended to either return a string XML representation
#   of the instance or return an Element or similar object, which allows recursive
#   serialization to XML via either printing strings or building an ElementTree.
#
#   See the Rpm, Rpmlist classes as an example.
##

##
# Coding style used:
#  1. Docstrings used for each class and public method def, unless its a very simple
#     method like Get*/Set*
#  2. Docstrings begin with one summary line, a blank line, then detail.  This format
#     is for compatibility with Python doc tools.
#  3. Docstrings come after each def.
#  4. No attempt to code for Python < 2.2
#  5. Public method names begin with Caps.  Private ones like _this.
##

import logging
import os
import time
from xml.parsers.expat import ExpatError
try:
    from elementtree.ElementTree import ElementTree, Element, iselement
except ImportError:
    from xml.etree.ElementTree import ElementTree, Element, iselement

from Rpm import Rpm
from Rpmlist import Rpmlist
from Filelist import Filelist
from UpgradePaths import UpgradePaths
from utils import *

# set up logging system
log = logging.getLogger("descriptor")


DESCVER = "2.0"       # Current version of descriptor


########################################################################
#
# Custom exception classes
#
class BadDescriptorError(Exception):
    def __init__(self, file, msg):
        self.msg = msg
        self.file = file
        Exception.__init__(self, file, msg)


########################################################################
#
# Class representing a release descriptor.
# - It can parse a descriptor file or create a new one from scratch.
# - Access methods to different fields.
# - API is independent of XML implementation.  XML functionality is
#   encapsulated mostly in _fromXml and _toXml.
#
# See ReleaseDescriptorFormat wiki for more information on the
#  descriptor fields themselves.
#
class Descriptor:
    """Creates, parses, modifies, writes release descriptors."""

    def __init__(self, file=None):
        """Create a blank descriptor or construct from an XML file.

        file:  a filename, file object, or none for blank descriptor.

        Throws:
        BadDescriptorError - XML syntax or bad descriptor fields
        """
        self._reset()

        if file:
            # If filename and not filehandle, open it
            if not hasattr(file, "read"):
                file = open(file, "rb")

            try:
                try:
                    tree = ElementTree(file=file)
                    self._fromXml(tree)
                except (ExpatError, AssertionError), e:
                    if hasattr( file, 'name' ):
                        name = file.name
                    elif hasattr( file, 'url' ):
                        #filehandles created by urllib don't have a .name
                        name = file.url
                    else:
                        name = str( file )
                    raise BadDescriptorError(name, str(e))
            finally:
                file.close()

    def AddFullRpmlist(self, dir):
        """ Create full list of rpms from directory. """
        if self.rpmlist:
            log.warn("Full rpmlist already exists, overwriting old one")
        self.rpmlist = Rpmlist(dir, tag="rpmlist")

    #
    # Get/Set methods for different RPM lists
    #
    def GetFullRpmlist(self):     return self.rpmlist
    def GetSkiplist(self):        return self.skiplist
    def GetNodeps(self):          return self.nodeps
    def GetPreinstall(self):      return self.preinstall
    def GetRemoves(self):         return self.removes
    def GetFullFileList(self):    return self.filelist

    def SetFullRpmlist(self, newlist):  self.rpmlist = newlist
    def SetSkiplist(self, newlist):     self.skiplist = newlist
    def SetNodeps(self, newlist):       self.nodeps = newlist
    def SetPreinstall(self, newlist):   self.preinstall = newlist
    def SetRemoves(self, newlist):      self.removes = newlist
    def SetFullFileList(self, newlist): self.filelist = newlist

    def GetUpgradePaths(self):
        return self.deps

    def SetUpgradePaths(self, deps):
        if not deps:
            self.deps = UpgradePaths()
        else:
            self.deps = deps

    def GetVendor(self):
        return self.vendor

    def SetVendor(self, newstr):
        self.vendor = newstr

    def GetProduct(self):
        return self.product

    def SetProduct(self, newstr):
        self.product = newstr

    def GetRelease(self):
        """Returns the value of <release>, which is the bundle ID."""
        return self.release

    def SetRelease(self, newstr):
        """Sets the value of <release>, which is the bundle ID."""
        self.release = newstr

    def GetReleasedate(self):
        return self.releasedate

    def SetReleasedate(self, newstr):
        self.releasedate = newstr

    def GetDescription(self):
        return self.description

    def SetDescription(self, newstr):
        self.description = newstr

    def GetSummary(self):
        return self.summary

    def SetSummary(self, newstr):
        self.summary = newstr

    def GetContact(self):
        return self.contact

    def SetContact(self, newstr):
        self.contact = newstr

    def GetURL(self):
        return self.url

    def GetType(self):
        return self.type

    def SetType(self, newstr):
        assert newstr in ('Full', 'Partial'), "Illegal type %s" % (newstr)
        self.type = newstr

    def GetBundleType(self):
        return self.bundleType

    def SetBundleType(self, value):
        assert type(value) == str, 'Illegal bundle type: %s' % value
        value = value.strip.title()
        assert value in ('Patch', 'Rollup', 'Update'), \
                        'Illegal bundle type: %s' % value
        self.bundleType = value

    def GetName(self):
        """Returns the value of <name>, which is currently visor specific."""
        return self.name

    def SetName(self, value):
        """Sets the value of <name>, which is currently visor specific
        (e.g., firmware, tools, viclient).
        """
        assert type(value) == str
        self.name = value.strip()

    def GetVersion(self):
        return self.version

    def SetVersion(self, value):
        assert type(value) == str
        self.version = value

    def GetRel(self):
        """Returns the value of <rel>, which is the build number."""
        return self.rel

    def SetRel(self, value):
        """Sets the value of <rel>, which is the build number."""
        assert type(value) == str
        self.rel = value

    def IsTypeFull(self):     return self.type == 'Full'
    def IsTypePartial(self):  return self.type == 'Partial'

    def GetUTCTimestamp(self):
        return self.UTCtimestamp

    def SetUTCTimestamp(self, utc_secs=None):
        if not utc_secs:
            utc_secs = time.mktime(time.localtime()) + time.timezone
        self.UTCtimestamp = utc_secs

    def SetPkgBldnum(self, pkgbld):
        self.pkgbld = pkgbld

    def GetPkgBldnum(self):
        return self.pkgbld

    def _fromXml(self, tree):
        """Fill fields from ElementTree tree."""
        root = tree.getroot()
        self.desc_version = root.get('version', DESCVER).strip()
        try:
            self.UTCtimestamp = int(root.get('UTCtimestamp'))
        except:
            self.UTCtimestamp = 0
        self.vendor = root.findtext('vendor', '').strip()
        self.product = root.findtext('product', '').strip()
        self.name = root.findtext('name', '').strip()
        self.version = root.findtext('version', '').strip()
        self.release = root.findtext('release', '').strip()
        self.releasedate = root.findtext('releasedate', '').strip()
        self.summary = root.findtext('summary', '').strip()
        self.description = root.findtext('description', '').strip()
        self.contact = root.findtext('contact', '').strip()
        self.url = root.findtext('URL', '').strip()
        self.type = root.findtext('type', '').strip()
        self.bundleType = root.findtext('bundleType', 'Patch').strip()
        self.rel = root.findtext('rel', '').strip()
        self.pkgbld = root.findtext('pkgbld')

        rpmlistElem = root.find('rpmlist')
        if rpmlistElem:
            self.rpmlist = Rpmlist(rpmlistElem)

        removesElem = root.find('removes')
        if removesElem:
            self.removes = Rpmlist(removesElem, tag='removes')

        skiplistElem = root.find('skiplist')
        if skiplistElem:
            self.skiplist = Rpmlist(skiplistElem, tag='skiplist')

        nodepsElem = root.find('nodeps')
        if nodepsElem:
            self.nodeps = Rpmlist(nodepsElem, tag='nodeps')

        preinstElem = root.find('preinstall')
        if preinstElem:
            self.preinstall = Rpmlist(preinstElem, tag='preinstall')

        #
        # desc v1.0 has <upgradepaths> but not <dependencies>
        #   it will be converted to an UpgradePath instance
        #
        if self.desc_version < '2.0':
            pathsElem = root.find('upgradepaths')
            if pathsElem:
                self.pathsElem = pathsElem
            self.deps = UpgradePaths(None, oldpaths=self.pathsElem)
        else:
            #
            # COS uses 'dependencies' tag name, but Visor uses 'upgradepaths'.
            #
            depsElem = root.find('dependencies')
            if not depsElem:
                depsElem = root.find('upgradepaths')

            if depsElem:
                try:
                    self.deps = UpgradePaths(depsElem)
                except ValueError, msg:
                    raise BadDescriptorError("", msg)

        filelistElem = root.find('filelist')
        if filelistElem:
            self.filelist = Filelist(filelistElem)

    def _toXml(self):
        #
        # This is a private method because users don't need to
        # extract XML Elements.  We'd like to keep the rough
        # structure of having a ToXml() func that serializes
        # to XML Elements, as with other classes here.
        #
        root = Element('descriptor', version=self.desc_version,
                       UTCtimestamp="%d" % (self.UTCtimestamp))

        SetElem(root, 'vendor', self.vendor)
        SetElem(root, 'product', self.product)
        SetElem(root, 'name', self.name)
        SetElem(root, 'version', self.version)
        SetElem(root, 'release', self.release)
        SetElem(root, 'releasedate', self.releasedate)
        SetElem(root, 'summary', self.summary)
        SetElem(root, 'description', self.description)
        SetElem(root, 'contact', self.contact)
        SetElem(root, 'URL', self.url)
        SetElem(root, 'type', self.type)
        SetElem(root, 'bundleType', self.bundleType)
        SetElem(root, 'rel', self.rel)

        #
        # Only output <dependencies> for 2.0 descriptors
        # But always output <upgradepaths> to preserve
        # 1.0 desc's and for upgrade bundles
        # TODO: Create a real version comparator
        #
        if self.desc_version >= "2.0":
            root.append(self.deps.ToXml())

        if self.pathsElem:
            root.append(self.pathsElem)

        if self.preinstall:
            root.append(self.preinstall.ToXml())

        # Add the full list of RPMs
        if self.rpmlist:
            root.append(self.rpmlist.ToXml())

        # The RPMs to be removed during an upgrade
        if self.removes:
            root.append(self.removes.ToXml())

        # The RPMs to be skipped during installs
        if self.skiplist:
            root.append(self.skiplist.ToXml())

        # These must be installed --nodeps
        if self.nodeps:
            root.append(self.nodeps.ToXml())

        if self.filelist:
            root.append(self.filelist.ToXml())

        if self.pkgbld:
            SetElem(root, 'pkgbld', self.pkgbld)

        return root

    def Write(self, filename):
        """Write descriptor as XML to filename."""
        tree = ElementTree(element = self._toXml())
        IndentElementTree(tree.getroot())
        tree.write(filename)

    def _reset(self):
        self.desc_version = DESCVER
        self.vendor = ""
        self.product = ""
        self.release = 'none'
        self.releasedate = ""
        self.summary = ""
        self.description = ""
        self.contact = ""
        self.type = ""
        self.rpmlist = None
        self.deps = UpgradePaths()
        self.pathsElem = None
        self.preinstall = None
        self.skiplist = None
        self.nodeps = None
        self.url = ""
        self.removes = None
        self.UTCtimestamp = 0
        self.bundleType = 'Patch'
        self.filelist = None
        self.rel = ""
        self.name = ""
        self.version = ""
        self.pkgbld = None
