'''
Implements File class, representing a <file> element in a descriptor.xml
<filelist>.
'''
try:
    from elementtree.ElementTree import Element, iselement, tostring
except ImportError:
    from xml.etree.ElementTree import Element, iselement, tostring

class File:
    '''File class, representing <file> element, with optional attribute 'type'
       being one of 'install-script' or 'tar-gz-install' and value (CDATA)
       specifying a file name.
       E.g. <file type='install-script'>install.sh</file>'''
    def __init__(self, filename, filetype=''):
        '''File constructor.
           First element must be either a filename or an ElementTree element.
           If it is an ElementTree element, the filetype argument is discarded.
           May raise TypeError on invalid parameters.'''
        if iselement(filename):
            self.SetFileName(filename.text)
            self.SetFileType(filename.get('type', ''))
        elif type(filename) == str:
            if type(filetype) != str:
                raise("Value for 'filetype' parameter ('%s') is not a string" %
                      filetype)
            self.filename = filename
            self.filetype = filetype
        else:
            raise TypeError("Value for 'filename' parameter ('%s') is neither "
                            "a string nor an ElementTree element" % filename)

    def GetFileName(self):
        return self.filename

    def SetFileName(self, value):
        if type(value) != str:
            raise TypeError('Argument must be string, not %s' % type(value))
        self.filename = value

    def GetFileType(self):
        return self.filetype

    def SetFileType(self, value):
        if type(value) != str:
            raise TypeError('Argument must be string, not %s' % type(value))
        value = value.strip().lower()
        if value not in ('', 'install-script', 'tar-gz-install'):
            raise TypeError("Argument must be one of '', 'install-script' or "
                            "'tar-gz-install', not '%s'" % value)
        self.filetype = value

    def ToXml(self):
        elem = Element('file')
        elem.text = self.filename
        if self.filetype:
            elem.set('type', self.filetype)
        return elem

    def __str__(self):
        return tostring(self.ToXml())
