'''
Implements Filelist class, representing a <filelist> element in descriptor.xml.
'''
try:
    from elementtree.ElementTree import Element, iselement, tostring
except ImportError:
    from xml.etree.ElementTree import Element, iselement, tostring

from File import File

class Filelist:
    '''Filelist class, representing <filelist> element with zero or more
       <file> child elements.'''
    def __init__(self, filelist=None):
        '''Filelist constructor.
           First element may be one of:
               1.  An ElemenTree element, in which case it is taken to be an
                   element read from a <filelist> tag.
               2.  A list or tuple, where each value is a valid parameter or
                   parameters for the File() class constructor.
           May raise TypeError on invalid parameters.'''
        if iselement(filelist) or type(filelist) in (list, tuple):
            self.SetFiles(filelist)
        elif filelist is None:
            self.files = list()
        else:
            raise TypeError("Value for 'filelist' parameter ('%s') is neither "
                            "a list, tuple, or ElementTree element" % filelist)

    def AddFile(self, value):
        if isinstance(value, File):
            self.files.append(value)
        elif type(value) == dict:
            self.files.append(File(**value))
        elif type(value) in (list, tuple):
            self.files.append(File(*value))
        else:
            self.files.append(File(value))

    def SetFiles(self, value):
        if type(value) in (list, tuple):
            self.files = list()
            for x in value:
                self.AddFile(x)
        elif iselement(value):
            self.files = [File(x) for x in value]

    def GetFiles(self):
        return self.files

    def GetFileNames(self):
        return [x.GetFileName() for x in self.files]

    def ToXml(self):
        elem = Element('filelist')
        for x in self.files:
            elem.append(x.ToXml())
        return elem

    def __str__(self):
        return tostring(self.ToXml())
