#######################################################################
# Copyright (C) 2005 VMWare, Inc.
# All Rights Reserved
########################################################################
#
# Rpm.py
#   module defining the Rpm class
#
# Testing
#   see testdesc.py
#
import logging
import os
import rpm
try:
    from elementtree.ElementTree import ElementTree, Element, iselement
except ImportError:
    from xml.etree.ElementTree import ElementTree, Element, iselement

from utils import *

# set up logging system
log = logging.getLogger("rpm")


########################################################################
#
# Class to model an Rpm package or Rpm file.
#   - serializes from/to XML
#   - models just package name or n-v-r.a.rpm file
#   - uses the rpm library to verify package signature and other info
#   - comparison of packages
#
# Used: Multiple lists in a release descriptor have Rpm elements,
#    including <rpmlist>, <skiplist>, <preinstall>.  If we have
#    incremental patches then we might have a <difflist>.
#
#
class Rpm:
    """Models an Rpm package in the release descriptor.

    When fully populated (n,v,r,a) it represents an Rpm package in the release
    descriptor and in the bundle.
    It can also be used to represent a package in a manifest list, with or without
    version info.
    It can handle package filenames that don't fit in with the n-v-r.a model.

    Two independent comparison functions IsSame(), Cmp(), could be combined.
    IsSame() compares name and arch while Cmp() compares versions only.
    rpma == rpmb  is the same as rpma.IsSame(rpmb) and rpma.Cmp(rpmb) == 0
    rpma != rpmb  is (not rpma == rpmb)
    The >, >=, <, <= operators are also supported.
    """

    def __init__(self, pkg=None, name=None, ts=None, hdr=None):
        """Create instance from Element, package file, or with name info only

        newrpm = Rpm(myElement)
        newrpm = Rpm('emacs-21.9.4-foo.i386.rpm')   # must be a string
        newrpm = Rpm(name='emacs')                  # must be a string
        newrpm = Rpm(name='emacs.i386')             #  can be name.arch
        newrpm = Rpm(hdr = h)                   # for working with the RpmDB

        A TypeError is raised if one of the valid input combinations is not met.
        """
        self.arch = ''
        self.epoch = None
        self.ver  = ''
        self.rel  = ''
        self.md5  = ''
        self.path = ''      # Source path for package
        self.file = ''      # if filename doesn't fit n-v-r.a.rpm scheme

        # pkg is an rpm Element, parse it and release it
        if iselement(pkg):
            assert pkg.tag == 'rpm', "Cannot create Rpm instance with non-rpm element"
            assert pkg.text, "Empty rpm Xml element"
            self.name = pkg.text
            self.epoch = pkg.get('epoch', None)
            self.ver  = pkg.get('ver', '')
            self.rel  = pkg.get('rel', '')
            self.arch = pkg.get('arch', '')
            self.md5  = pkg.get('md5', '')
            self.path = pkg.get('path', '')
            self.file = pkg.get('file', '')
            pkg.clear()

        # pkg is an rpm package filename
        elif IsStringLike(pkg):
            self.SetFromHeader(pkg, ts)

        # hdr is an rpm Header instance
        elif hdr:
            self.name = hdr['name']
            self.ver  = hdr['version']
            self.rel  = hdr['release']
            self.arch = hdr['arch']
            self.epoch = str(hdr['epoch'] or '')

        # pkg=None, create Rpm from name (name.arch)
        elif IsStringLike(name):
            n, a = ParseNAString(name)
            self.name = n
            if a:  self.arch = a

        else:
            raise TypeError, "Rpm(pkg=%s, name=%s, ts=%s, hdr=%s) invalid" % (pkg, name,
                                                                              ts, hdr)

    def __repr__(self):
        return 'Rpm(name=%s,rel=%s,ver=%s,arch=%s)' % (self.name,
                                                       self.rel,
                                                       self.ver,
                                                       self.arch)

    def SetFromHeader(self, file, ts=None):
        """Set n-v-r-a fields from reading header of RPM file.

        file:  valid pathname to package
        ts:    use transaction set if supplied.  If not, use a default
               which turns off header verification.

        Passing in a ts with custom verification flags is a good idea.
        """

        if not ts:
            ts = rpm.TransactionSet()
            ts.setVSFlags(-1)
        h = ReadRpmHeader(ts, file)

        self.name = h['name']
        self.ver  = h['version']
        self.rel  = h['release']
        self.arch = h['arch']
        self.epoch = str(h['epoch'] or '')

        if os.path.basename(file) != self.GetBasename():
            self.file = os.path.basename(file)

        self.path = os.path.dirname(file)

    def SetName(self, name):
        self.name = name

    def SetVer(self, ver):
        self.ver = ver

    def SetRelease(self, rel):
        self.rel = rel

    def SetArch(self, arch):
        self.arch = arch

    def SetMD5(self, md5):
        self.md5 = md5

    def GetName(self):
        return self.name

    def GetVer(self):
        return self.ver

    def GetRelease(self):
        return self.rel

    def GetNameArch(self):
        return '%s.%s' % (self.name, self.arch)

    def GetNVR(self):
        return '%s-%s-%s' % (self.name, self.ver, self.rel)

    def GetPath(self):
        return self.path

    def GetMD5(self):
        return self.md5

    def GetBasename(self):
        """ Returns a package filename based on name-ver-rel.arch.rpm """
        if self.file:
            return self.file
        else:
            return '%s-%s-%s.%s.rpm' % (self.name, self.ver, self.rel, self.arch)

    def IsSame(self, other):
        """ Returns true if this Rpm has the same name and arch as other Rpm.

        An empty arch field in either package means arch is not compared.
        """
        assert hasattr(other, 'SetArch'), "Attempt to compare against non-Rpm object %s" % \
               (other.__class__)

        sameArch = True
        if self.arch and other.arch:
            sameArch = (self.arch == other.arch)

        return (self.name == other.name) and sameArch

    def Cmp(self, other):
        """ Compares the version of this package against the other.

        Return value:
        0       Same version, release (or neither field defined)
        1       This package is newer
        -1      This package is older
        This means Cmp() can be used as a sort function.

        (Epoch, Version, Release) will be compared in that order, using
        the labelCompare algorithm in rpmlib.  If ver/rel is blank in
        at least one of the Rpm objects, then it is not compared.
        Epoch is not compared if its none since 3.0 esxupdate did not
        express the epoch value.
        """
        assert hasattr(other, 'SetArch'), "Attempt to compare " \
               "against non-Rpm object %s" % (other.__class__)

        v1 = v2 = ''
        r1 = r2 = ''
        e1 = e2 = ''
        if self.ver and other.ver:
            v1 = self.ver
            v2 = other.ver
        if self.rel and other.rel:
            r1 = self.rel
            r2 = other.rel
        if self.epoch is not None and other.epoch is not None:
            e1 = self.epoch
            e2 = other.epoch

        return rpm.labelCompare( (e1, v1, r1), (e2, v2, r2) )

    def __eq__(self, other):
        return self.IsSame(other) and (self.Cmp(other) == 0)

    def __ne__(self, other): return not self.__eq__(other)

    def __cmp__(self, other): return self.Cmp(other)

    def ToXml(self):
        """Serializes to XML and returns as Element instance.

        The path attribute is not written out.  Path is intended only for internal
        use building repositories.
        """

        elem = Element('rpm')
        elem.text = self.name
        if self.epoch is not None: elem.set('epoch', self.epoch)
        if self.ver: elem.set('ver', self.ver)
        if self.rel: elem.set('rel', self.rel)
        if self.arch: elem.set('arch', self.arch)
        if self.md5: elem.set('md5', self.md5)
        if self.file: elem.set('file', self.file)
        return elem
