#######################################################################
# Copyright (C) 2005,2006 VMware, Inc.
# All Rights Reserved
########################################################################
#
# UpgradePaths.py
#   module defining the UpgradePaths class
#
# Testing
#   see test/test_UpgradePaths.py
#
import logging
import copy
import re
try:
    from elementtree.ElementTree import Element, iselement
except ImportError:
    from xml.etree.ElementTree import Element, iselement

from utils import GetIDLists, IDListElem, GetBools, BoolDictElem, SetElem

#
# Constants / globals
#
log = logging.getLogger("paths")

#
# Use the most conservative defaults - we must reboot, power off VMs, etc.
# Also, True/1 must be the conservative setting.
#
INSTALLFLAGS = {'MModeRequired' : 1,   # Maintenance Mode before install
                'RebootRequired': 1,   # reboot required after install
                'HostdRestart'  : 1,   # Hostd must be restarted
                }


########################################################################
#
# A collection of upgrade dependencies and installation flags, including
# dependencies on other patches, obsolescence of older patches, and
# specification of the required base release of ESX.
#
# See the ReleaseDescriptorFormat wiki for more details.
#
class UpgradePaths:
    """ A specification of inter-bundle dependencies and installation
    flags. Members:
      obsoletedIDs   (list)      Bundle IDs obsoleted by this one
      requiredIDs    (list)      These bundle IDs must be installed first
      conflictIDs    (list)      These bundle IDs cannot be installed
      sysStateDeps   (dict of dict) system state deps for each bundle ID, eg
                                  {'ESX-11': {'rebooted': True},
                                   'ESX-22': {'hostdRestarted': False},
                                   }
                                 If there are no system state deps then
                                 there will be no key/item.
      flags          (dict)      Installation flags
    """

    def __init__(self, elem=None, oldpaths=None):
        """ Create an instance from a <deps> Element node
        oldpaths: convert from older <upgradepaths> Element node.
        """

        self.obsoletedIDs = []
        self.requiredIDs = []
        self.conflictIDs = []
        self.sysStateDeps = {}
        self.flags = copy.copy(INSTALLFLAGS)
        self._tag = 'dependencies'

        if iselement(elem):
            #
            # Visor descriptors will use 'upgradepaths' tag name instead of
            # 'dependencies'.  Need to store the tag name so that we can
            # return the same one in ToXml().
            #
            self._tag = elem.tag
            lists, self.sysStateDeps = GetIDLists(elem)
            if 'obsoletes' in lists:
                self.obsoletedIDs = lists['obsoletes'][:]
            if 'requires' in lists:
                self.requiredIDs = lists['requires'][:]
            if 'conflicts' in lists:
                self.conflictIDs = lists['conflicts'][:]
            xmlflags = GetBools(elem)
            self.flags.update(xmlflags)
            elem.clear()
        elif oldpaths:
            assert oldpaths.tag == 'upgradepaths', \
                   "Expected <upgradepaths>, got <%s>" % (elem.tag)
            self._convertPaths(oldpaths)

    def _convertPaths(self, elem):
        """ Convert old style <path> elements to requiredIDs and flags.
        Only the first <path> is converted.  Wildcards are inserted
        to convert build #'s or x.y.z version numers to 'x.y.z-buildnum':
           <path release="1234" />   ->  requiredIDs("*-1234")
           <path release="3.0.0" />  ->  requiredIDs("3.0.0-*")
        """
        for child in elem.findall('path'):
            id = child.get('release', '')
            if not id:
                id = child.get('patch', '')
                if not id:
                    continue
            #
            # Correct for build-number-only or x.y.z-only strings
            #
            if re.search(r'^\d+$', id):
                id = '*-' + id
            elif re.search(r'^\d+\.\d+\.\d+$', id):
                id = id + '-*'

            self.requiredIDs.append(id)
            break
        #
        # Now take care of RebootOptional and MModeOptional
        #
        flag = elem.findtext('path/pre')
        if flag and flag == 'MModeOptional':
            self.SetFlag('MModeRequired', False)

        flag = elem.findtext('path/post')
        if flag and flag == 'RebootOptional':
            self.SetFlag('RebootRequired', False)


    def GetObsoletedIDs(self):   return self.obsoletedIDs
    def GetRequiredIDs(self):    return self.requiredIDs
    def GetConflictIDs(self):    return self.conflictIDs
    def GetSysStateDeps(self):   return self.sysStateDeps

    def IsRebootRequired(self):  return self.flags.get('RebootRequired', False)
    def IsMModeRequired(self):   return self.flags.get('MModeRequired', False)
    def RestartHostd(self):      return self.flags.get('HostdRestart', False)

    def SetObsoletedIDs(self, IDs):   self.obsoletedIDs = IDs
    def SetRequiredIDs(self, IDs):    self.requiredIDs = IDs
    def SetConflictIDs(self, IDs):    self.conflictIDs = IDs

    def UpdateSysStateDeps(self, newdeps):
        """ newdeps is a dict keyed by bundleID; each value is the new
        dict of system state dependencies to replace the old state deps.
        The values should be boolean - True to require that system state to
        be present, and False to require that it be absent.  Not listing
        the system state in the dict represents a don't care.
        """
        self.sysStateDeps.update(newdeps)

    def MergeFlags(self, otherUP):
        """ Merge the installation flags (self.flags) with those of
        another UpgradePaths instance. For each flag, if either one is true,
        then the result is true. Used to create a set of merged
        installation flags from a whole bunch of descriptors.
        self.flags is modified, otherUP is not.
        """
        for flag in self.flags:
            if flag in otherUP.flags:
                self.flags[flag] = self.flags[flag] or otherUP.flags[flag]

    def SetFlag(self, flag, value=True):
        """ Sets the install flag or KeyError if the flag is unknown """
        if flag in INSTALLFLAGS:
            self.flags[flag] = value
        else:
            raise KeyError, "%s is not a valid install flag" % (flag)

    def ToXml(self):
        """Serialize instance into <upgradepaths> XML structure."""
        elem = Element(self._tag)
        IDListElem(elem, "obsoletes", self.obsoletedIDs)
        IDListElem(elem, "requires", self.requiredIDs, self.sysStateDeps)
        IDListElem(elem, "conflicts", self.conflictIDs)
        BoolDictElem(elem, self.flags)
        return elem
