#######################################################################
# Copyright (C) 2005 VMWare, Inc.
# All Rights Reserved
########################################################################
#
# utils.py
#    Utility functions employed by the descriptor subpackage
#    NOT public, for subpackage use only
#
import os
import rpm
import logging
try:
    from elementtree.ElementTree import ElementTree, Element, iselement
except ImportError:
    from xml.etree.ElementTree import ElementTree, Element, iselement

# set up logging system
log = logging.getLogger("descutils")

#
# Globals and Constants
#
SPACING = "  "        # spaces to insert for each indent level


#
# Returns an RPM header object from a package "filename".
#  ts = RPM transaction set object
#
def ReadRpmHeader(ts, filename):
    """ Read an rpm header. """
    fd = os.open(filename, os.O_RDONLY)

    h = None
    try:
        h = ts.hdrFromFdno(fd)
    except rpm.error, e:
        log.warn("ReadRpmHeader(%s): %s" % (filename, str(e)))
        #if str(e) == "error reading package header":
        h = None
        raise

    os.close(fd)
    return h


#
# Returns a string  \n + SPACING * indent
#
def Indentstr(indent):
    return "\n" + SPACING * indent


#
# Recursively indents an Element tree for readable XML output.
# NOTE: Mixed-mode elements (text + child Elements) are not
# supported properly;  the assumption is that an Element
# contains only text or only child Elements.
#
# Side effects: Modifies text and tail members of Element instances
#
def IndentElementTree(elem, indent=0):
    """ Recursively indents Element elem and all children,
    starting at indentation level indent.
    """
    assert iselement(elem), "Expecting Element type but got %s" % (Type(elem))
    elem.tail = Indentstr(indent)
    for child in elem:
        IndentElementTree(child, indent + 1)

    if len(elem):
        #
        # Fix tail string of last child element so it returns us to our indent level
        #
        elem.text = Indentstr(indent + 1)
        elem[-1].tail = Indentstr(indent)


#
# Creates an XML Element list holding all the children.
# Children must have ToXml() method.
#
def ElementList(tag, children):
    elem = Element(tag)
    for child in children:
        elem.append(child.ToXml())
    return elem


#
# Maps string boolean values to True/False values
#
def StrToBool(strdict):
    booldict = {}
    for key in strdict:
        if strdict[key] == "0" or strdict[key].lower() == "false":
            booldict[key] = False
        else:
            booldict[key] = True
    return booldict


#
# Maps dict of bool values to dict of string values
#
def BoolToStr(booldict):
    strdict = {}
    for key in booldict:
        if booldict[key]:
            strdict[key] = "1"
        else:
            strdict[key] = "0"
    return strdict


#
# Adds <tag **attrs>newVal</tag> element to the end of root.
# Does nothing if newVal is empty string or None.
# **attrs are keyword arguments that become element attributes.
#
def SetElem(root, tag, newVal, **attrs):
    assert iselement(root), "root is not an Element instance"
    if newVal:
        elem = Element(tag, **attrs)
        elem.text = newVal
        root.append(elem)


#
# Adds <IDList field=""> element to the end of root.
# Does nothing if IDList is empty or None.
# if sysStateDeps is passed, then the attributes of <ID>id</ID>
# are stuffed with the key-value pairs in sysStateDeps[id].
# The values are bools.
#
def IDListElem(root, fieldname, IDList, sysStateDeps={}):
    assert iselement(root), "Expecting Element type but got %s" % (Type(root))
    if IDList:
        elem = Element("IDList", field=fieldname)
        for ID in IDList:
            if sysStateDeps and ID in sysStateDeps:
                SetElem(elem, "ID", ID, **BoolToStr(sysStateDeps[ID]))
            else:
                SetElem(elem, "ID", ID)
        root.append(elem)


#
# Extracts all the IDList children from elem,
# returning a dict keyed by the IDList field string,
# with each value being a list of IDs, in the same order
# they appear in the XML.
#
# In addition, a second dict is returned with the system state
# dependencies for each ID.  It is in the form of a dict:
#    {
#       'ESX-002': {'rebooted': True,
#                   'hostdRestarted': False }
#    }
# If there are no state dependencies for an ID, then no entry
# will be added to the second dict.
#
def GetIDLists(elem):
    assert iselement(elem), "Expecting Element type but got %s" % (Type(elem))
    lists = {}
    ssdeps = {}
    for idlist in elem.findall('IDList'):
        key = idlist.get('field', '')
        lists[key] = [id.text for id in idlist]
        for id in idlist:
            booldeps = StrToBool(dict(id.items()))
            if booldeps:
                ssdeps[id.text] = booldeps
    return lists, ssdeps


#
# Adds <Bool field="">True/False</Bool> element to the end of root.
#
def BoolElem(root, fieldname, bool, tag='Bool'):
    assert iselement(root), "Expecting Element type but got %s" % (Type(root))
    elem = Element(tag, field=fieldname)
    if bool:
        elem.text = "True"
    else:
        elem.text = "False"
    root.append(elem)


#
# Turns a dict of bools into a series of <Bool> elements,
# appended at the end of root.
#
def BoolDictElem(root, booldict, tag='Bool'):
    for key in booldict:
        BoolElem(root, key, booldict[key], tag)


#
# Extracts all the <Bool> children from elem, returning
# a dict keyed by the field attribute with either 1 or 0
# as the value.
#
def GetBools(elem, tag='Bool'):
    assert iselement(elem), "Expecting Element type but got %s" % (Type(elem))
    bools = {}
    for child in elem.findall(tag):
        bools[child.get('field', '')] = (child.text == 'True')
    return bools


#
# Returns 1 if an object behaves like a string
# (Python Cookbook recipe 3.3)
#
def IsStringLike(anobj):
    try: anobj + ''
    except: return 0
    else: return 1


#
# Returns 1 if an object behaves like a list
#
def IsListLike(anobj):
    try: anobj + []
    except: return 0
    else: return 1


#
## Return the name portion of a name.Arch package description
## for use with yum.
#
def NameFromNA(na):
    name, arch = ParseNAString(na)
    return name


#
## Returns (name, arch) or (name, "") if no valid arch is found.
## The tricky thing is dealing with names with
## periods in them, such as mydriver-scsi-v7.07.i386.  In this case,
## yum will not deal with the entire name.arch and can only deal with
## the name itself.
##
## How can we tell the difference between
##    name.arch  =>  'emacs.i386'
## or name.name  =>  'driver_v7.07'   which has no arch component?
##
## We compare the word after the final dot against a list of known
## Intel arch strings.  The chances of the final word coinciding with
## an arch string is extremely small, and someone is really asking for it
## if they do that, especially since periods are not recommended for
## rpm names in the first place.
##
# To get a definitive list of architectures supported by RPM, do a
#  #> grep arch_canon /usr/lib/rpm/rpmrc
#
intelArches = ['athlon', 'i386', 'i486', 'i586', 'i686',
               'x86_64', 'amd64', 'ia32e', 'noarch']

def ParseNAString(na):
    try:
        finaldot = na.rindex('.')
    except ValueError:
        return na, ""

    arch = na[finaldot+1:]
    if (arch in intelArches) or arch == "":
        return na[:finaldot], arch
    else:
        return na, ""
