#######################################################################
# Copyright (C) 2006 VMWare, Inc.
# All Rights Reserved
########################################################################
#
#   Depot.py
#
#
LOCAL_CACHE_ROOT="/var/spool/esxupdate"


import os
import sys
import logging
import shutil
import errno
import fnmatch
from vmware.descriptor.Descriptor import Descriptor, BadDescriptorError
from vmware.Signature import OneGoodSignature
import errors
from elementtree.ElementTree import ElementTree
from xml.parsers.expat import ExpatError

sys.path.append('/usr/share/yum')
import urlgrabber


log = logging.getLogger( "Depot" )

class SetupError(Exception): pass
class BadXML(Exception): pass

class DownloadError(Exception):
	"""This is a module-local DownloadError exception, used to 
	later construct the DepotDownloadError Exception in errors.py"""
	def __init__(self, msg, url):
		Exception.__init__(self, msg, url)
		self.msg = msg
		self.url = url

DESCRIPTOR_XML = 'descriptor.xml'

#=============================================================================
#python 2.2 doesn't have set()
if not hasattr(__builtins__, 'set'):
	class set(dict):
		"""This is the most ghetto set implementation ever.  In this
		module, we only use add() and update() and iterate"""
		def add(self, item):
			self[item] = 1
#=============================================================================


def readfile(filename):
	try:
		f = open(filename)
		str = f.read()
		f.close()
		return str
	except:
		return ''

#-----------------------------------------------------------------------------
class DownloadRules:
	""" Declares the bundles or files that should be included / excluded.

	Constructor takes two arguments which should be lists of bulletin IDs, 
	with * permitted as a wildcard.

	blacklist: if a bundleID/file is found in this list, it will not be
	           downloaded, superceding the exlcusives list.
	exclusives: if this is empty, all bundles/files will be downloaded
	            if it is nonempty, only those found in this list will
	            be downloaded
	"""
	def __init__(self, blacklist, exclusives):
		self.blacklist = blacklist
		self.exclusives = exclusives
	
	def IsExcluded(self, entry):
		fullPath = entry.remoteLocation

		if self.exclusives:
			isExcluded = True
		else:
			isExcluded = False

		for pattern in self.exclusives:
			#use fnmatch here in case there are wildcards
			if fnmatch.fnmatch(fullPath, pattern):
				isExcluded = False
				break
		for pattern in self.blacklist:
			if fnmatch.fnmatch(fullPath, pattern):
				isExcluded = True
				break
		return isExcluded

#-----------------------------------------------------------------------------
class IntegrityRules:
	"""Determines the filenames that should be excluded,
	the signature class to check each entry, and the GPG keyring
	directory to use.

	blacklist: a list of filenames that should be excluded from 
	the integrity check.  * is permitted as a wildcard.

	keyringDir: GPG keyring dir, or none for default GPG keyring.
	"""
	def __init__(self, blacklist=None, keyringDir=None):
		self.keydir = keyringDir
		self.SetBlacklist(blacklist)
		self._multiSigClass = OneGoodSignature

	def SetBlacklist(self, blacklist=None):
		defaultBlacklist = []
		if blacklist == None:
			self.blacklist = defaultBlacklist
		else:
			self.blacklist = blacklist
	
	def IsExcluded(self, entry):
		fullPath = entry.localLocation

		isExcluded = False

		for pattern in self.blacklist:
			if fnmatch.fnmatch(fullPath, pattern):
				isExcluded = True
				break
		return isExcluded

	def NewPolicyChecker(self, entry):
		return self._multiSigClass(keyringDir=self.keydir)


#-----------------------------------------------------------------------------
def LocalDownload( url, localPath ):
	""" Just checks for the existence of the remote file.  We do not
	actually copy local/NFS files to a cache.  The check for existence
	will help prevent a variety of other errors reported by RPM/yum/etc
	later on. """
	if not os.path.exists( localPath ):
		raise DownloadError( 'File does not exist: %s' % (localPath),
				     url )

	
#-----------------------------------------------------------------------------
def UrlgrabDownload( url, localPath ):
	""" Download a remote file at url via urlgrabber """
	try:
		urlgrabber.retrygrab( url, filename=localPath )
		assert os.path.getsize(localPath) != 0, 'IOError: not found'
	except Exception, e:
		try:
			os.unlink( localPath )
		except:
			pass
		raise DownloadError( "%s: %s" % (e, url), url )


#-----------------------------------------------------------------------------
def LocalUrl( url ):
	""" Returns true if the URL is NFS or local """
	return url.startswith( '/' ) or url.startswith( 'file:/' )


#-----------------------------------------------------------------------------
def Download( url, localPath ):
	""" Perform the download action based on url type """
	#
	# There is a problem with copy'ing NFS or local large files
	# reliably using either HTTPDownload or UrlgrabDownload.
	# It's better to just not cache local files.
	#
	try:
		if LocalUrl( url ):
			LocalDownload( url, localPath )
		elif url.startswith( 'http://' ):
			UrlgrabDownload( url, localPath )
		elif url.startswith( 'ftp://' ):
			UrlgrabDownload( url, localPath )
		else:
			msg = 'unrecognized URL protocol %s' % url
			raise errors.DepotAccessError( msg )
	except DownloadError, e:
		raise errors.DepotDownloadError(e.msg, 'Unknown', e.url)
	

#-----------------------------------------------------------------------------
def LocalLocation( url, localPath ):
	""" We do not cache local or file:/ urls, so the 'localLocation'
	is the same as the file path of the remote URL.  However,
	for http/ftp, return the cache-based local path.
	"""
	if url.startswith( '/' ):
		return url
	elif url.startswith( 'file://' ):
		return url[7:]
	elif url.startswith( 'file:' ):
		return url[5:]
	else:
		return localPath


#-----------------------------------------------------------------------------
class DepotEntry(object):
	"""A representation of any file or folder that lives on the Depot"""
	def __init__(self, name, dirEntry):
		name = os.path.normpath( name )
		self.name = name
		self.parentDir = dirEntry
		join = '/'
		if dirEntry.remoteLocation[-1] == '/':
			join = ''
		self.remoteLocation = dirEntry.remoteLocation + join  + name
		self.localLocation = dirEntry.localLocation + os.sep + name
		self.isInSync = False
		self.checkPassed = None #None, True, or False
		self.policyChecker = None
		self.errorEntry = self

	def __hash__(self):
		return self.localLocation.__hash__()
	
	def GetDownloadRules(self):
		"""Works its way up the tree until it gets the 
		DownloadRules object from the root entry"""
		return self.parentDir.GetDownloadRules()
	
	def GetIntegrityRules(self):
		"""Works its way up the tree until it gets the 
		IntegrityRules object from the root entry"""
		return self.parentDir.GetIntegrityRules()
		
	def GetBundleID(self):
		"""Works its way up the tree until it encounters a DirEntry
		that is a bundle."""
		assert self.parentDir, 'parentDir should exist'
		return self.parentDir.GetBundleID()
	
	def SyncFromRemote(self, forceSync=False):
		"""Pulls the file down from the remote location
		If the file can not be downloaded it should throw an exception
		If the file already exists locally, just set isInSync
		"""
		if forceSync or not os.path.exists( self.localLocation ):
			Download( self.remoteLocation, self.localLocation )
		self.isInSync = True

	def CheckIntegrity(self):
		"""Checks the policyChecker against the actual contents of the
		file (in the localLocation) to see if it is authentic"""
		if not self.isInSync:
			return

		self.checkPassed = False
		if not self.policyChecker:
			return

		log.debug( 'About to check policyChecker for ' +\
		           self.localLocation )
		self.checkPassed = self.policyChecker.Check(
			fileLocation=self.localLocation )


#-----------------------------------------------------------------------------
class DepotXMLFile(DepotEntry):
	"""A superclass for xml files that live on the Depot.  After such
	files are downloaded, the self.eRoot (ElementTree) attribute is 
	filled in from the contents of the file"""
	def __init__(self, name, parentDir):
		DepotEntry.__init__(self, name, parentDir)
		self.eRoot = None

	def SyncFromRemote(self, forceSync=False):
		DepotEntry.SyncFromRemote(self, forceSync)
		try:
			eTree = ElementTree( file=self.localLocation )
			self.eRoot = eTree.getroot()
		except ExpatError, e:
			msg = "XML Processing Error: %s" % e
			raise errors.DepotDownloadError(msg, 
			                                self.parentDir.bundleID,
			                                self.remoteLocation )

	def _find(self, parentNode, name):
		"""This gets one of the nodes from the internal elementtree.
		If the node is not present raise an error.  Missing nodes
		means a corrupt XML file."""
		node = parentNode.find(name)
		if node == None:
			msg = 'Missing node in xml: %s' % name
			raise BadXML( msg, self )
		return node

	def GetPolicyChecker(self, parentNode):
		"""Extract all the <signature> elements from parentNode
		and return an instance of MultiSignature.
		"""
		pChecker = self.GetIntegrityRules().NewPolicyChecker(self)
		for sigElem in parentNode.findall('signature'):
			#
			# If a <signature> element doesn't have id attrib, 
			# a default constant is assigned, in which case 
			# only one signature is possible.
			#
			sID = sigElem.get('id', 'none')

			sigValNode = self._find(  sigElem, 
			                          'signatureValue' )
			sigMethNode = self._find( sigElem, 
			                          'signedInfo/signatureMethod' )
			digestNode = self._find(  sigElem, 
			                          'signedInfo/digest' )
			try:
				sigAlg = sigMethNode.attrib['algorithm']
				digestAlg = digestNode.attrib['algorithm']
			except KeyError, e:
				msg = 'Missing attribute in xml node. Info: '+\
				      str(e)
				raise BadXML( msg, self )
			try:
				sig = sigValNode.text
				digest = digestNode.text
			except AttributeError, e:
				msg = 'Missing text in xml node. Info: %s' % e
				raise BadXML( msg, self )

			pChecker.AddSignature( sID, sig, sigAlg, 
			                       digest, digestAlg )

		return pChecker


#-----------------------------------------------------------------------------
class ContentsSig(DepotXMLFile):
	"""Special DepotEntry representing a contents.xml.sig file"""
	FILENAME = 'contents.xml.sig'

	def __init__(self, parentDir):
		DepotXMLFile.__init__(self, self.FILENAME, parentDir)

	def SyncFromRemote(self):
		"""Always download the .sig file. If the contents of
		the remote directory has changed, then the new and old
		.sig files will not match, and isInSync will be set
		to False."""
		oldfile = readfile( self.localLocation )
		DepotXMLFile.SyncFromRemote( self, forceSync=True )
		self.isInSync = oldfile == readfile( self.localLocation )

	def MakePolicyChecker(self):
		return self.GetPolicyChecker(self.eRoot)

	def CheckIntegrity(self):
		# contents.xml.sig is not checked
		self.checkPassed = True


#-----------------------------------------------------------------------------
class ContentsFile(DepotXMLFile):
	"""Special DepotEntry representing a contents.xml file"""
	FILENAME = 'contents.xml'

	def __init__(self, parentDir):
		DepotXMLFile.__init__(self, self.FILENAME, parentDir)

	def _FileTupleFromElement( self, elem ):
		"""A file tuple is a (name, signatureDict).  This method 
		generates one from the element tree"""
		fname = elem.attrib['name']
		pChecker = self.GetPolicyChecker(elem)
		return (fname, pChecker)

	def GetFileTuples(self):
		"""Returns a list of tuples, each tuple represents the 
		information about a file (name and signatures) that was 
		parsed from the XML file"""
		tupleEntries = []
		for fileElem in self.eRoot.findall('file'):
			fname, sigObj = self._FileTupleFromElement( fileElem )
			tupleEntries.append( (fname, sigObj) )
		return tupleEntries

	def GetFolderTuples(self):
		"""Returns a list of tuples, each tuple represents the 
		information about a folder (name, bundleID) that was 
		pulled from the XML file"""
		tupleEntries = []
		for folderElem in self.eRoot.findall('folder'):
			name = folderElem.attrib['name']
			bundleID = folderElem.get( 'bundleID', None )
			tupleEntries.append( (name, bundleID) )
		return tupleEntries


#-----------------------------------------------------------------------------
class DepotDirEntry(DepotEntry):
	"""Special DepotEntry representing a (sub)directory of the Depot"""

	def __init__(self, name, parentDir):
		DepotEntry.__init__(self, name, parentDir)

		self.contentsEntry = None
		self.entries = []

		#self.bundleID should only be set if this folder is the root
		#of a bundle
		self.bundleID = None

	def getPChecker(self):
		if not self.contentsEntry:
			return None
		return self.contentsEntry.policyChecker
	def setPChecker(self, sigs):
		pass

	policyChecker = property( getPChecker, setPChecker )

	def __str__(self):
		return '<'+ self.__class__.__name__ +' '+self.localLocation +'>'

	def DownloadAllFiles(self, forceSync=False ):
		"""Download all regular (non-dir) files from the remote 
		location, unless they are excluded by the DownloadRules"""
		for fname, sigObj in self.contentsEntry.GetFileTuples():
			if fname in [entry.name for entry in self.entries]:
				# it has already been added.
				# this is likely to happen if 
				# "contents.xml" or "contents.xml.sig"
				# somehow end up listed in the contents
				log.log(1, "Already have "+fname)
				continue

			# always add file entries so they could be
			# downloaded on demand
			newEntry = DepotEntry(fname, self)
			self.entries.append( newEntry )
			newEntry.policyChecker = sigObj

			if self.GetDownloadRules().IsExcluded(newEntry):
				log.debug('not downloading: '+ newEntry.name )
				continue

			newEntry.SyncFromRemote( forceSync )

	def DownloadAllFolders(self):
		"""Download all directories from the remote 
		location, unless they are excluded by the DownloadRules"""
		fTuples = self.contentsEntry.GetFolderTuples()
		for name, bundleID in fTuples:
			if name.startswith( ".." ):
				log.warn( "Parent dir not allowed" )
				continue
			elif name in [".", "./"]:
				#special case: usually the standalone 
				#update with a downloaded bundle
				if not self.bundleID:
					self.bundleID = bundleID
				continue

			newEntry = DepotDirEntry( name, self )

			if self.GetDownloadRules().IsExcluded(newEntry):
				log.debug('not downloading: '+ newEntry.name )
				continue

			if bundleID:
				newEntry.bundleID = bundleID

			#
			# If a bundle folder does not exist, just skip the bundle
			#
			try:
				newEntry.SyncFromRemote()
			except errors.DepotDownloadError, e:
				if bundleID:
					log.debug( e.msg )
					log.debug( 'Bundle %s is listed in'
							' contents.xml but one or more'
							' files not found. Skipping.' %(bundleID) )
					continue
				else:
					raise

			self.entries.append( newEntry )

	def SyncFromRemote(self):
		"""TODO: more docstring
		Note: Recursively (depth-first) calls SyncFromRemote() on all
		its children"""

		if self.remoteLocation == None:
			raise SetupError("Entry should have remoteLocation")
		if self.localLocation == None:
			raise SetupError("Entry should have localLocation")

		if not os.path.isdir( self.localLocation ):
			if LocalUrl( self.remoteLocation ):
				raise errors.DepotDownloadError(
					'Dir does not exist: %s' % (
					self.localLocation),
					'Unknown',
					self.remoteLocation )
			else:
				try:
					os.mkdir( self.localLocation )
				except OSError, e:
					raise errors.FileError( self.localLocation, 
								str(e) )

		#download the contents.xml.sig file
		contentsSigEntry = ContentsSig(self)
		self.entries.append( contentsSigEntry )
		contentsSigEntry.SyncFromRemote()
		changed = not contentsSigEntry.isInSync

		#download the contents.xml file
		self.contentsEntry = ContentsFile(self)
		self.entries.append( self.contentsEntry )
		self.contentsEntry.SyncFromRemote( forceSync=changed )
		self.contentsEntry.policyChecker = contentsSigEntry.MakePolicyChecker()

		try:
			self.DownloadAllFiles( forceSync=changed )
			self.DownloadAllFolders()
		except KeyError:
			log.error( 'Bad contents XML file in %s' % self.name )
			raise

	def GetIntegrityRules(self):
		rules = DepotEntry.GetIntegrityRules(self)
		return rules

	def CheckIntegrity(self):
		#TODO: this should update the progress bar
		self.checkPassed = False
		for entry in self.entries:

			if self.GetIntegrityRules().IsExcluded(entry):
				log.debug('not sigchecking: '+ entry.name )
				continue

			if entry.checkPassed == None:
				entry.CheckIntegrity()
			if not entry.checkPassed:
				self.errorEntry = entry.errorEntry
				return
		self.checkPassed = True

	def GetBundleID(self):
		"""If this DirEntry represents a bundle, returns its bundleID,
		otherwise, works its way up the tree until it encounters 
		a DirEntry that is a bundle."""
		if self.bundleID:
			return self.bundleID
		return DepotEntry.GetBundleID(self)

	def GetDirEntries(self):
		return [ entry for entry in self.entries
		         if isinstance( entry, DepotDirEntry ) ]

	def GetFileEntries(self):
		return [ entry for entry in self.entries
		         if not isinstance( entry, DepotDirEntry ) ]

	def GetEntry(self, basename):
		"""return the entry that matches basename."""
		for entry in self.entries:
			if entry.name == basename:
				return entry
		return None

	def DeepFindFileMatching( self, pattern ):
		results = []
		for entry in self.entries:
			if isinstance( entry, DepotDirEntry ):
				results += entry.DeepFindFileMatching( pattern )
			else:
				if fnmatch.fnmatch(entry.name, pattern):
					results.append( entry )
		return results
		

	def DeepGetBundleEntries(self):
		"""return a Set (or pseudo-Set if py version < 2.4) of any
		descendant DirEntries that qualify as bundles"""
		bundleEntries = set()
		if self.bundleID != None:
			bundleEntries.add( self )

		for d in self.GetDirEntries():
			bundleEntries.update( d.DeepGetBundleEntries() )

		return bundleEntries

	def GetBundleDict(self):
		"""return a dict of bundles that can be consumed by 
		external code - ie, it can be used by scan.py and the 
		remediate code
		throws errors.BadDepotDescriptorError"""
		bundleEntries = self.DeepGetBundleEntries()
		bundleDict = {}
		bundleDirEntry = {}
		for bEntry in bundleEntries:
			descLoc = None
			for fEntry in bEntry.GetFileEntries():
				if fEntry.name == DESCRIPTOR_XML:
					descLoc = fEntry.localLocation
					break
			if not descLoc:
				msg = DESCRIPTOR_XML +' not downloaded locally'
				raise errors.DepotDownloadError(
				                              msg,
				                              bEntry.bundleID,
				                              DESCRIPTOR_XML )


			try:
				descriptor = Descriptor( descLoc )
			except BadDescriptorError, e:
				msg = str( e )
				raise errors.BadDepotDescriptorError( 
				                              bEntry.bundleID,
				                              descLoc,
				                              msg )
			#
			# bundleDict cannot reference any classes
			# as this fails when Python 2.2's implementation
			# of deepcopy() is used on it
			#
			bundleDict[bEntry.bundleID] = {
				            'url': bEntry.remoteLocation,
				            'localPath': bEntry.localLocation,
				            'desc': descriptor
				            }
			bundleDirEntry[bEntry.bundleID] = bEntry
		return bundleDict, bundleDirEntry


#-----------------------------------------------------------------------------
class DepotRootDirEntry(DepotDirEntry):
	"""Special DepotDir representing the root directory of the Depot"""
	def __init__(self, depot, remoteURL, localPath):
		self.depot = depot
		localPath = os.path.normpath( localPath )
		self.name = os.path.basename( localPath )
		self.parentDir = None
		self.remoteLocation = remoteURL
		self.localLocation = LocalLocation( remoteURL, localPath )
		self.isInSync = False
		self.checkPassed = None #None, True, or False
		self.entries = []
		self.errorEntry = self
		#self.bundleID should only be set if this folder is the root
		#of a bundle
		self.bundleID = None

	def GetIntegrityRules(self):
		return self.depot.integrityRules

	def GetDownloadRules(self):
		return self.depot.downloadRules

	def GetBundleID(self):
		if not self.bundleID:
			return 'None/Unknown'
		return self.bundleID

	def SyncFromRemote(self):
		try:
			DepotDirEntry.SyncFromRemote(self)
		except DownloadError, e:
			raise errors.DepotDownloadError(e.msg, 'Unknown', e.url)
		except BadXML, e:
			msg, entry = e.args
			raise errors.BadDepotMetadataError(entry.GetBundleID(),
			                                   entry.remoteLocation,
			                                   msg )

	def CheckIntegrity(self, raiseAnyError=False):
		#TODO: this should update the progress bar
		self.checkPassed = True
		for entry in self.entries:

			if self.GetIntegrityRules().IsExcluded(entry):
				log.debug('not sigchecking: '+ entry.name )
				continue

			if entry.checkPassed == None:
				entry.CheckIntegrity()

			bundleID = entry.errorEntry.GetBundleID()
			filename = entry.errorEntry.localLocation
			sigs = entry.errorEntry.policyChecker

			if isinstance( entry, DepotDirEntry ): 
				if bundleID not in self.depot.bundles:
					log.error("filename=%s" % (filename))
					raise KeyError, "bundleID " + \
					      bundleID + "not in bundledict"
				b = self.depot.bundles[bundleID]
				b['integritySuccess'] = entry.checkPassed
				msg = 'File: ' + filename + '\n' + sigs.errmsg
				b['integrityDescription'] = msg

			if not entry.checkPassed:
				self.checkPassed = False
				self.errorEntry = entry.errorEntry
				
				if raiseAnyError or not isinstance( entry,
                                                            DepotDirEntry ):
					raise errors.IntegrityError(
						bundleID, filename,
						'Integrity Error!\n' + \
						sigs.errmsg )


#-----------------------------------------------------------------------------
class Depot:
	"""Manages a depot of update bundles, each of which contains RPMs
	and metadata.  The depot file structure is as follows:
	  +-- Depot-esx-x.y.z ---
	    +--- ESX-3.0.0-GA
	      +---- headers
	      |---- contents.xml
	      |---- contents.xml.sig
	      |---- descriptor.xml
	      |---- foo.1.2.3.rpm
	      |---- bar.3.2.1.rpm
	    +--- patch1
	    +--- patch2
	    |--- contents.xml
	    |--- contents.xml.sig
	"""
	def __init__(self, depotURL, scanlist=[], flushCache=False,
		     keyringDir=None, localCacheRoot=LOCAL_CACHE_ROOT):
		self.url = depotURL
		self.downloadRules = None
		self.integrityRules = IntegrityRules(keyringDir=keyringDir)
		self.reqBundles = []

		if flushCache and os.path.isdir(localCacheRoot):
			try:
				shutil.rmtree(localCacheRoot)
			except OSError:
				msg = "Could not flush cache"
				raise errors.FileError( localCacheRoot, msg )
			
		try:
			os.mkdir(localCacheRoot) 
		except OSError, e:
			if not e.errno == errno.EEXIST:
				raise errors.FileError(localCacheRoot, str(e))
			#do nothing if it was "File already exists"

		self.root = DepotRootDirEntry(self, depotURL, localCacheRoot)
		self.SetScanlist(scanlist)
		self.bundles = {}
		self.bundleDirEntry = {}

	def SyncFromRemote(self):
		log.debug( 'Download Rules: %s' % vars(self.downloadRules) )
		self.root.SyncFromRemote()
		self.bundles, self.bundleDirEntry = self.root.GetBundleDict()

	def SanityCheck(self):
		self.CheckDescriptorAndFolderNameMatch() #raises BadDepotDescr
		missingBuns = self.GetMissingRequestedBundles(
			self.bundles.keys() )
		if missingBuns:
			bundles = ', '.join( missingBuns )
			msg = 'Requested bundle(s) not found: %s' % (bundles)
			raise errors.DepotDownloadError(msg, bundles, self.url)

	def __str__(self):
		urlString = self.url
		if len( urlString ) > 24:
			urlString = urlString[:10] +'...'+ urlString[-10:]
		return '<Depot %s size: %s>' % (urlString, len(self.bundles) )


	def GetMissingRequestedBundles(self, bundleIDs):
		"""For every requested bundleID pattern, check that we've 
		got it.  Return a list of all the missing requests, or [].  
		"""
		missing = []
		for pattern in self.reqBundles:
			found = False
			for bundleID in bundleIDs:
				if fnmatch.fnmatch(bundleID, pattern):
					found = True
					break
			if not found:
				missing.append( pattern )
		return missing
					
	def CheckDescriptorAndFolderNameMatch(self):
		"""Goes through all the bundles and checks that the bundleID
		inside the descriptor.xml file matches the folder name.
		returns None
		raises errors.BadDepotDescriptorError if it finds a mismatch"""
		for bundleID, bundleDict in self.bundles.items():
			localPath = bundleDict['localPath']
			basename = os.path.basename( localPath )
			release = bundleDict['desc'].GetRelease()
			if release != basename:
				msg = ('Descriptor release (%s) ' +\
				      'doesnt match directory name (%s)') %\
				      (release, localPath)
				raise errors.BadDepotDescriptorError( release,
				                                      localPath,
				                                      msg )
			if bundleID != release:
				msg = ('Descriptor release (%s) ' +\
				      'doesnt match bundleID of contents.xml (%s)') %\
				      (release, bundleID)
				raise errors.BadDepotDescriptorError( release,
				                                      localPath,
				                                      msg )
		return None

	def CheckIntegrity(self, newBlacklist=None, raiseAnyError=False):
                """ Validates the signatures in the entire depot.
                newBlacklist    - list of what not to check
                raiseAnyError   - True: raise IntegrityError at first failure
                                  False: bundle failures are stored in
                                    bundledict, not raised.
                """
		if newBlacklist != None:
			self.integrityRules.SetBlacklist(newBlacklist)
		self.root.CheckIntegrity(raiseAnyError)
		return self.root.checkPassed
			
				
	def SetScanlist(self, scanlist):
		""" Sets the bulletins or bundles to scan.  Scanlist is a 
		list of bulletin IDs, with * permitted as a wildcard.
		"""
		if not scanlist:
			self.downloadRules = DownloadRules( [], [] )
			self.reqBundles = []
			return

		self.reqBundles = scanlist[:]

		rootPath = self.root.remoteLocation
		if rootPath[-1] != '/':
			rootPath += '/'
		exclusives = [ rootPath,
		               rootPath +'contents.xml',
		               rootPath +'contents.xml.sig' ]
		for pat in scanlist:
			exclusives.append( rootPath + pat )
			exclusives.append( rootPath + pat +'/*' )
			
		self.downloadRules = DownloadRules( [], exclusives )

	def SetBlacklist(self, blacklist):
		""" Sets the file patterns to not download.
		Also exclude those patterns from sig check. """
		self.downloadRules.blacklist = blacklist
		self.integrityRules.SetBlacklist(blacklist)

	def GetAllHeaderFiles(self):
		hdrEntries = self.root.DeepFindFileMatching( "*.hdr" )
		return [e.localLocation for e in hdrEntries]


#-----------------------------------------------------------------------------
class DepotInBundle(Depot):
	""" A special class for treating a bundle as a depot. The
	bundle-level contents.xml must have a special entry
	referring to itself: <folder name="." bundleID="patch1" />
	
	The bundle file structure is as follows:
	    +--- ESX310-20070605-VMcore-BA
	      +---- headers
	      |---- contents.xml
	      |---- contents.xml.sig
	      |---- descriptor.xml
	      |---- foo.1.2.3.rpm
	      |---- bar.3.2.1.rpm

	The scanlist argument is ignored.
	"""
	def SetScanlist(self, scanlist):
		""" Only one bundle to scan """
		self.downloadRules = DownloadRules( [], [] )
		self.reqBundles = []

	def CheckDescriptorAndFolderNameMatch(self):
		#
		# Disable this check for bundle depots.  The bundle entry
		# will never match the ID as it has a name of "."
		#
		pass
