#######################################################################
# Copyright (C) 2005 VMWare, Inc.
# All Rights Reserved
########################################################################
#
# InstallState.py
#
#   InstallState and each of the derived installation state classes
#
# Testing:
#   test/testinstall.py
#

from time import strftime, time, localtime
import logging
import insthelper
import scan
import errors
import cmdline
import os, os.path
from vmware.system import LogCommand, LogCommandError, Reboot

#
# Constants/globals
#
log = logging.getLogger('states')
sumlog = logging.getLogger('summary')

REPACKAGE_DIR    	= '/var/spool/repackage'
SPOOL_DIR_PATH	 	= '/var/spool'
SPOOL_DIR		= 'esxupdate'
SYSTEM_BACKUP_FILE	= 'systemfilesbackup.tar.gz'
PCIID_SUPPRESS_FILE     = '/tmp/suppress-esxcfg-pciid'


########################################################################
#
# Base installation state class. Defines the Action/Next paradigm.
#

class InstallState:
    """ Base class for an installation step.

    Think of it as one state in a state diagram.  It can perform an action, and
    save the results in the parent session, and based on session's state, determine
    the next state to go to.  Some notes:
      * Action() is always done, even if IsDone() == 1
      * If an error occurs in Action, the error object should be saved in self.err
        and the NextState() logic should take that into account.

    Members:
    session   - ref to session that owns this step
    starttime - time instance was created, in seconds since the Epoch
    """
    
    def __init__(self, session):
        """ Initialize session & timestamp; check state transition """
        self.session = session
        self.err = None
        self.starttime = time()
        log.debug('Entering ' + self.__repr__())
        #
        # cache local value of commonly used session state
        #
        self.test = session.GetOption('test')


    def GetState(self): return str(self.__class__).split('.')[-1]
    def GetTime(self):  return self.starttime
    
    def __repr__(self):
        """ Stringify step state... str(), print, `` """
        return '[%20s] [%-20s]' % (strftime('%X %x', localtime(self.starttime)),
                                   self.GetState())

    def Action(self):
        """ This is where the step is carried out """
        pass

    def NextState(self):
        """ Returns the next InstallStep instance, None if terminal state """
        return None

    def IsDone(self):    return 0


########################################################################
#
# Functional install states
#

class ConfigState(InstallState):
    """ Validate bundle dependencies and check maintenance mode.
    - download descriptors and do integrity check
    - scan the depot using passed in bundle list
    - weed out inapplicable bundles and order the remainder
    - exit if no bundles can be applied
    - combine upgradepaths; check for maintenanceMode if any
      bundle requires it

    If Action() is completed, session state is updated thus:
    - bundleGroups is filled out
    - flags is filled out
    - descs[] is filled out
    - lastDesclist (the list of descriptors scanned)
    - bundleEntry[] is filled out

    self.session.options[]:
    'pullDeps'  - automatically pull in dependencies
    """

    def _parseScanResult(self, bundleID, resHash):
        """Parse a scan result and raise appropriate error"""
        #
        # Raise appropriate error for scan result of bundleID
        #
        explanations = resHash['explanations']
        if resHash['installed']:
            raise errors.AlreadyInstalledError(bundleID)
        elif resHash['obsolete']:
            raise errors.IsObsoleteError(bundleID,
                    resHash['newerAvailable'].keys())
        elif resHash['availableIDs']:
            raise errors.PatchRequiresError(bundleID, resHash['availableIDs'],
                                            explanations)
        elif resHash['missingIDs']:
            raise errors.PatchRequiresError(bundleID, resHash['missingIDs'],
                                            explanations)
        elif resHash['conflictIDs']:
            raise errors.ConflictsError(bundleID,
                                        resHash['conflictIDs'].keys(),
                                        explanations)

        msg = "Bundle %s not applicable for unknown reason" % (bundleID)
        raise errors.PatchDependencyError(bundleID, msg=msg,
                                          explanations=explanations)

    def _parseScanResults(self, results, baseline=[], rollup=None):
        """ Analyze scan results and throw appropriate errors """
        instbundles = self.session.OrderedBundleList()
        explanations = []
        for bundleID, resHash in results.items():
            if bundleID not in instbundles:
                explanations += resHash['explanations']

        # Check if the bundles specified by -b option are applicable.
        for bundleID in baseline:
            if bundleID not in instbundles:
                # baseline bundle is not applicable.
                # raise error and exit unless the bundle is already installed
                # or obsoleted.
                if not results[bundleID]['installed'] and not results[bundleID]['obsolete']:
                    self._parseScanResult(bundleID, results[bundleID])

        # check if the rollup bundle is applicable.
        if rollup and rollup not in instbundles:
            # rollup bundle is not applicable. raise error and exit.
            self._parseScanResult(rollup, results[rollup])

        #
        # If just one bundle was requested, propagate the error.
        # If multiple bundles, log the inapplicable reason for each bundle
        # NB: scan.ComputeBundleGroups() will allow a re-installation
        # if --force is specified, so we don't have to handle that
        # here.
        #
        if not instbundles:
            if len(results) == 1:
                bundleID = results.keys()[0]
                self._parseScanResult(bundleID, results[bundleID])
            else:
                msg = "Bundles %s cannot be installed, check log for " \
                      "reasons" % (results.keys())
                raise errors.PatchDependencyError(None, results.keys(),
                                                  msg, explanations)
        elif explanations:
            log.info("The following bundles are being skipped.")
            for msg in explanations:
                log.info(msg)

    def __action__(self):
        sessOp = self.session.GetOption
        depot = self.session.depot
        log.info('Configuring...')

        #
        # This first pass should just download the bundles passed
        # in on the command line.  This is necessary to figure
        # out if we're doing rollup or not.
        #
        self.session.DownloadDescriptors()
        if not sessOp('nosigcheck'):
            hasIntegrity = depot.CheckIntegrity(raiseAnyError=True)

        #
        # Set rollup flag if appropriate.  Error out if rollup specified
        # with other bundle(s).
        #
        # Unless a rollup bundle is being installed, create a baseline.
        # A baseline is a list of bundle IDs specified on the command line.
        # Only these bundle IDs and their direct requirements will be added 
        # to the scan list.
        baseline = list()
        rollup = None
        expandedIDs = sessOp('bundleIDs')

        if '*' not in expandedIDs:
            expandedIDs = scan.FindKeysByPatternList(depot.bundles, expandedIDs)
            for bundleID in expandedIDs:
                baseline.append(bundleID)
                bundleType = depot.bundles[bundleID]['desc'].bundleType
                if bundleType.strip().lower() in ('update', 'rollup'):
                    rollup = bundleID

        # if a rollup is being installed in HA mode, we assume that
        # dependencies have been resolved by VUM, and all of rollup's required
        # bundles are in the baseline.  Therefore its not necessary to use the
        # rollup variable as a means of controlling which dependency resolution
        # method to use.
        if sessOp('hostagent'):
            rollup = None

        if rollup and len(expandedIDs) != 1:
            raise errors.NotApplicableError('Cannot specify a rollup or update '
                                            'bundle (%s) with multiple bundle '
                                            'operation.' % bundleID)

        #
        # If pullDeps=1, first do a pre-scan of all
        # the descriptors in the depot, and determine
        # which other bundles need to be pulled in
        # by the requested ID list.
        #
        if sessOp('pullDeps'):
            depot.SetScanlist([])
            depot.SetBlacklist(['*.rpm'])
            depot.SyncFromRemote()
            depot.SanityCheck()
            if not sessOp('nosigcheck'):
                hasIntegrity = depot.CheckIntegrity(raiseAnyError=True)
            results = scan.ScanDepot(depot, self.session.db,
                                     sessOp('force'), groupMode=True,
                                     rollup=rollup, baseline=baseline)
            newScanlist = scan.PulledDeps(sessOp('bundleIDs'), results)
            #
            # Don't pass SetScanlist an empty list - it defaults to '*'
            if newScanlist:
                depot.SetScanlist(newScanlist)
            else:
                depot.SetScanlist(sessOp('bundleIDs'))
            depot.root.entries = []

        #
        # download descriptors and do integrity check
        # raises IntegrityError
        #
        self.session.DownloadDescriptors()
        if not sessOp('nosigcheck'):
            hasIntegrity = depot.CheckIntegrity(raiseAnyError=True)

        #
        # scan the depot using passed in bundle list
        # weed out inapplicable bundles and order the remainder
        # exception if no bundles can be applied
        #
        results = scan.ScanDepot(self.session.depot,
                                 self.session.db,
                                 scanDepotOnly=sessOp('force'),
                                 groupMode=True, rollup=rollup,
                                 baseline=baseline)
        groups = scan.ComputeBundleGroups(results,
                                          reinstall=sessOp('force'))
        self.session.bundleGroups = groups
        self.session.lastDesclist = scan.lastDesclist
        self._parseScanResults(results, baseline, rollup)
        self.session.results = results
        
        #
        # check for maintenanceMode if any bundle requires it
        # In host agent mode, rely on --maintenancemode flag from hostd
        # Otherwise, check using HaveRunningVMs().
        #
        self.session.MergeInstallFlags()
        paths = self.session.flags
        if sessOp('hostagent'):
            if paths.IsMModeRequired() and not sessOp('maintenancemode'):
                raise errors.VMsRunningError()
        else:
            if paths.IsMModeRequired() and insthelper.HaveRunningVMs():
                raise errors.VMsRunningError()
        
    #
    # Catches expected errors from __action__ and saves them in session.
    # Unexpected errors will still be raised all the way up.
    def Action(self):
        try:
            self.__action__()
        except EnvironmentError, e:
            log.error( str(e) )
            self.err = e
        except (errors.DepotAccessError, errors.DepotDownloadError,
                errors.BadDepotDescriptorError, errors.IntegrityError,
                errors.FileError, errors.NotApplicableError), e:
            self.err = e

        try:
            self.session.SaveForRestore()
        except EnvironmentError, e:
            log.error('Cannot save install session state: %s' % str(e))
            self.err = e
       
    def NextState(self):
        if self.err:
            return ConfigFailed(self.session)
        else:
            return PrepState(self.session)


#
# Used to install newer versions of esxupdate before updating the
# other packages.
#
class PreInstallState(InstallState):
    """ yum install the rpms (and their deps) in <preinstall> list.
    """
    def Action(self):
        try:
            preInstList = self.session.GetPending('preinstall')
            if preInstList:
                sumlog.info('Pre-installing %s...' % (preInstList))
                insthelper.InvokeYum('install', preInstList, [],
                                     None, self.test,
                                     repackage=self.session.repackaged)
                self.session.UpdatePendingFromHost()

        except LogCommandError, e:
            log.error("Error (%d) executing [%s]\n%s" % (e.res>>8, e.cmd, e.output))
            self.err = e
        except errors.YumGenericError, e:
            self.err = e
        else:
            log.info('Pre-install packages up-to-date')

    def NextState(self):
        preinstalls = self.session.GetPending('preinstall')
        if self.err:
            return UpgradeFailed(self.session)
        elif preinstalls:
            sumlog.info('Pre-install failed for %s' % (preinstalls))
            self.err = errors.IncompleteUpgradeError(
                'Yum did not pre-install some pkgs', preinstalls)
            return UpgradeFailed(self.session)
        else:
            return PreInstDoneState(self.session)


class PrepState(InstallState):
    """ Prepares ESX host for installation.  Computes the list of
    pending packages, assigning an action to each package. Configures
    yum.  Session state by the end of this state:
      latestlist    : combined Rpmlist() of all bundles packages
      pending       : dict of package names and associated action
    Relevant options:
      force      - reinstall same rpms and --force older ones
      baseurls   - QA mode
      rpmroot    - use a custom RPM database (for testing)
    Side effects:
      Overwrites /etc/yum.conf
    """
    def __action__(self):
        sessOp = self.session.GetOption
        if not sessOp('rpmroot'):
            self.session.SetOptions({'rpmroot': '/'})
        bundles = self.session.OrderedBundleList()
        sumlog.info('Preparing to install %s...' % (bundles))

        #
        # Remove what's installed already from pending pkg list
        # If all RPMs were already installed, go through the steps anyway --
        #   as long as the installed dict is empty, PostState will be
        #   skipped so the system won't be touched.
        # Unless --force specified, in which case write over installed pkgs
        #
        log.debug('Prepare pending package list...')
         
        self.session.InitPending()
        latestlist = self.session.latestlist
        dblist = insthelper.GetRpmlistFromRpmdb(sessOp('rpmroot'))
        matches = latestlist.Union(dblist).GetPkgNAList()
        if not sessOp('force') and len(matches):
            log.info('The following RPMs are already installed on the '
                     'system and will be skipped: %s' % (matches))
            if self.session.preinstUpdatesDb:
                self.session.RemovePending(matches)
            else:
                #
                # Esxupdate was restarted after preinstall, and the prev
                # esxupdate did not update the installed list with the
                # preinstalled packages.  Since yum may pull in
                # dependencies, we just move every match in <rpmlist>
                # over to installed, rather than trying to figure it out.
                # For -l query output accuracy, putting too many pkgs
                # on installed list is way better than not enough.
                #
                self.session.VerifiedPending(matches)
        else:
            self.session.MarkPending(matches, 'downgrade')

        #
        # Don't attempt to remove packages that aren't in RPMDB
        # or test transaction will complain
        # Don't acknowledge any of those pkgs as removed either
        #
        self.session.UpdateRemovedFromHost()
        self.session.removed = {}

        #
        # Mark new packages or those that need downgrade
        #
        donttouch = self.session.MatchPending(['exclude', 'nodeps'])
        updates, new, older = latestlist.GetUpdates(dblist)
        downlist = [na for na in older.GetPkgNAList() if na not in donttouch]
        uplist = [na for na in updates.GetPkgNAList() if na not in donttouch]

        self.session.downgrades = [(dblist.GetPkgByName(rpm.GetName()), rpm)
                                   for rpm in older.GetRpms()]

        #
        # Support 'QA' mode where secondary -r urls point to base repos.
        # By default, all pkgs are marked 'install' and yum install of
        # specific pkgs is used rather than yum upgrade, as yum upgrade
        # could accidentally install the wrong packages, in case of <removes>
        # or <nodeps>.  Change mark to 'upgrade' for packages in uplist
        # so that UpgradeState will not be skipped, when secondary -r
        # arguments are specified.
        # XXX: Secondary repos are not officially supported.  Please remove the
        # code below when we move to using /build/depot/depot
        #
        if sessOp('baseurls'):
            self.session.MarkPending(uplist, 'upgrade')

        #
        # --force is needed to install older pkgs on an update.
        # Without --force, older packages will be skipped.  We assume
        # that newer pkgs have been installed as part of a newer patch, and
        # the user may just be installing patches out of order.
        # NB: --force is not needed to install the packages on a descriptor's
        #     <nodeps> list.
        #
        if sessOp('force'):
            self.session.MarkPending(downlist, 'downgrade')
        elif len(downlist) > 0:
            log.info('The following RPMs have been superseded '
                     'and will be skipped: %s' % (downlist))
            self.session.RemovePending(downlist)
        self.session.DumpPkglists()
        self.downlist = self.session.GetPending('downgrade')

        #
        # Check $PATH and suppress esxcfg-pciid for driver updates
        #
        log.debug('Checking $PATH...')
        insthelper.CheckEnvPath()

        if self.session.DriverPending():
            log.debug('Suppressing esxcfg-pciid run during driver installation...')
            supfile = open(PCIID_SUPPRESS_FILE, 'w')
            supfile.write(str(os.getpid()))
            supfile.close()

        if sessOp('flushcache'):
            insthelper.FlushYumCache()

        #
        # Check for version of yum with --repackage, if not don't
        # prepare for rollbacks.
        #
        if insthelper.GetYumVer() < ('2.0.7', '2vmw'):
            self.session.repackaged = 0
            log.warn('You are using a version of yum which does not support ' +
                     'rollbacks, so the rollback functionality will be disabled ' +
                     'for this installation.')
            
        #
        # Calculate the number of "steps" for the progress indication
        # Note - for yum, each package has 3-4 steps:
        #   download, (repackage), install, post-install
        # Add steps for post-install, etc.
        #
        forcepkgs = len(self.session.MatchPending(['downgrade', 'nodeps']))
        yumpkgs = len(self.session.MatchPending(['upgrade', 'install']))
        removepkgs = len(self.session.GetPending('remove'))
        if self.session.repackaged:
           steps = yumpkgs * 4 + forcepkgs + removepkgs
        else:
           steps = yumpkgs * 3 + forcepkgs + removepkgs
        steps += 1
        self.session.CreateProgressBar(steps)

        if self.session.repackaged:
               log.debug('Enabling rollbacks for this install...')
               #
               # Delete backedup files from last install
               # Also, perform backup of system files.  Fail -> repackage=0
               #
               insthelper.BackupSystemFiles(SPOOL_DIR_PATH, SPOOL_DIR,
                                            SYSTEM_BACKUP_FILE)               

               #
               # Delete repackaged RPMs from before this install
               # We cannot just do a 'rm -f *' since we might have
               # repackaged RPMs from the pre-install stage.
               #
               installstart = self.session.GetTXStartTime()
               insthelper.RmdirByTime(REPACKAGE_DIR, 0, installstart)

    def Action(self):
        try:
            self.__action__()
        except (EnvironmentError, errors.NoYumError), e:
            self.err = e

        try:
            self.session.SaveForRestore()
        except EnvironmentError, e:
            log.error('Cannot save install session state: %s' % (e.msg))
            self.err = e
            
    def NextState(self):
        if self.err:
            return PrepFailed(self.session)
        else:
            return DownloadState(self.session)


class DownloadState(InstallState):
    """ Download rpms to be installed.
    New session state added:
      pkgEntry[]    - DepotEntry instance for each rpm
    Side effects:
      Fills up depot cache with rpms and yum headers
    """
    def __action__(self):
        # TODO: Integrate this with progressbar
        sessOp = self.session.GetOption
        self.session.SetEntryInstances()

        for na in self.session.pending:
            if self.session.pending[na] == 'exclude':
                continue
            if na not in self.session.pkgEntry:
                continue
            entry = self.session.pkgEntry[na]
            log.info('Downloading %s...' % (entry.name))
            entry.SyncFromRemote()
            if not sessOp('nosigcheck'):
                self._ConfirmIntegrityOrRaiseException( entry )

        #
        # Write /etc/yum.conf and generate yum headers for
        # HTTP/FTP depots based on the just-downloaded RPMs.
        #
        self.session.SetRepo()

    def Action(self):
        try:
            self.__action__()
        except (errors.DepotAccessError,
                errors.DepotDownloadError,
                errors.IntegrityError, errors.NoYumError,
                EnvironmentError), e:
            self.err = e
            
    def NextState(self):
        if self.err:
            return DownloadFailed(self.session)
        else:
            return TestState(self.session)

    def _ConfirmIntegrityOrRaiseException(self, entry):
        import Depot # the import is done here instead of in the module
                     # scope to ensure we don't bring in "new code"
                     # in the case that the user used the --use-old-code
                     # argument
        if isinstance(entry, Depot.DepotDirEntry):
            entry.CheckIntegrity(raiseAnyError=True)
        else:
            entry.CheckIntegrity()
            if not entry.checkPassed:
                raise errors.IntegrityError(entry.GetBundleID(),
                                            entry.localLocation,
                                            'Integrity Error!\n' + \
                                            entry.policyChecker.errmsg)


class TestState(InstallState):
    """ Run a test RPM transaction and check for problems """
    def Action(self):
        pkgs = self.session.MatchPending(['preinstall', 'upgrade',
                                          'install', 'downgrade', 'nodeps'])
        rpmfiles = [self.session.pkgEntry[na].localLocation for na in pkgs]
        log.info('Checking disk space and running test transaction...')
        try:
            # Check some common cases for /boot, /tmp and /.
            # 24 MB free space on /boot and /tmp?
            insthelper.CheckPartitionFreeSpace('/boot', 24 * 1024 * 1024)
            insthelper.CheckPartitionFreeSpace('/tmp', 24 * 1024 * 1024)
            # 50 MB free space on '/'?
            insthelper.CheckPartitionFreeSpace('/', 50 * 1024 * 1024)
            insthelper.TestTransaction(rpmfiles,
                                       self.session.GetPending('remove'),
                                       self.session.repackaged,
                                       self.session.GetOption('rpmroot'))
        except errors.DiskSpaceError, e:
            self.err = e
    
    def NextState(self):
        if self.err:
            return TestFailed(self.session)
        elif self.session.GetOption('test'):
            return TestCompletedState(self.session)
        elif self.session.GetPending('preinstall'):
            return PreInstallState(self.session)
        else:
            return RemoveState(self.session)


class RemoveState(InstallState):
    """ Remove packages from the system that are obsolete.
    rpm -e --nodeps is used.  Packages that are not removed will
    not cause an error but a warning will be printed out.
    The exception list (-x) applies here, as well as --test.
    """
    def __action__(self):
        removeNA = self.session.GetPending('remove')
        if removeNA:
           sumlog.info('Removing packages %s...' % (removeNA))
           insthelper.RemoveRpms(removeNA,
                                 test=self.test,
                                 progressbar=self.session.progress)
           self.session.UpdatePendingFromHost()
           notremoved = self.session.GetPending('remove')
           if notremoved:
              log.warning('The following packages could ' \
                          'not be removed: ' + str(notremoved))

    def Action(self):
        try:
            self.__action__()
        except LogCommandError, e:
            log.error("Error (%d) executing [%s]\n%s" % (e.res>>8, e.cmd, e.output))
            self.err = e
        except EnvironmentError, e:
            log.error(str(e))
            self.err = e
            
    def NextState(self):
        if self.err:
            return UpgradeFailed(self.session)
        else:
            return ForceState(self.session)


class ForceState(InstallState):
    """ Force install packages to be downgraded, or dependency problems.
    Assume that we can ignore any dependency probs with this set of pkgs.
    We rely on the fact that each full repository has sufficient dependencies.
    NB: Yum will never support downgrading or force installing packages,
    so this step will be necessary as long as our ISOs have dep conflicts.
    """
    def __action__(self):
        """ Use rpm --force --nodeps to take care of these problem pkgs.
        """
        # TODO
        downlist = self.session.GetPending('downgrade')
        if len(downlist):
            sumlog.info('Using rpm to downgrade %d pkgs' % (len(downlist)))
            insthelper.ForceInstallRpms(self.session.pkgEntry,
                                        self.session.latestlist,
                                        downlist, self.test,
                                        self.session.repackaged,
                                        self.session.progress)
            
        depslist = self.session.GetPending('nodeps')
        if len(depslist):
            sumlog.info('Using rpm to force install %d pkgs' % (len(depslist)))
            insthelper.ForceInstallRpms(self.session.pkgEntry,
                                        self.session.latestlist,
                                        depslist, self.test,
                                        self.session.repackaged,
                                        self.session.progress)

        if len(downlist) or len(depslist):
            self.session.UpdatePendingFromHost()

    #
    # Error handling wrapper. Catch LogCommandError when yum returns non-zero result.
    # Also, if this hangs and user reboots, call esxupdate back and assume the
    # install failed.
    #
    def Action(self):
        try:
            self.__action__()
        except LogCommandError, e:
            log.error("Error (%d) executing [%s]\n%s" % (e.res>>8, e.cmd, e.output))
            self.err = e
        except errors.IntegrityError, e:
            log.error(str(e.msg))
            self.err = e

    def NextState(self):
        #
        # Check what was installed, then proceed
        #
        downs = self.session.GetPending('downgrade')
        nodeps = self.session.GetPending('nodeps')
        if self.err:
            return ForceFailed(self.session)
        #
        # No errors, why packages not taken care of?
        #
        elif not self.test and downs:
            sumlog.info('Following pkgs were not downgraded: %s' % (downs))
            self.err = errors.IncompleteUpgradeError('Not all pkgs downgraded', downs)
            return ForceFailed(self.session)
        
        elif not self.test and nodeps:
            sumlog.info('Following deps were not installed: %s' % (nodeps))
            self.err = errors.IncompleteUpgradeError('Not all deps forced', nodeps)
            return ForceFailed(self.session)

        else:
            return UpgradeState(self.session)

        
class UpgradeState(InstallState):
    """ Invoke the yum program to upgrade existing packages.
    Use yum upgrade so that newer packages that obsolete older ones
    are handled automatically.

    After ForceState, we should not have any pending packages marked downgrade
    or nodeps. 
    """
    def __action__(self):
        if self.test:
            log.info('Test mode - yum will not actually be invoked')

        sumlog.info('Running yum upgrade...')
        
        exceptlist = self.session.GetPending('exclude')
        insthelper.InvokeYum('upgrade', [], exceptlist,
                             self.session.progress, self.test,
                             repackage=self.session.repackaged)
        self.session.UpdatePendingFromHost()

    #
    # Error handling wrapper. Catch LogCommandError when yum returns non-zero result.
    # For dependency errors, add problem packages to the 'nodeps' list for
    # potential force installation.
    #
    def Action(self):
        #
        # Skip yum upgrade if this is a partial repo
        # XXX: Remove this logic when we move to one-tier repos
        #
        if not self.session.GetPending('upgrade'):
            return

        try:
            self.__action__()
        except LogCommandError, e:
            log.error("Error (%d) executing [%s]\n%s" % (e.res>>8, e.cmd, e.output))
            self.err = e
        except errors.YumGenericError, e:
            log.debug('Adding problem packages to nodeps list: %s' % (e.forcelist))
            self.session.MarkPendingNames(e.forcelist, 'nodeps')

    def NextState(self):
        #
        # Check what was installed, then proceed
        #
        upgrades = self.session.GetPending('upgrade')
        nodeps = self.session.GetPending('nodeps')
        if self.err:
            return UpgradeFailed(self.session)
        #
        # Any packages marked nodeps comes from filtering yum output upon a
        # dependency conflict.
        # Treat dep problems as errors.
        #
        elif nodeps:
            if self.session.GetOption('force'):
                log.info('Forcing installation of dependency problem packages %s' % (nodeps))
                return ForceState(self.session)
            else:
                log.error('Yum could not resolve conflicts/deps for %s' % (nodeps))
                self.err = errors.RpmDependencyError( nodeps )
                return RpmDependencyUnmet(self.session)
        #
        # No errors nor dep problems, so why packages not upgraded?
        #
        elif upgrades:
            sumlog.info('yum upgrade failed for %s' % (upgrades))
            self.err = errors.IncompleteUpgradeError('Yum did not upgrade some pkgs',
                                                     upgrades)
            return UpgradeFailed(self.session)
        #
        # Any new packages not on the host?
        #
        else:
            return YumInstallState(self.session)


class YumInstallState(InstallState):
    """ Invoke yum install <pkgs>.  This takes care of new packages not on
    the system, except for dependencies pulled in by yum upgrade.
    """
    def __action__(self):
        if self.test:
            log.info('Test mode - yum will not actually be invoked')

        # TODO: order packages using bundleGroups
        instlist = self.session.GetPending('install')
        if instlist:
           sumlog.info('Running yum install <%d packages>...' % (len(instlist)))
        
           exceptlist = self.session.GetPending('exclude')
           insthelper.InvokeYum('install', instlist, exceptlist,
                                self.session.progress, self.test,
                                repackage=self.session.repackaged)
           self.session.UpdatePendingFromHost()

    def Action(self):
        try:
            self.__action__()
        except LogCommandError, e:
            log.error("Error (%d) executing [%s]\n%s" % (e.res>>8, e.cmd, e.output))
            self.err = e
        except errors.YumGenericError, e:
            log.debug('Adding problem packages to nodeps list: %s' % (e.forcelist))
            self.session.MarkPendingNames(e.forcelist, 'nodeps')

    def NextState(self):
        #
        # Check what was installed, then proceed
        #
        instlist = self.session.GetPending('install')
        nodeps = self.session.GetPending('nodeps')
        if self.err:
            return UpgradeFailed(self.session)
        #
        # Any packages marked nodeps comes from filtering yum output upon a
        # dependency conflict.
        # Treat dep problems as errors.
        #
        elif nodeps:
            if self.session.GetOption('force'):
                log.info('Forcing installation of dependency problem packages %s' % (nodeps))
                return ForceState(self.session)
            else:
                log.error('Yum could not resolve conflicts/deps for %s' % (nodeps))
                self.err = errors.RpmDependencyError( nodeps )
                return RpmDependencyUnmet(self.session)
        #
        # By this point, we know that upgrades == downgrades == nodeps == []
        # So the only failure is if not all the packages we want installed
        # got installed.
        #
        elif instlist:
            sumlog.info('Yum failed to install new pkgs %s' % (instlist))
            self.err = errors.IncompleteUpgradeError('Yum did not install some new pkgs',
                                                     instlist)
            return UpgradeFailed(self.session)
        else:
            return PostState(self.session)


class PostState(InstallState):
    """ Perform any post-install procedures """

    def Action(self):
        self.session.progress.StartStep('Post-installation')
        flags = self.session.flags
        haMode = self.session.GetOption('hostagent')
        try:
            #
            # XXX: This is a hack for the vmkdrivers check-in
            #      mkinitrd fails since VMware-esx package is installed
            #      way too early, before other drivers rpms.
            #      Remove when proper dependencies have been created
            #      for the vmkdrivers.  See PR 74082.
            #
            if not self.test and len(self.session.installed) > 0:
                if os.path.isfile(PCIID_SUPPRESS_FILE):
                    log.info('Running esxcfg-pciid to update the PCI ID tables...')
                    LogCommand('rm -f %s' % (PCIID_SUPPRESS_FILE))
                    LogCommand('esxcfg-pciid')

                # Check whether the configuration file is locked by active
                # process.  If it is, check it again after 30 seconds and try
                # 10 times.
                sec = 30
                retry = 10
                if insthelper.LockfileTimeout('/etc/vmware/esx.conf.LOCK', sec,
 retry):
                    log.error("/etc/vmware/esx.conf is locked by another "
                              "process")
                    log.error("Need to re-running esxcfg-boot -p")
                    self.err = "%d seconds timeout exceeded checking" % \
                               (sec*retry) + \
                               " lock on configuration file"
                else:
                    log.info('Running esxcfg-boot to regenerate initrds...')
                    LogCommand('esxcfg-boot -p')

                if not haMode and flags.RestartHostd() and \
                   not flags.IsRebootRequired():
                    log.info('Restarting hostd...')
                    LogCommand("NOCHVT=1 /etc/init.d/mgmt-vmware restart")
        except LogCommandError, e:
            log.error("Error (%d) executing [%s]\n%s" % (e.res>>8, e.cmd, e.output))
            self.err = e

    def NextState(self):
        #
        # If we didn't error out, the default is to reboot, unless:
        #  1) --noreboot option was specified on cmd line
        #  2) The RebootRequired install flag is false
        #
        flags = self.session.flags
        haMode = self.session.GetOption('hostagent')
        
        if self.err:
            return PostInstallFailed(self.session)
        elif flags.IsRebootRequired() and len(self.session.installed) > 0:
            if self.session.GetOption('noreboot'):
                return SuccessNeedsRebootState(self.session)
            else:
                return RebootState(self.session)
        elif haMode and flags.RestartHostd():
            return SuccessNeedsHostdRestart(self.session)
        else:
            return SuccessState(self.session)


class RollbackState(InstallState):
    """ Roll back this current transaction """
    def __action__(self):
        #
        # TODO: If new RPMs were installed but the older ones have not been
        # deleted yet, we may need to delete them ourselves
        #

        #
        # Convert install time to Internet date format and roll back
        # all RPM transactions back to that time
        #
        rollbacktime = self.session.GetTXStartTime()
        if rollbacktime == 0:
            raise errors.RollbackError('Missing initial install time to roll back to')

        options = '-Uv'
        if self.session.GetOption('force'):
            options += ' --force'
            
        tstr = strftime('%d %b %Y %H:%M:%S', localtime(rollbacktime))
        cmd = "rpm %s --rollback '%s'" % (options, tstr)
        out = LogCommand(cmd, returnOutput = 1, loglevel = logging.INFO,
                         timeout = insthelper.RPM_TIMEOUT)

        #
        # rpm --rollback has the annoying habit (4.2.3-*) of returning 0
        # even if the rollback doesn't happen.  We parse the output, using
        # a fixed string from rpmlib/rpminstall.c, and raise an error
        # if the rollback transaction did not take place.
        #
        # TODO: More ideal is calling the rollback method in rpm.ts.  We
        # might have to replicate most of the work from rpmRollback() in
        # rpminstall.c from rpmlib.   We would also have to set up
        # our own callbacks (or borrow yum's)
        #
        # TODO: Check that all RPMs on installed list are rolled back
        # The complicated part is RPM prints out the N-V-R and we don't
        # have the original NVR easily available.
        #
        found = 0
        for line in out.splitlines():
            if line.startswith('Rollback packages'):
                found = 1
        if not found:
            raise LogCommandError(cmd, 0, 0, out)

        #
        # Restore saved system files
        #
        backupfile = SPOOL_DIR_PATH + '/' + SPOOL_DIR + '/' + SYSTEM_BACKUP_FILE
        insthelper.RestoreSystemFiles(backupfile)               

    def Action(self):
        try:
            self.__action__()
        except errors.RollbackError, e:
            log.error('Rollback failed - %s' % (e.msg))
            self.err = e
        except LogCommandError, e:
            log.error("Error (%d) executing [%s]\n%s" % (e.res>>8, e.cmd, e.output))
            self.err = e

    def NextState(self):
        if self.err:
            return RollbackFailed(self.session)
        else:
            return RollbackDone(self.session)
        


########################################################################
#
# Terminal install states
#

#
# Save install record to patchdb before reboot
#
class RebootState(InstallState):
    """ Initiate system reboot after installation. """
    def Action(self):
        self.session.SaveInstallRecord()
        insthelper.Cleanup()
        self.session.LogSummary()
        self.session.progress.Done()
        log.info('Install succeeded - please standby for reboot.')
        Reboot()

    def IsDone(self):   return 1


class PreInstDoneState(InstallState):
    """ Save session state and re-start esxupdate to continue install """
    def Action(self):
        insthelper.Cleanup()
        try:
            self.session.SaveForRestore()
            insthelper.RestartProcess()
        except EnvironmentError, e:
            #
            # Esxupdate could not be exec'ed, or we couldn't save session.
            # Not a fatal error; we can continue the install, but with
            # the current code instead of the new esxupdate code.
            #
            log.warn('Cannot save session state or exec Esxupdate: %s' % (e.msg))
            log.warn('Proceeding with current esxupdate code...')
            self.err = e
        
    def IsDone(self):
        if self.err:  return False
        else:         return True

    def NextState(self):
        return RemoveState(self.session)


############################################################################
# Failure states (a subset of the terminal states)

class FailureState(InstallState):
    """ Superclass for all failure states. Should not be instantiated. """
    def Action(self):
        insthelper.Cleanup()
        if hasattr( self, "FailLogAction" ) and callable( self.FailLogAction ):
            self.session.LogSummary()
            self.FailLogAction()
    def IsDone(self):   return 1


class ConfigFailed(FailureState):
    """ Error during ConfigState or reboot after ConfigState crash """


class PrepFailed(FailureState):
    """ Error during PrepState or reboot after PrepState crash """

class DownloadFailed(FailureState):
    """ Error downloading RPMs or generating yum headers """

class TestFailed(FailureState):
    """ Error during test transaction """

class ForceFailed(FailureState):
    """ Error during ForceState or reboot after rpm hang"""
    def FailLogAction(self):
        log.error('A failure occurred while downgrading packages.')
        log.error('Check /var/log/vmware/esxupdate.log for details.')
        

class UpgradeFailed(FailureState):
    """Error during UpgradeState or YumInstallState or reboot after yum crash"""
    def FailLogAction(self):
        log.error('A failure occurred while installing or removing packages.')
        try:
            log.error(self.session.GetLastErr().errout)
        except:
            pass
        log.error('Check /var/log/vmware/esxupdate.log for details.')
        

class RpmDependencyUnmet(FailureState):
    """ Conflict or RPM dependency failure """
    def FailLogAction(self):
        log.error('Please uninstall the third party RPM or install a version')
        log.error('of it that does not have a dependency conflict, and '
                  'rerun esxupdate.' )


class PostInstallFailed(FailureState):
    """ Error during PostState or reboot after PostState crash """
    def Action(self):
        self.session.SaveInstallRecord()
        FailureState.Action(self)

    def FailLogAction(self):
        log.error('Package installation has finished, but a failure '
                  'occurred during system configuration.  The system '
                  'may not be able to boot.  Check '
                  '/var/log/vmware/esxupdate.log for details.')
        

class RollbackFailed(FailureState):
    """ Error during RollbackState """
    def Action(self):
        pass



############################################################################
# Success states (a subset of the terminal states)

class SuccessState(InstallState):
    """ The ultimate measure of success """
    def Action(self):
        self.session.SaveInstallRecord()
        insthelper.Cleanup(removeRpms=True)
        self.session.LogSummary()
        self.session.progress.Done()
        self._logMessage()

    def _logMessage(self):
        log.info('Install of %s succeeded.' % (
            self.session.OrderedBundleList()) )

    def IsDone(self):   return 1


class TestCompletedState(SuccessState):
    """ --test was used. Don't save anything to PatchDB. """
    def Action(self):
        insthelper.Cleanup()
        self.session.LogSummary()
        self.session.progress.Done()
        log.info('Test install of %s succeeded.' % (
            self.session.OrderedBundleList()) )
        if 'hostagent' not in self.session.options:
            cmdline.PrintTestReport(self.session.results,
                                    self.session.OrderedBundleList())

    def IsDone(self):   return 1


class SuccessNeedsRebootState(SuccessState):
    """ The RPM packages installed, but we still need to be rebooted """
    def _logMessage(self):
        log.info('Install of %s is almost done.' % (
            self.session.OrderedBundleList()) )
        log.info("Please reboot the ESX Server by typing 'reboot' "
                 "to complete the update.")
        log.info("Until a reboot is performed, any attempts to restart "
                 "management agents may fail.")


class SuccessNeedsHostdRestart(SuccessState):
    """ Success, but hostd needs to be restarted """


class RollbackDone(InstallState):
    """ Rollback succeeded - save state info """
    def Action(self):
        relstr = self.session.desc.GetRelease()
        #
        # Obsolete current entry so it's no longer installed and
        # disallow future rollbacks.
        #
        self.session.repackaged = 0
        self.session.obsolete = 1
        self.session.SaveForRestore()
        if self.session.db.HasEntry(relstr):
            self.session.SaveInstallRecord()

        sumlog.info('Rollback of %s completed' % (relstr))

    def IsDone(self):   return 1
    
