#######################################################################
# Copyright (C) 2006 VMWare, Inc.
# All Rights Reserved
########################################################################
#
# A simple progress-tracking class
#

#
# Callback function prototype as well as the no-op default callback,
#
def DefaultProgressCB(done_steps, total_steps, stepname, sub_done, sub_total):
   pass


class ProgressBar:
   """ Basic class for measuring progress as a series of steps with tracking of
   progress for each individual step.  The total is kept >= the step number.
   Completion is marked by a callback with the step name set to DONE_STEPNAME,
   and done_steps equal to total_steps.

   The callback can power a GUI with either one continuous progress bar, or
   separate step / substep (or package/bytes) progress bars.

   Example, with 5 steps, 100 substeps:
   StartStep(1): done 0/5 steps, 0/100 substeps
   ...
   DoneStep(1):  done 0/5 steps, 100/100
                 done 1/5 steps, 0/100
   ...
   DoneStep(5):  done 5/5 steps, 0/100
   """

   DONE_STEPNAME = "Done"

   def __init__(self, progCallback=DefaultProgressCB, total_steps=0):
      """ Create a new instance of progressbar.
      progCallback   - a function (done_steps, total_steps, msg, sub_done, sub_total)
                       called whenever the progress bar is updated.
      total_steps    - Set a total # of steps for the process
      """
      assert callable(progCallback), "First arg %s is not callable" % type(progCallback)
      self.cb = progCallback
      self.__total_steps = total_steps
      self._setMarker(step_num=1, sub_done=0, sub_total=100, name="")

   def SetTotalSteps(self, total_steps):
      self.__total_steps = total_steps

   def _setMarker(self, step_num=None, sub_done=None, sub_total=None, name=None):
      """ Computes done_steps; wraps sub_done back to 0% when a step is done;
      ensures total_steps does not fall behind.
      No callback is done.  1 is the first step number.
      ValueError is thrown if step_num is less than 1.
      """
      if name is not None:      self.stepname = name
      if sub_done is not None:  self.sub_done = sub_done
      if sub_total is not None: self.sub_total = max(sub_total, 1)
      #
      # Assume that this step is not done but all previous ones are done
      if step_num is not None:
         if step_num < 1:    raise ValueError, "step_num must be at least 1"
         self.done_steps = step_num - 1
         self.__total_steps = max(self.__total_steps, step_num)
      #
      # Round up done_steps if we've done all sub steps
      if self.sub_done >= self.sub_total:
         self.done_steps += 1
         self.__total_steps = max(self.__total_steps, self.done_steps)
         self.sub_done = 0
      
   def StartStep(self, stepname, sub_total=100, step_num=None):
      """ Start the next step with name stepname, optionally skipping to step_num
      first, where 1 is the first step.
      """
      self._setMarker(step_num, 0, sub_total, stepname)
      self.cb(self.done_steps, self.__total_steps, stepname, 0, self.sub_total)

   def DoneStep(self, stepname=None, step_num=None):
      """ Step is done, callback with subtotal=100%, then callback with done_steps
      incremented.  Optionally, finish at step step_num,
      where 1 is the first step.
      """
      self._setMarker(step_num, name=stepname)
      self.cb(self.done_steps, self.__total_steps, self.stepname,
              self.sub_total, self.sub_total)
      
      self._setMarker(sub_done=self.sub_total)
      self.cb(self.done_steps, self.__total_steps, self.stepname,
              self.sub_done, self.sub_total)

   def Update(self, sub_step, sub_total=None):
      """ Update progress for an individual step. Done is sub_step == sub_total  """
      self._setMarker(sub_done=sub_step, sub_total=sub_total)
      self.cb(self.done_steps, self.__total_steps, self.stepname,
              self.sub_done, self.sub_total)

   def Done(self):
      """ Mark all steps as complete. """
      self.done_steps = self.__total_steps
      self.sub_done = 0
      self.cb(self.done_steps, self.__total_steps, self.DONE_STEPNAME,
              0, self.sub_total)
