#######################################################################
# Copyright (C) 2007 VMWare, Inc.
# All Rights Reserved
########################################################################
#
#    cmdline.py  -- command line output routines
#
# TODO: Move insthelper output formatting routines here
#
import insthelper

#
# Scan summary output format:
# BundleName---  ApplicableFlags Summary installFlags
#  (optional) Explanations
#
SCAN_OUTPUT_FORMAT = "%-20.20s %-8s %-40.40s %-5s\n"

#
# iFlags: [.r][.m][.h]
#    r = this bundle requires reboot after install
#    m = this bundle requires Maintenance Mode
#    h = hostd must be restarted
#
def FormatInstallFlags(desc):
   """ Formats the installation flags of a bundle into a very
   short fixed-width string, and return it.
   """
   uPaths = desc.GetUpgradePaths()
   s = ['-', '-', '-']
   if uPaths.IsRebootRequired():
      s[0] = 'r'
   if uPaths.IsMModeRequired():
      s[1] = 'm'
   if uPaths.RestartHostd():
      s[2] = 'h'
      
   return ''.join(s)


#
# appFlags:  irmcoNdv or -------- if applicable
#    i = bundle already installed
#    r = required depot bundle not applicable or missing system states
#    m = required bundle missing from depot or from bundle selection
#    c = conflict with another bundle
#    o = bundle obsolete
#    N = iNtegrity error
#    d = disk space error
#    v = VMs are on but required to be off
#
def FormatAppFlags(scanresult, vmsOff=False):
   i = r = m = c = o = N = d = v = '-'
   applicable = scanresult['applicable']
   if scanresult['installed']:
      i = 'i'
   if not applicable and scanresult['availableIDs']:
      r = 'r'
   if not applicable and scanresult['missingIDs']:
      m = 'm'
   if not applicable and scanresult['conflictIDs']:
      c = 'c'
   if scanresult['obsolete']:
      o = 'o'
   if not scanresult['integritySuccess']:
      N = 'N'
   if scanresult['noDiskSpace']:
      d = 'd'
   if scanresult['reqVMoff'] and not vmsOff:
      v = 'v'
   return i + r + m + c + o + N + d + v


def FormatScanTitles():
   s = SCAN_OUTPUT_FORMAT % ('- Bundle Name -'.center(20),
                             'AppFlags',
                             '--- Summary ---'.center(40),
                             'iFlags')
   return s


def FormatScanOneLine(scanresult, vmsOff=False):
   desc = scanresult['desc']
   s = SCAN_OUTPUT_FORMAT % (scanresult['bundleID'],
                             FormatAppFlags(scanresult, vmsOff),
                             desc.GetSummary(),
                             FormatInstallFlags(desc)
                             )
   return s


def FormatNotApplicableExplanations(scanresult, vmsOff=False):
   explanations = scanresult['explanations'][:]
   bundleID = scanresult['bundleID']
   if not scanresult['integritySuccess']:
      msg = '[%s] did not pass the integrity check! Details:\n' % (bundleID)
      explanations.append(msg + scanresult['integrityDescription'])
   if scanresult['noDiskSpace']:
      msg = '[%s] does not have enough disk space for installation.' % (
         bundleID)
      explanations.append(msg)
   if scanresult['reqVMoff'] and not vmsOff:
      msg = '[%s] cannot be installed when VMs are still running.' % (
         bundleID)
      explanations.append(msg)

   if len(explanations):
      return '  ' + '\n  '.join(explanations) + '\n'
   else:
      return ''


def PrintScanSummary(scanHash, explain=False):
   """ Prints a pretty formatted summary of the scan results.
   If explain is true, below the summary line of each bundle,
   print the reasons why a bundle is not applicable.
   """
   vmsOff = not insthelper.HaveRunningVMs()
   s = ''
   s += FormatScanTitles()
   scanIDs = scanHash.keys()
   scanIDs.sort()
   for key in scanIDs:
      bundle = scanHash[key]
      s += FormatScanOneLine(bundle, vmsOff)
      if explain:
         s += FormatNotApplicableExplanations(bundle, vmsOff)

   print s


def FormatBundleReport(scanresult, installBundles, vmsOff=False):
   bundleID = scanresult['bundleID']
   s = '----- %-20s -----: ' % (bundleID)
   if bundleID in installBundles:
      s += 'installed\n'
   else:
      s += 'not installed\n'
      s += FormatNotApplicableExplanations(scanresult, vmsOff)
      
   for pkg, newerID in scanresult['newerRpms'].items():
      s += '  Package %s superseded by bundle %s\n' % (pkg, newerID)

   return s


def PrintTestReport(scanResults, installBundles):
   """ Prints out a report of every scanned bundle, whether it
   will be installed or not, if not, the explanations; finally,
   which RPMs were obsoleted (and by which bundles).

   installBundles - list of bundles to be installed
   """
   vmsOff = not insthelper.HaveRunningVMs()
   print '== The esxupdate --test report =='
   for result in scanResults.values():
      s = FormatBundleReport(result, installBundles, vmsOff)
      print s.rstrip()
