#######################################################################
# Copyright (C) 2005 VMWare, Inc.
# All Rights Reserved
########################################################################
#
# Exceptions module for esxupdate
#

class Codes:
    NOT_ROOT = 1        # Not the root user
    BAD_ARGS = 2        # Invalid command line syntax
    BAD_STATE = 4       # Unexpected terminal state
    IO       = 5        # General I/O error
    NOYUM    = 6        # No yum.conf or no yum
    LOCK     = 7        # Cannot acquire lock
    DEPOT    = 10       # Cannot access bundle depot
    DOWNLOAD = 11       # Download error or file not in cache
    BAD_META = 12       # Malformed descriptor
    BAD_DB   = 13       # Bad PatchDB / Host scanning error
    ROLLBACK = 14       # Error during rollback
    NO_SUCH_REL = 20    # No such release in database
    NO_BACKUPS = 21     # Cannot rollback: no backup data saved
    BAD_PATH = 31       # Trying to upgrade from invalid host version
    VMS_ON   = 32       # VMs not shut down or in maintenance mode
    DISK_FULL = 33      # Not enough space to install bundle
    RPM_DEP  = 34       # RPM dependency error
    NO_CHAIN = 35       # Previous install requires reboot and is not chainable
    INTEGRITY = 37      # Signature issues: some file didn't check out
    NOT_APP  = 40       # General error for not applicable bundle(s)
    REQ_BUND = 41       # Missing pre-requisite bundles
    SUP_BUND = 42       # Superceding bundle(s) is/are already installed
    CONF_BUND = 43      # Conflicting bundle(s) is/are already installed
    SHELL    = 50       # Install error: running shell command
    GENERIC  = 51       # Install error: generic error
    YUM      = 52       # Error using yum to install/upgrade pkgs
    RPM      = 53       # Error using rpm to downgrade/force pkgs
    POST     = 54       # Error executing post-install commands
    NEEDS_REBOOT = 80   # Installation succeeded, but system needs reboot
    HOSTD_RESTART = 81  # Installation succeeded, but hostd needs to restart


class EsxupdateError(Exception):
    """ Umbrella class for all esxupdate errors """

class MsgBundleFileReporter(EsxupdateError):
    """ An interface for any esxupdate error that needs to report it's msg,
    bundleID, and filename.  This is fairly common.  Just inherit from this
    class to gain the Report method """
    def Report(self, log, halog):
        log.error( self.msg )
        halog.error( 'BundleID:%s' % self.bundleID )
        halog.error( 'File:%s' % self.filename )


class MsgReporter(EsxupdateError):
    """ A base class for reporting a single message """
    def __init__(self, msg):
        self.msg = msg
        self.args = (msg)
        
    def Report(self, log, halog):
        log.error( self.msg )


class NoYumError(MsgReporter):
    """ yum binary or yum.conf is missing """
    def __init__(self):
        MsgReporter.__init__(self, "yum not installed or yum.conf missing "
                             "- please reinstall yum and retry.")
        self.exitCode = Codes.NOYUM

class NotApplicableError(MsgReporter):
    """ Generic reason why bundle cannot be installed """
    def __init__(self, msg):
        MsgReporter.__init__(self, msg)
        self.exitCode = Codes.NOT_APP

class PatchDependencyError(NotApplicableError):
    """ Umbrella class for patch to patch dep errors """
    def __init__(self, bundleID, IDList=None, msg='', explanations=[]):
        self.bundleID = bundleID
        self.IDList = IDList
        if explanations:
            self.explanations = explanations
            msg = '\n '.join(explanations)
        NotApplicableError.__init__(self, msg)
    def Report(self, log, halog):
        log.error( self.msg )
        halog.error( 'BundleID:%s' % self.bundleID )

class IsObsoleteError(PatchDependencyError):
    def __init__(self, bundleID, supersedingIDs):
        msg = "Bundle %s is obsoleted by %s" % (bundleID, supersedingIDs)
        PatchDependencyError.__init__(self, bundleID, supersedingIDs, msg=msg)
        self.exitCode = Codes.SUP_BUND

class AlreadyInstalledError(PatchDependencyError):
    def __init__(self, bundleID):
        msg = "Bundle %s is already installed.  Nothing "\
              "for esxupdate to do." % (bundleID)
        PatchDependencyError.__init__(self, bundleID, msg=msg)
        self.exitCode = 0
        
class PatchRequiresError(PatchDependencyError):
    def __init__(self, bundleID, requireIDs, explanations=[]):
        msg = "Bundle %s needs bundles %s" % (bundleID, requireIDs)
        PatchDependencyError.__init__(self, bundleID, requireIDs,
                                      msg, explanations)
        self.exitCode = Codes.REQ_BUND
    def Report(self, log, halog):
        log.error( self.msg )
        halog.error( 'BundleID:%s' % self.bundleID )
        halog.error( 'Requires:%s' % self.IDList )

class ConflictsError(PatchDependencyError):
    def __init__(self, bundleID, conflictingIDs, explanations=[]):
        msg = "Bundle %s conflicts with bundles %s" % (bundleID, conflictingIDs)
        PatchDependencyError.__init__(self, bundleID, conflictingIDs,
                                      msg, explanations)
        self.exitCode = Codes.CONF_BUND
    def Report(self, log, halog):
        log.error( self.msg )
        halog.error( 'BundleID:%s' % self.bundleID )
        halog.error( 'Conflicts:%s' % self.IDList )

class IncompleteUpgradeError(EsxupdateError):
    """ Not all packages were installed from one of the Force, Upgrade, or
    Install states.  This error holds a message and the list of packages that
    should have been completed at that step.
    """
    def __init__(self, msg, pkglist):
        """ pkglist should be in NameArch format. """
        self.msg = msg
        self.pkglist = pkglist
        
class DiskSpaceError(MsgReporter):
    """ Disk space check failed """
    def __init__(self, msg):
        MsgReporter.__init__(self, msg)
        self.exitCode = Codes.DISK_FULL

class YumGenericError(EsxupdateError):
    """ Dependency missing, conflict, or some other case when yum returns 0 """
    def __init__(self, yumcmd, errout, forcelist):
        self.cmd = yumcmd
        self.errout = errout
        self.forcelist = forcelist
        
class TimeoutError(EsxupdateError):
    def __init__(self, msg, timeout):
        self.msg = msg
        self.timeout = timeout

class RollbackError(EsxupdateError):
    """ Unable to roll back last transaction """
    def __init__(self, msg):
        self.msg = msg

class VMsRunningError(NotApplicableError):
    """ VMs not shut down or in maintenance mode """
    def __init__(self):
        self.exitCode = Codes.VMS_ON
    def Report(self, log, halog):
        log.error( 'This bundle requires the host to be in maintenance '
                   'mode.  Since the host is not in maintenance mode, '
                   'esxupdate cannot proceed. The VMs need to be turned '
                   'off or migrated to another host first.' )

class RpmDependencyError(EsxupdateError):
    """ An RPM dependency was not met """
    def __init__(self, nodeps=[], msg=''):
        Exception.__init__(self, nodeps, msg)
        self.msg = msg
        self.nodeps = nodeps

class DepotAccessError(EsxupdateError):
    """ An error occurred while trying to access the Depot """
    def __init__(self, msg):
        Exception.__init__(self, msg)
        self.msg = msg
        self.exitCode = Codes.DEPOT
    def Report(self, log, halog):
        log.error(self.msg)

class DepotDownloadError(EsxupdateError, MsgBundleFileReporter):
    """ An error occurred while downloading a file from the Depot """
    def __init__(self, msg, bundleID, filename, *extraArgs):
        Exception.__init__(self, msg, bundleID, filename, extraArgs)
        self.msg = msg
        self.bundleID = bundleID
        self.filename = filename 
        self.extraArgs = extraArgs
        self.exitCode = Codes.DOWNLOAD
    def Report(self, log, halog):
        MsgBundleFileReporter.Report(self, log, halog)
        halog.error('Extra:%s' % str(self.extraArgs))

class BadDepotDescriptorError(EsxupdateError, MsgBundleFileReporter):
    """ The descriptor file in the depot could not be parsed, was corrupt, 
        or had an illegal value"""
    def __init__(self, bundleID, filename, msg="" ):
        Exception.__init__(self)
        self.bundleID = bundleID
        self.filename = filename
        self.msg = msg
        self.exitCode = Codes.BAD_META

class BadDepotMetadataError(EsxupdateError, MsgBundleFileReporter):
    """ The metadata (contents.xml, etc) in the depot could not be 
        parsed, was corrupt, or had an illegal value"""
    def __init__(self, bundleID, filename, msg="" ):
        Exception.__init__(self)
        self.bundleID = bundleID
        self.filename = filename
        self.msg = msg
        self.exitCode = Codes.BAD_META

class IntegrityError(EsxupdateError, MsgBundleFileReporter):
    """ Some file in the depot didn't pass GPG signature verification """
    def __init__(self, bundleID, filename, msg="" ):
        Exception.__init__(self, bundleID, filename, msg)
        self.bundleID = bundleID
        self.filename = filename
        self.msg = msg
        self.exitCode = Codes.INTEGRITY

class FileError(EsxupdateError):
    """An error related to the local filesystem (usually an OSError)"""
    def __init__(self, filename, msg="" ):
        Exception.__init__(self)
        self.filename = filename
        self.msg = msg
        self.exitCode = Codes.IO
    def Report(self, log, halog):
        log.error(self.msg)
        halog.error( 'File:%s' % self.filename )

