#######################################################################
# Copyright (C) 2006 VMWare, Inc.
# All Rights Reserved
########################################################################
#
#   scan.py
#
#      Logic for scanning depots and determining which bundles
#      are applicable, have been installed, etc.
#


import logging
import insthelper
import fnmatch
import re
from copy import copy, deepcopy

log = logging.getLogger( 'scan' )

class DBScanError(Exception): pass #should bubble up as error code 13

#
# global, the descriptor list from the last scan
lastDesclist = []
# dictionary to keep track of what bundles were visited when resolving
# dependencies
dependencies_visited = dict() 
#=============================================================================
#python 2.2 doesn't have set()
if not hasattr(__builtins__, 'set'):
	class set(dict):
		"""This is the most ghetto set implementation ever.  In this
		module, we only use add() and update() and iterate"""
		def add(self, item):
			self[item] = 1

def dictFromList(theList):
        return dict([(val, 1) for val in theList])

# ----------------------------------------------------------------------------
def ScanDepot(depot, patchDB, scanDepotOnly=False, groupMode=False,
              rollup=None, baseline=[]):
	"""Perform a scan of the depot, returning the scan results in a giant
	hash of hashes.

	Args:
	   depot - Depot instance   - the depot containing the bundles to scan
	   patchDB - PatchDB instance - what bundles are installed on this host
	   scanDepotOnly   - see description in ProcessDependencies()
	   groupMode - If false, gives scan results assuming that only
	      one bundle will be installed at a time. Only installed
	      bundles will satify "requires"; only installed bundles
	      matter for "conflicts". If true, assumes all the bundles
	      in the depot scanlist are to be installed together in one
	      transaction. "requires" of bundles not installed but in
	      the scanlist are OK; two conflicting bundles (one
	      conflicts w/ the other) both become inapplicable.

	Returns:
	   ['bundleID1' : scanResHash, 'bundleID2' : scanResHash2, ... ]
	   Each scanResHash contains many fields detailing the applicability,
	   dependencies, and install characteristics.  See the EsxupdateAPI wiki
	   for more details.
	"""
	#
	# deepcopy must be used here; otherwise, a shallow copy means
	# ScanDepot() will accidentally modify the original dicts of
	# each bundle in depot.bundles.  One nasty bug that manifests
	# itself is that bundles marked as inapplicable cannot be
	# re-evaluated to be applicable, since there is "memory" of the
	# inapplicable state in depot.bundle's copy of the bundledict.
	#
	# In the case of rollup, we filter the list of bundles that we copy to
	# include only those that are listed in the rollup descriptors
	# dependency list, and, of course, the rollup bundle itself.  This
	# permits us to avoid pulling in other bundles that might be in the
	# depot in order to satisfy dependencies, without having to drastically
	# change our scan logic.
	#
	if rollup:
		rollupids = depot.bundles[rollup]['desc'].deps.requiredIDs
		rollupids.append(rollup)
		depotBundles = dict()
		for id in rollupids:
			for key in FindKeysByPatternList(depot.bundles, [id]):
				depotBundles[key] = deepcopy(depot.bundles[key])
	elif baseline:
		depotBundles = dict()
                def addReqIDs(depotBundles, key):
                        """append specified bundle by 'key' and its req bundles 
                        recursively on 'depotBundles'
                        """
                        if depotBundles.has_key(key):
                                return
                        depotBundles[key] = deepcopy(depot.bundles[key])
			reqIds = depot.bundles[key]['desc'].deps.requiredIDs
			for match in FindKeysByPatternList(depot.bundles, reqIds):
                                addReqIDs(depotBundles, match)
		# add bundles specified on the command line reqursively
		for key in baseline:
                        addReqIDs(depotBundles, key)
                log.debug("depotBundles: %s" % depotBundles.keys())
	else:
		depotBundles = deepcopy( depot.bundles )
	installedBundles = {}

	try:
		installedPatches = patchDB.GetInstalledPatches(getObsolete=True)
		FillInstalledBundles( installedBundles, 
		                      installedPatches ) 
	except AssertionError, e:
		raise
	except Exception, e:
		logging.exception(e)
		raise DBScanError( "DB Scan Error: %s" % e )

	FillDepotBundles( depotBundles, installedBundles )

	if groupMode:
		reqFilter = GroupReqFilter
		confFilter = GroupConfFilter
	else:
		reqFilter = IndividualReqFilter
		confFilter = IndividualConfFilter
	global dependencies_visited
	dependencies_visited = dict()
        try:
                ProcessDependencies( depotBundles, installedBundles,
                                     reqFilter, confFilter, scanDepotOnly, rollup )
        except Exception, e:
                raise DBScanError( "DB Scan Error: %s" % e )

	return depotBundles

# ----------------------------------------------------------------------------
def FindKeysByPatternList( srcDict, patternList ):
	foundKeys = []
	srcKeys = srcDict.keys()
	for targetPattern in patternList:
		#use fnmatch here in case there are wildcards
		foundKeys += fnmatch.filter(srcKeys, targetPattern)
	return foundKeys
		

# ----------------------------------------------------------------------------
def FillInstalledBundles( installedBundles, installedPatches ):

	for key, installedPatch in installedPatches.items():
		bundle = {}

		if installedPatch['desc']:
			bundle['desc'] = installedPatch['desc']
		else:
			log.warn("bundleID %s has invalid desc. Skipping" % key)
			continue

		uPaths = installedPatch['desc'].GetUpgradePaths()
		assert uPaths, "GetUpgradePaths should never return None"

		bundle['bundleID'] = key
		bundle['installed'] = True
		bundle['noDiskSpace'] = False

		bundle['reqIDs'] = uPaths.GetRequiredIDs()
		bundle['sysStateDeps'] = uPaths.GetSysStateDeps()
		bundle['systemStates'] = GetSystemStates(installedPatch)
		bundle['obsoleteIDs'] = uPaths.GetObsoletedIDs()
		bundle['reqReboot'] = uPaths.IsRebootRequired()
		bundle['restartHostd'] = uPaths.RestartHostd()
		bundle['reqVMoff'] = uPaths.IsMModeRequired()
		bundle['integritySuccess'] = True
		bundle['integrityDescription'] = ""

		bundle['obsolete'] = installedPatch.get('obsolete', False)
		bundle['conflictIDs'] = set()
		bundle['conflictDeps'] = []
		bundle['missingIDs'] = []
		bundle['missingDeps'] = []
		bundle['availableIDs'] = []
		bundle['newerAvailable'] = set()
		bundle['newerRpms'] = {}
		bundle['visited'] = set({key: 1})

		bundle['applicable'] = True

		installedBundles[key] = bundle


# ----------------------------------------------------------------------------
def FillDepotBundles( depotBundles, installedBundles ):
	""" Fills out the information in the depotBundles dicts """
	
	for key, bundle in depotBundles.items():
		uPaths = bundle['desc'].GetUpgradePaths()

		bundle['bundleID'] = key
		bundle['reqIDs'] = uPaths.GetRequiredIDs()
		bundle['sysStateDeps'] = uPaths.GetSysStateDeps()
		bundle['obsoleteIDs'] = uPaths.GetObsoletedIDs()
		bundle['reqReboot'] = uPaths.IsRebootRequired()
		bundle['restartHostd'] = uPaths.RestartHostd()
		bundle['reqVMoff'] = uPaths.IsMModeRequired()
		bundle['installed'] = key in installedBundles
		bundle['noDiskSpace'] = not EnoughDiskSpace( bundle )
		bundle['integritySuccess'] = bundle.get('integritySuccess',
							False)
		integrityErr = bundle.get('integrityDescription', "")
		bundle['integrityDescription'] = integrityErr
		if key in installedBundles:
			bundle['systemStates'] = installedBundles[key] \
						 ['systemStates']
		else:
			bundle['systemStates'] = []
		
		bundle['obsolete'] = False
		bundle['conflictIDs'] = set()
		bundle['conflictDeps'] = []

		bundle['missingIDs'] = []
		bundle['missingDeps'] = [] #TODO: no implementation yet
		bundle['availableIDs'] = []
		bundle['newerAvailable'] = set()
		bundle['newerRpms'] = {}
		bundle['visited'] = set({key: 1})

		bundle['applicable'] = None


# ----------------------------------------------------------------------------
def ProcessDependencies( depotBundles, installedBundles,
			 reqFilterFunc, confFilterFunc,
			 scanDepotOnly=False, rollup=None ):
	"""
	scanDepotOnly   - include only the depot bundles when calculating
	                  obsolescence. It means installed bundles cannot
			  obsolete any depot bundles.
	"""
	installedBundleList = installedBundles.keys()
	#
	# compute a dict with both installed & depot bundles
	# for dependency check
	allBundles = copy(depotBundles)
	for key in installedBundles:
		if key not in depotBundles:
			allBundles[key] = installedBundles[key]

	# Scan sysSateDeps checking.
	for bundle in depotBundles.values():
                for reqID in bundle['reqIDs']:
                        if not CheckStateDeps( allBundles, reqID, bundle, installedBundleList ):
                                bundle['applicable'] = False
                                break
        
	desclist = [bundle['desc'] for bundle in depotBundles.values() if bundle['applicable'] is not False]
	if not scanDepotOnly:
		for key, bundlehash in installedBundles.items():
			if key not in depotBundles:
				desclist.append( bundlehash['desc'] )
	#
	# TODO: properly return the desclist, dont use globals
	global lastDesclist
	lastDesclist = desclist
	latest, removes, nodeps = insthelper.GetRpmBaseline(desclist)
	bundleIDs = insthelper.GetLatestBundles(desclist, latest, removes)
	#depotBundles.latest = latest
	#depotBundles.extras = (removes, nodeps, bundleIDs)
	extras = (removes, nodeps, bundleIDs)
                        
	#
	# Any bundle, installed or depot, can obsolete any other.
	#
	for bundle in allBundles.values():
		CheckObsolescence( bundle, allBundles, latest, extras, rollup )

	#
	# Conflicts comes after, since a conflict with an obsoleted
	# bundle does not count
	#
	for bundle in allBundles.values():
		CheckConflicts( bundle, allBundles, confFilterFunc )

	#
	# Finally, we check the requires.  Handle these cases:
	# Requiring an obsoleted bundle
	# Requiring a bundle in conflict
	#
	for bundle in depotBundles.values():
		CheckRequirements( bundle, allBundles, reqFilterFunc, rollup)
		AddScanExplanation( bundle, allBundles )


# ----------------------------------------------------------------------------
def GetSystemStates( installedPatch ):
	""" installedPatch is the dict for one installed bundle returned
	from PatchDB. This returns a comma-separated string of the
	current system states for this bundle.
	"""
	return installedPatch.get('systemStates', [])


# ----------------------------------------------------------------------------
def CheckStateDeps( bundledict, reqID, bundle, installedBundleList ):
	""" Check that the bundle satisfy the system state dependency
	requirements for reqID in bundle.
        bundle['sysStateDeps'][reqID]: dict with states and required values:
	  { 'rebooted': True  }  means rebooted must be present as a state
	"""
        sysStateDeps = bundle['sysStateDeps'].get(reqID, {})
	if not sysStateDeps:
		return True
	# only installed bundle hashes have a populated systemStates
	if reqID not in installedBundleList:
		return False
	flag = False
	sysStates = bundledict[reqID]['systemStates']
	for state in sysStateDeps:
		if sysStateDeps[state] == True:
			flag = state in sysStates
		else:
			flag = state not in sysStates
		if not flag:
                        msg = "Bundle %s required %s to have %s=%s, " \
                              "but the system states were %s." % (
				bundle['bundleID'], reqID, state,
				sysStateDeps[state], sysStates)
                        log.debug( msg )
			return False
	return flag


# ----------------------------------------------------------------------------
def NewerBundleAllowed( bundlehash, visited ):
	""" Return False if an obsolete bundle should not be allowed to find
	a newer, superseding bundle in order to satisfy dependencies. """
	if bundlehash['bundleID'] in visited:
		log.log(1, bundlehash['bundleID'] + ' visited already')
		return False
	if not bundlehash['newerAvailable']:
		log.debug('Bundle %s is obsolete and does not '
			  'have newerAvailable; cannot re-route. '
			  'This is probably a bug.'
			  % (bundlehash['bundleID']) )
		return False
	#
	# Do not allow x.y.z bundle IDs to find
	# newer ones.  A bundle requiring a certain version
	# x.y.z of ESX should not be allowed to install
	# on a different x.y.z version.
	#
	if re.match( r'\w\.\w\.\w', bundlehash['bundleID'] ):
		log.debug('Require of %s cannot be superseded' %
                          (bundlehash['bundleID']) )
		return False
	return True


# ----------------------------------------------------------------------------
def FindNewestBundles( bundleIDs, bundledict, visited ):
	""" For each bundle in bundleIDs, if it is obsolete, traces down
	the tree via 'newerAvailable' to find the newest bundleID.
	Then it returns back a list of newest bundleIDs, in the same order.
	If the bundleID is not in bundledict, or it is but has no 
	'newerAvailable', then it is dropped from the new list.
	"""
	newlist = []
	for bundle in bundleIDs:
		if bundle not in bundledict:
			log.log(1, bundle+' not in bundledict, skipping')
			continue
		mybundle = bundledict[bundle]
		if mybundle['obsolete']:
			if not NewerBundleAllowed( mybundle, visited ):
				continue
			visited.add( bundle )
			newbundles = FindNewestBundles(
				mybundle['newerAvailable'].keys(),
				bundledict, visited )
			log.log(1, 'Dependency for obsolete %s satisfied '
				'by %s' % (bundle, newbundles))
			newlist.extend( newbundles )
		else:
			newlist.append( bundle )

	return newlist

	
# ----------------------------------------------------------------------------
def IndividualReqFilter( matches, bundledict, reqFilterFunc, rollup=None ):
        """ For individual scan mode, only installed bundles will
        satisfy dependencies immediately. """
	return [key for key in matches if bundledict[key]['installed'] ], \
	       set()

# ----------------------------------------------------------------------------
def GroupReqFilter( matches, bundledict, reqFilterFunc, rollup=None ):
        """ Return which bundles satisfy requirements for group install
        - this means tracing the requires through the depot until one
        is found either applicable or non applicable. """
        log.log(1, "GroupReqFilter(%s)" % (matches))
        goodones = []
	visited = set()
        for match in matches:
                CheckRequirements( bundledict[match], bundledict,
				   reqFilterFunc, rollup )
                if bundledict[match]['applicable']:
                        goodones.append( match )
                visited.update( bundledict[match]['visited'] )
	return goodones, visited
        
# ----------------------------------------------------------------------------
def CheckRequirements( bundle, allBundles, reqFilterFunc, rollup=None ):
	""" Checks that the required bundles are installed.
	Side Effects:
	May turn OFF bundle['applicable']
	(will never turn it on, it should be on by default)
	bundle['missingIDs']  : list of reqIDs not finding matches in
	                        allBundles, incld obsoleted x.y.z
				releases in PatchDB
	bundle['availableIDs']: matching bundle IDs in allBundles not
	                        satisfying filter criteria
	"""
        #
        # Don't visit bundles already visited by recursive calls
        # to CheckRequirements(). This breaks dependency loops.
        #
        global dependencies_visited
        if bundle['bundleID'] in dependencies_visited:
                return
        dependencies_visited[bundle['bundleID']] = True
        
        if bundle['applicable'] is None:
                bundle['applicable'] = True
                log.log(1, "CR: [%s] marked as applicable" % (bundle['bundleID']) )

	for reqID in bundle['reqIDs']:
		touched = set()
		matches = FindKeysByPatternList( allBundles, [reqID] )
		if rollup and matches:
			newest = matches
		else:
			newest = FindNewestBundles(matches, allBundles, touched)

		if not newest:
		#
		# If our requirement doesn't exist in the depot, we assume that
		# anything that obsoletes our requirement can be used to meet
		# the requirement instead.
		#
			newest = list()
			for b in allBundles.values():
				if reqID in b['obsoleteIDs']:
					newest.append(b['bundleID'])

		qualified, visited = reqFilterFunc( newest, allBundles,
						    reqFilterFunc, rollup )
		bundle['visited'].update( touched )
		bundle['visited'].update( visited )
                #
		# Multiple bundles may obsolete a single one, and a least one of the
		# obsoleting bundles would be required.
		#
		if not newest or len(qualified) < 1:
			bundle['applicable'] = False
                        log.debug("CR: [%s] dep %s not satisfied: "
                                  "matches=%s newest=%s qualified=%s" %
                                  (bundle['bundleID'], reqID,
                                   matches, newest, qualified))
			if not matches or not newest:
				bundle['missingIDs'].append( reqID )
			else:
				bundle['availableIDs'].extend( newest )


# ----------------------------------------------------------------------------
def CheckObsolescence( bundle, allBundles, latest, extras, rollup=None ):
	"""May turn OFF bundle['applicable'] (will never turn it on, it should
	be on by default.
	Side Effects:
	May append entries to bundle['newerAvailable'] of other bundles
	"""
	removes, nodeps, bundleIDs = extras
	obKeys = FindKeysByPatternList( allBundles,
					bundle['obsoleteIDs'] )
	# cannot obsolete bundle itself
	if bundle['bundleID'] in obKeys:
		obKeys.remove( bundle['bundleID'] )

	if rollup:
		return

	for obKey in obKeys:
		obBundle = allBundles[obKey]
		obBundle['obsolete'] = True
		if bundle['installed']:
			obBundle['obsoletedByHost'] = True
		obBundle['applicable'] = False
		obBundle['newerAvailable'].add( bundle['bundleID'] )
		log.debug("%s obsoleted by %s" % (obKey,
                                                  bundle['bundleID']))

	newerBundles, newerPkg = insthelper.SupersedingBundleInfo(
		bundle['desc'], latest, removes, bundleIDs, allBundles)
	instflag = len([x for x in newerBundles if allBundles[x]['installed']])
	if newerBundles:
		bundle['obsolete'] = True
		if instflag:
			bundle['obsoletedByHost'] = True
		bundle['applicable'] = False
		bundle['newerAvailable'].update( dictFromList( newerBundles ) )
		bundle['newerRpms'] = newerPkg
		log.debug("%s implicitly obsoleted by %s" % (
			bundle['bundleID'], newerBundles))

	#
	# Report RPM obsolescence info to log
	for pkg, newerID in bundle['newerRpms'].items():
		installedStr = ''
		if allBundles[newerID]['installed']:
			installedStr = 'installed '
		log.debug('%s: %s superseded by %sbundle %s' % (
			bundle['bundleID'], pkg, installedStr, newerID))


# ----------------------------------------------------------------------------
def IndividualConfFilter( bundleID, confKeys, allBundles ):
	""" Any bundle that conflicts with an installed bundle will become
	inapplicable. Returns list of bundles to be marked inapplicable. """
	if allBundles[bundleID]['installed']:
		return confKeys
	else:
		for key in confKeys:
			if allBundles[key]['installed']:
				return [ bundleID ]
		return []

# ----------------------------------------------------------------------------
def GroupConfFilter( bundleID, confKeys, allBundles ):
	""" Any conflict between two bundles makes both of them
	inapplicable. Installed bundles are exempted. """
        if confKeys:
                return [key for key in ([ bundleID ] + confKeys) \
			if not allBundles[key]['installed'] ]
	else:
                return []

# ----------------------------------------------------------------------------
def CheckConflicts( bundle, allBundles, conflictFilter ):
	""" Modifies 'conflictIDs': what bundles this bundle conflicts with
	"""
	uPaths = bundle['desc'].GetUpgradePaths()
	confKeys = FindKeysByPatternList( allBundles, uPaths.GetConflictIDs() )
        bundle['conflictIDs'].update( dictFromList( confKeys ) )
	myID = bundle['bundleID']
	conflicts = conflictFilter( myID, confKeys, allBundles )
	for key in conflicts:
		confBundle = allBundles[key]
		confBundle['applicable'] = False
		if key != myID:
			log.debug("[%s] inapplicable due to conflict with %s" % (
				key, myID) )
			confBundle['conflictIDs'].add( myID )
		else:
			log.debug("[%s] inapplicable due to conflict with %s" % (
				myID, conflicts) )


# ----------------------------------------------------------------------------
def AddScanExplanation( scanresult, bundledict ):
	""" For scanresult being one of the dict values of the ScanDepot()
	result, a list scanresult['explanations'] is added, with a string
	for each reason that the scanresult may be found inapplicable.
	bundledict - should contain bundle dicts for both depot and
	             installed bundles, like allBundles.  """
	explanations = []
	prefix = '[%s] ' % (scanresult['bundleID'])
	if scanresult['obsolete']:
		msg = "superseded by %s." % (scanresult['newerAvailable'].keys())
		explanations.append(prefix + msg)
	if scanresult['missingIDs']:
		msg = "requires %s but these bundles cannot be found. " \
		      "Please make sure they are in the depot and specified " \
		      "in the bundle list." % (scanresult['missingIDs'])
		explanations.append(prefix + msg)
		
	for confID in scanresult['conflictIDs']:
		if bundledict[confID]['installed']:
			msg = "installed bundle %s and cannot be installed."
		else:
			msg = "%s. Only one of these may be installed."
		explanations.append(prefix +"conflicts with "+ msg % (confID))

	sysStateDepsDict = {}
	for reqID in scanresult['sysStateDeps']:
		matchIDs = fnmatch.filter( scanresult['availableIDs'],
					   reqID )
		for key in matchIDs:
			sysStateDepsDict[key] = scanresult['sysStateDeps'][reqID]
			
	for availID in scanresult['availableIDs']:
		stateDeps = sysStateDepsDict.get(availID, None)
		states = bundledict[availID]['systemStates']
		if stateDeps and stateDeps.get('rebooted', False):
			msg = "%s to be installed and the ESX Server to " \
			      "be rebooted afterwards before proceeding." \
			      % (availID)
		elif stateDeps:
			msg = "%s to have system states of %s, but the "\
			      "states found were %s." % (availID, stateDeps,
							 states)
		else:
			msg = "%s, but it is not applicable." % (availID)
		explanations.append(prefix +"requires "+ msg)

	scanresult['explanations'] = explanations


# ----------------------------------------------------------------------------
def ComputeBundleGroups( scanResults, reinstall=False ):
	""" Figure out an ordered list of bundles to install, given the
	results from ScanDepot(groupMode=True).
	Args
	   scanResults   : hash of hashes as returned by ScanDepot()
	   reinstall     : (bool) True if re-installing a bundle is allowed
	Returns
	   groups        : [ [bundle-list1] [bundle-list2] ... ]
	       The packages from bundle-list1 must be installed before
	       bundle-list2.  Typically, it means bundles in bundle-list2
	       require the bundles in bundle-list1.
        scanresult['explanations'] may be modified.
	"""
	groups = []
	bundles = []
	log.log(1, 'Starting ComputeBundleGroups, reinstall=%s' % (reinstall))
	#
	# If scan was called with groupMode=True, then the applicable
	# flag will take care of all reasons not to include a bundle
	# for multi-bundle installs related to dependencies.  We just
	# have to take care of already installed bundles.
	#
	for bundleID, scanresult in scanResults.items():
		if scanresult['installed'] and not reinstall:
			msg = "[%s] is already installed, nothing for " \
			      "esxupdate to do." % (bundleID)
			scanresult['explanations'].append( msg )
			continue

		if scanresult['applicable']:
			bundles.append(bundleID)

	groups.append(bundles)
	return groups


# ----------------------------------------------------------------------------
def PulledDeps( reqBundles, bundledict ):
	""" Returns a list of the bundle IDs that are pulled in by the
	reqBundles via requires.
	reqBundles - list of -b wildcards passed on cmd line """
	if reqBundles is None:
	      return []
	visited = set()
	bundles = FindKeysByPatternList( bundledict, reqBundles )
	for key in bundles:
		visited.update( bundledict[key]['visited'] )
	return FindKeysByPatternList( bundledict, visited.keys() )


# ----------------------------------------------------------------------------
def EnoughDiskSpace( bundle ):
	"""TODO: docstr"""
	#print "TODO: JUST A STUB"
	return True

			
