#######################################################################
# Copyright (C) 2005 VMWare, Inc.
# All Rights Reserved
########################################################################
#
# system.py
#
#    A standard system library for VMware Python scripts.
#    Provides standardized logging; system/shell functionality;
#    config file reading/writing
#
#    Inspired by handrew's Perl libraries... System.pm, etc.
#
import os, os.path
import sys
import re
import shutil
import time
import pty
import types
import signal
import select
from popen2 import MAXFD

import logging
from logging import handlers

log = logging.getLogger('system')
stdoutHdl = None
ptypid = -1


########################################################################
#
#   Setting up logging.  Requires the logging package.
#

#
# Define the default logging directory.
#
LOGDIR = '/tmp'

#
# # of backup log files to keep
#
BACKUP_LOG_COUNT = 5

#
# Log format string for stdout (see Python logging module Formatter docs)
#
FMT_STDOUT = '%(levelname)s: %(message)s'

#
# Log format string for log file
#
FMT_LOGFILE = '[%(asctime)s] %(levelname)7s:%(name)10s: %(message)s'

#
# strftime()-style time and date format string
#
FMT_DATETIME = '%y%3b%d-%H%M'

class StderrFallbackRotatingFileHandler(handlers.RotatingFileHandler):
    def __init__(self, filename, mode, maxBytes, backupCount, stdoutHdl):
        handlers.RotatingFileHandler.__init__(self, filename, mode, 
                                              maxBytes, backupCount)
        self.stdoutHdl = stdoutHdl
        self.alreadyAnnouncedFallback = False

    def handleError(self, record):
        if self.alreadyAnnouncedFallback:
            return

        self.alreadyAnnouncedFallback = True
        sys.stderr.write( 'WARN: Could not write to log file\n' )
        sys.stderr.write( 'WARN: Probably out of disk space in /var/log\n' )
        sys.stderr.write( 'WARN: Falling back to stdout logging\n' )

        #make sure stdout handler is set at least as low as self from now on.
        if self.level < self.stdoutHdl.level:
            self.stdoutHdl.setLevel( self.level )

        #we need to output this one message so it doesn't get skipped.
        msg = self.format( record ) +'\n'
        sys.stderr.write( msg )

        handlers.RotatingFileHandler.handleError(self, record)

#
# Provide a logging setup function for scripts that sets some defaults
# and returns a log object.
#
# The defaults are:
#   Log output to console and to a file in LOGDIR/argv[0].log
#
#   Default logging level is:
#    * INFO or above for console
#    * DEBUG or above for file
#
#   Log rotation is off (set maxFileSize to enable log rotation)
#
def LogSetup(logfile=None, maxFileSize=0):
    """ Default logging setup and returns the log object.

    logfile    - Use logfile rather than LOGDIR/argv[0].log
    maxFileSize - Max size of log file before it's rotated, 0 = never rotate
    """
    
    log = logging.getLogger(None)
    #
    # Sometimes the log instance ends up with a handler.
    # Make sure there are no existing handlers to avoid duplicate
    # log output.
    #
    log.handlers = []

    #
    # Set defaults for stdout logging
    #
    global stdoutHdl
    stdoutFmt = logging.Formatter(FMT_STDOUT)
    stdoutHdl = logging.StreamHandler()
    stdoutHdl.setFormatter(stdoutFmt)
    stdoutHdl.setLevel(logging.INFO)
    log.addHandler(stdoutHdl)
    
    #
    # Log file: default to argv[0].log;  set log level
    #
    fileFmt = logging.Formatter(FMT_LOGFILE, FMT_DATETIME)
    if not logfile:
        logfile = os.path.join(LOGDIR, os.path.basename(sys.argv[0]) + '.log')

    #
    # Create log file directory if not present
    #
    logfiledir = os.path.dirname(logfile)
    if not os.access(logfiledir, os.F_OK):
        os.makedirs(logfiledir)
        
    try:
        hdlr = StderrFallbackRotatingFileHandler(logfile,
                                                 'a', maxFileSize,
                                                 BACKUP_LOG_COUNT,
                                                 stdoutHdl)
        hdlr.setFormatter(fileFmt)
        hdlr.setLevel(logging.DEBUG)
        log.addHandler(hdlr)
    except IOError:
        log.exception('Cannot append to log file, disabling file logging')

    #
    # Set the logger to send all log events to the handlers (don't do
    # any pre-filtering).
    #
    log.setLevel(logging.NOTSET)
    return log


#
# Set the log level for the stdout handler. Note that LogSetup
# must be called before setting the level.
#
def LogSetLevel(level):
    """ Set the log level for the stdout handler.

    level - The new log level as an integer.
    """

    stdoutHdl.setLevel(level)


#
# Retrieve the log level for the stdout handler.  Note that LogSetup
# must be called before getting the level.
#
def LogGetLevel():
    """ Returns the log level for the stdout handler. """

    return stdoutHdl.level


########################################################################
#
#   Shell commands, output redirection, etc.
#

#
# The default timeout in seconds for LogCommand.
# Use None for no timeout.
#
LOGCMD_TIMEOUT = None

#
# Note that return value = res >> 8
#
class LogCommandError(Exception):
    def __init__(self, cmd, res, expected, output):
        self.args = (cmd, res, expected, output)
        self.cmd = cmd
        self.res = res
        self.expected = expected
        self.output = output
    def __repr__(self):
        return "Error executing [%s]:\nReturn value was %d, but expected %d\n%s" % (
            self.cmd, self.res >> 8, self.expected, self.output)
        
#
# Ported from VMware::System.pm.
# Unlike our Perl counterpart, we throw an exception on failure
# (when the return value does not equal expectedResult).
# The idea is that Python callers should check for all errors
# through the exception system.  If you want to ignore a process'
# return value, then catch and ignore the LogCommandError.
#
# If you just want piped output without worrying about errors,
# it's simpler to do
#    for line in popen('cmd'):
# The garbage collector will close the file object.  But you may not
# want a pipe to remain open that long.
#
def LogCommand(cmd, returnOutput=0, expectedResult=0, dontRedirectStdErr=0,
               loglevel=logging.DEBUG, timeout=LOGCMD_TIMEOUT):
    """ Wrap system calls to log commands and output.

    Args:
      cmd              - command to run. Mandatory.
      returnOutput     - (opt., default 0) if true, return the output
      expectedResult   - (opt., default 0) Expected return value
                         Note: 0 matches 0 or None
      dontRedirectStdErr - (opt.) Ignore stderr, windows can bug out
      loglevel         - (opt., default 10) log level for command output
      timeout          - (optional) Timeout in seconds

    Results:
      Output string if returnOutput was true;
      otherwise, returns true if return value was expectedResult.

    Throws:
      LogCommandError if return value was not expectedResult.
        Note that sh returns 127 if the command was not found.
      Other exceptions as thrown by os.popen, file.read, os.close

    Side effects:
      cmd is executed.  Output is logged at DEBUG level.
    """

    log.debug('LogCommand(%s)' % (cmd))

    out = ''
    if not dontRedirectStdErr:
        cmdout = os.popen(cmd + ' 2>&1')
    else:
        cmdout = os.popen(cmd)

    #
    # Use select to implement a timeout.  A value of 0 doesn't seem to work.
    # Empty file lists returned indicate timeout was exceeded.
    #
    while 1:
        #
        # a SIGCHLD could interrupt the select syscall.  Time to end.
        #
        try:
            inlist, outlist, exlist = select.select([cmdout], [], [], timeout)
        except select.error:
            break
        if not inlist and not outlist and not exlist:
            raise LogCommandError(cmd, -1, -1,
                                  '%d sec timeout exceeded' % (timeout))
        line = cmdout.readline()
        if not line: break
        log.log(loglevel, '| ' + line.rstrip())
        out += line

    #
    # If the child process already died, IOError is returned (PR 92188).
    #
    try:
        res = cmdout.close()
    except IOError:
        res = 0
        pass

    if res is None: res = 0
    if (res >> 8) == expectedResult:
        if returnOutput:
            return out
        else:
            return 1
    else:
        raise LogCommandError(cmd, res, expectedResult, out)


#
# Signal handler for SIGCHLD.  Reap child processes by
# repeatedly calling waitpid.
#
def _ignoreSIGCHLD(*unused):
    while 1:
        try: os.waitpid(-1, os.WNOHANG)
        except OSError: break


#
# Why use a pty-based spawn instead of the normal popen?
# For some reason, I found this combo of pty and SIGCHLD
# handler actually works in handling rpm installs with
# daemon launches.
#
def Ptypipe(cmd):
    """ Spawn a sub-process in a pseudo-terminal, and return
    a file object for reading the subprocess output.
    """

    #
    # This is needed since piped reads will hang if child processes become
    # zombie, say if an rpm launches a daemon and inherits the fd's.
    # Installing a signal handler will kill off zombies before they
    # hang us.  See Perl Cookbook Recipe 16.19.
    #
    global ptypid
    signal.signal(signal.SIGCHLD, _ignoreSIGCHLD)
    pid, fd = pty.fork()
    ptypid = pid
    if pid == 0:
        # Child.
        # Use the sh shell to run string cmd if cmd not a sequence
        if isinstance(cmd, types.StringTypes):
            cmd = ['/bin/sh', '-c', cmd]

        # Close all filehandles > 2 and goto subprocess
        for i in range(3, MAXFD):
            try:
                os.close(i)
            except:
                pass
        try:
            os.execvp(cmd[0], cmd)
        finally:
            os._exit(1)

    #
    #  -1 for unbuffered I/O
    fromChild = os.fdopen(fd, "r", -1)

    return fromChild


#
# Restore the default SIGCHLD handler.  This should be done
# when the ptypipe is no longer needed.
#
def ClosePtypipe():
    """ Closes a Ptypipe session by restoring SIGCHLD handler and
    setting ptypid to -1.
    """
    global ptypid
    signal.signal(signal.SIGCHLD, signal.SIG_DFL)
    ptypid = -1


#
# Obtain the pid from the Ptypipe() child process
#
def GetPtyPid():
    return ptypid


########################################################################
#
#   Rebooting and init.d utilities
#

#
# TODO: port GetPlatform, GetOS functions.
# TODO: Windows support?
#

#
# vmware.SetLinuxBoot --
#
#       For Linux, sets or clears a command to run on the next boot.
#       If $bootCmd is defined, sets a command to run that command with
#       any options passed in $options.  Otherwise, clears any existing
#       command.  A backup is made and the original rc.local is restored
#       after successful command execution.
#
# Results:
#       True on success.
#       IO/OSError exceptions on opening/closing rc.local, backing it up,
#               or writing out the new rc.local.
#
# Side effects:
#       As described above.
#

def SetLinuxBoot(bootCmd, options=None):
    """ Sets or clears a command to run on the next boot """

    if sys.platform[:5] != 'linux':
        raise RuntimeError, 'SetLinuxBoot not supported on platform %s' % (sys.platform)

    marker = '####VMware####'
    rcLocal = '/etc/rc.d/rc.local'
    backupFile = '/etc/rc.d/rc.local.save'

    lines = open(rcLocal).readlines()

    #
    # If callback exists, remove it (all lines from marker onwards)
    #
    existing = 0
    markre = re.compile(marker)
    for i in range(len(lines)):
        if markre.search(lines[i]):
            existing = 1
            del lines[i:]
            break

    if bootCmd:
        if options: bootCmd += ' ' + options
        if 'HOME' in os.environ:
            bootCmd = "HOME=%s %s" % (os.environ['HOME'], bootCmd)

        #
        # If no existing callback, make a backup of the original
        #
        if not existing:
            shutil.copy(rcLocal, backupFile)

        log.debug('Adding [%s] to rc.local...' % (bootCmd))
        lines.append(marker + '\n')
        lines.append(bootCmd + ' &\n')
        lines.append('cp -f %s %s\n' % (backupFile, rcLocal))
        lines.append('rm -f %s\n' % (backupFile))

    #
    # If removing the callback, and none was found, we're done.
    #
    elif not existing:
        return 1

    #
    # Write the new rc file
    # TODO: Replace this with Utility module write_file func
    #
    fd = open(rcLocal, 'w')
    fd.writelines(lines)
    fd.close()

    return 1


#
# vmware.Reboot --
#
# 	Initiates a graceful reboot of the system (Windows or Linux)
#
# Results:
#       Does not return.
#
# Side effects:
# 	Exits the current process immediately after initiating reboot.
#

def Reboot():
    log.info('Rebooting in 5 seconds...')
    time.sleep(5)
    if sys.platform[:5] == 'linux':
        LogCommand('/sbin/shutdown -r now')
    else:
        raise RuntimeError, 'Reboot not supported on platform %s' % (sys.platform)

    #
    # Ensure open filehandles are closed by exiting
    #
    sys.exit(0)
