/** 
 * EditorListener: provides a data listener for HtmlEditor. The callback function 'dataReceived' 
 * is only a stub and needs to be implemented before using it.
 *
 * @param element  the element upon a the callback function can operate
 */
vpx.context.EditorListener = function EditorListener(element) {
   this.element = element;
};
vpx.context.EditorListener.prototype.element = null;
vpx.context.EditorListener.prototype.dataReceived = function(data) {}


/** 
 * setMemoryPerfLabel
 *
 * @param intMem
 * @param editorListener
 */
function setMemoryPerfLabel(intMem, editorListener) {
   if ( !isDefined(intMem) ) {
      return;
   }
   if(intMem == 0 || intMem == '') {
      _setLabelUnavailable(editorListener.element, vpx.context.ZERO_BYTES);
      return;
   }
   var editor = new vpx.util.HtmlEditor();
   editor.setDataListener(editorListener);

   var req = tle.getRequest("unitsConvert.do?amt=" + intMem + 
                            "&inputUnit=" + "MB" + 
                            "&unitType=Byte" );
   if (req == null) {
      /* Wait for thread to become available and try again */
      window.setTimeout(setMemoryPerfLabel.bind(this, intMem, editorListener), 500);
      return;
   }
   try {
      req.addResponseListener(editor);
      req.send();
   } finally {
      tle.releaseRequest(req);
   }
}

/** 
 * setMemoryPerfLabel
 *
 * @param intMem
 * @param intMaxMem
 * @param editorListener
 */
function setMemoryPerfBar(intMem, intMaxMem, editorListener) {
   if ( !isDefined(intMem) || !isDefined(intMaxMem) ) {
      return;
   }
   var editor = new vpx.util.HtmlEditor();
   editor.setDataListener(editorListener);  

   /* @see com.vmware.webcenter.control.action.passive.UnitsConversionAction */
   var req = tle.getRequest("unitsConvert.do?amt=" + intMem + 
	                        "&unitType=Byte" +
	                        "&inputUnit=" + "MB" + 
	                        "&outputUnit=" + "%25" +
	                        "&percentageMax=" + intMaxMem
	                        ); 
   if (req == null) {
      /* Wait for thread to become available and try again */
      window.setTimeout(setMemoryPerfBar.bind(this, intMem, intMaxMem, editorListener), 500);
      return;
   }
   try {
      req.addResponseListener(editor);
      req.send();
   } finally {
      tle.releaseRequest(req);
   }
}

/** 
 * setCpuPerfLabel
 *
 * @param intMHz
 * @param editorListener
 */
function setCpuPerfLabel(intMHz, editorListener) {
   if ( !isDefined(intMHz) ) {
      return;
   }
   if(intMHz == 0 || intMHz == '') {
      _setLabelUnavailable(editorListener.element, vpx.context.ZERO_HZ);
      return;
   }
   var editor = new vpx.util.HtmlEditor();
   editor.setDataListener(editorListener);

   /* @see com.vmware.webcenter.control.action.passive.UnitsConversionAction */
   var req = tle.getRequest("unitsConvert.do?amt=" + intMHz + 
                            "&inputUnit=" + "MHz" + 
                            "&unitType=Hertz" );
   if (req == null) {
      /* Wait for thread to become available and try again */
      window.setTimeout(setCpuPerfLabel.bind(this, intMHz, editorListener), 500);
      return;
   }
   try {
      req.addResponseListener(editor);
      req.send();
   } finally {
      tle.releaseRequest(req);
   }
}

/** 
 * setCpuPerfBar
 *
 * @param intMHz
 * @param intMaxCpuUsage
 * @param editorListener
 */
function setCpuPerfBar(intMHz, intMaxCpuUsage, editorListener) {
   if ( !isDefined(intMHz) || !isDefined(intMaxCpuUsage) ) {
      return;
   }
   var editor = new vpx.util.HtmlEditor();
   editor.setDataListener(editorListener); 

   /* @see com.vmware.webcenter.control.action.passive.UnitsConversionAction */
   var req = tle.getRequest("unitsConvert.do?amt=" + intMHz + 
		                    "&unitType=Hertz" +
		                    "&inputUnit=" + "MHz" + 
		                    "&outputUnit=" + "%25" +
		                    "&percentageMax=" + intMaxCpuUsage
		                    );
   if (req == null) {
      /* Wait for thread to become available and try again */
      window.setTimeout(setCpuPerfBar.bind(this, intMHz, intMaxCpuUsage, editorListener), 500);
      return;
   }
   try {
      req.addResponseListener(editor);
      req.send();
   } finally {
      tle.releaseRequest(req);
   }
}


/** 
 * setMemoryPerf
 *
 * @param intMem
 * @param intMemMax
 * @param memLabelId
 * @param memBarId
 */
function setMemoryPerf(intMem, intMemMax, memLabelId, memBarId) {
   if (memLabelId!=null) {
      var label = vpx.xua.getElement(memLabelId);
      var editorListener = new vpx.context.EditorListener(label);
      editorListener.dataReceived = function(data) {
         vpx.xua.removeClass(this.element, 'disabled');
         this.element.innerHTML = data;
      }
      setMemoryPerfLabel(intMem, editorListener);
   }
   if (memBarId!=null) {
      var bar = vpx.xua.getElement(memBarId);
      var editorListener = new vpx.context.EditorListener(bar); 
      editorListener.dataReceived = function(data) {
         this.element.style.backgroundPosition=data+'px';
      }
      setMemoryPerfBar(intMem, intMemMax, editorListener);
   }
}

/** 
 * setCpuPerf
 *
 * @param intMHz
 * @param intMHzMax
 * @param cpuLabelId
 * @param cpuBarId
 */
function setCpuPerf(intMHz, intMHzMax, cpuLabelId, cpuBarId) {
   if (cpuLabelId!=null) {
      var label = vpx.xua.getElement(cpuLabelId);
      var editorListener = new vpx.context.EditorListener(label);
      editorListener.dataReceived = function(data) {
         vpx.xua.removeClass(this.element, 'disabled');
         this.element.innerHTML = data;
      }
      setCpuPerfLabel(intMHz, editorListener);
   }
   if (cpuBarId!=null) {
      var bar = vpx.xua.getElement(cpuBarId);
      var editorListener = new vpx.context.EditorListener(bar); 
      editorListener.dataReceived = function(data) {
         this.element.style.backgroundPosition=data+'px';
      }
      setCpuPerfBar(intMHz, intMHzMax, editorListener);
   }
}

/**
 * _init: priviate initializer function for setting Strings for unavailable values.
 *        this is necessary for allowing internationalization of those constants.
 */
vpx.context.ZERO_HZ = null;
vpx.context.ZERO_BYTES = null;
function _init() {
   var editorMHz = new vpx.util.HtmlEditor();
   var editorMB = new vpx.util.HtmlEditor();

   var elMHz = new vpx.context.EditorListener();
   elMHz.dataReceived = function (data) {
      vpx.context.ZERO_HZ = data;
   }
   
   var elMB = new vpx.context.EditorListener(vpx.context.ZERO_BYTES);
   elMB.dataReceived = function (data) {
      vpx.context.ZERO_BYTES = data;
   }
   
   editorMHz.setDataListener(elMHz);
   editorMB.setDataListener(elMB);
    
   /* @see com.vmware.webcenter.control.action.passive.UnitsConversionAction */
   var reqMHz = tle.getRequest("unitsConvert.do?amt=0&unitType=Hertz&inputUnit=MHz");
   var reqMB = tle.getRequest("unitsConvert.do?amt=0&unitType=Byte&inputUnit=MB");
   if (reqMHz == null || reqMB == null) {
      /* Wait for thread to become available and try again */
      window.setTimeout(_init.bind(this), 500);
      return;
   }
   try {
      reqMHz.addResponseListener(editorMHz);
      reqMB.addResponseListener(editorMB);
      reqMHz.send();
      reqMB.send();
   } finally {
      tle.releaseRequest(reqMHz);
      tle.releaseRequest(reqMB);
   }
}
_init(); // do it once at load time

/** 
 * private helper function for disabling bar
 */
function _setBarUnavailable(barId){
   vpx.xua.getElement(barId).style.backgroundPosition='0px';
}   

/** 
 * private helper function for disabling label
 */
function _setLabelUnavailable(labelId, msg){
   var elem=vpx.xua.getElement(labelId);
   elem.innerHTML=msg;
}

/** 
 * setMemoryUnavailable
 *
 * @param memLabelId
 * @param memBarId
 */
function setMemoryUnavailable(memLabelId, memBarId) {
   if (memLabelId!=null) {
      _setLabelUnavailable(memLabelId, vpx.context.ZERO_BYTES);
   }
   if (memBarId!=null) {
      _setBarUnavailable(memBarId);
   }
}

/** 
 * setCpuUnavailable
 *
 * @param cpuLabelId
 * @param cpuBarId
 */
function setCpuUnavailable(cpuLabelId, cpuBarId) {
   if (cpuLabelId!=null) {
      _setLabelUnavailable(cpuLabelId, vpx.context.ZERO_HZ);
   }
   if (cpuBarId!=null) {
      _setBarUnavailable(cpuBarId);
   }
}


/** 
 * setPerfUnavailable
 *
 * @param cpuLabelId
 * @param memLabelId
 * @param cpuBarId
 * @param memBarId
 */
function setPerfUnavailable(cpuLabelId, memLabelId, cpuBarId, memBarId) {
   if (cpuLabelId!=null) {
      _setLabelUnavailable(cpuLabelId, vpx.context.ZERO_HZ);
   }
   if (memLabelId!=null) {
      _setLabelUnavailable(memLabelId, vpx.context.ZERO_BYTES);
   }
   if (cpuBarId!=null) {
      _setBarUnavailable(cpuBarId);
   }
   if (memBarId!=null) {
      _setBarUnavailable(memBarId);
   }
}